/*************************************************************************************
 * Copyright (c) 2012 Red Hat, Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     JBoss by Red Hat - Initial implementation.
 ************************************************************************************/
package org.switchyard.tools.ui.editor.diagram.implementation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.graphiti.features.IFeatureProvider;
import org.eclipse.graphiti.features.context.ICreateContext;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.soa.sca.sca1_1.model.sca.Component;
import org.eclipse.soa.sca.sca1_1.model.sca.ComponentReference;
import org.eclipse.soa.sca.sca1_1.model.sca.ComponentService;
import org.eclipse.soa.sca.sca1_1.model.sca.Implementation;
import org.switchyard.tools.ui.editor.ComponentTypeExtensionManager;
import org.switchyard.tools.ui.editor.ImageProvider;
import org.switchyard.tools.ui.editor.Messages;
import org.switchyard.tools.ui.editor.diagram.shared.CreateTypeFeature;

/**
 * CreateImplementationFeature
 * 
 * <p/>
 * Feature for creating new Implementation types.
 * 
 * @author Rob Cernich
 */
public class CreateImplementationFeature extends CreateTypeFeature<Implementation, Component> {

    private String _imageId = ImageProvider.IMG_16_IMPLEMENTATION_TYPE;

    /**
     * Create a new CreateImplementationFeature.
     * 
     * @param fp the feature provider
     * @param factory the factory creating the implementation
     * @param name the name of this feature
     * @param description the description for this feature
     */
    public CreateImplementationFeature(IFeatureProvider fp, IImplementationTypeFactory factory, String name,
            String description) {
        super(fp, factory, Component.class, name, description);
    }

    /**
     * Create a new CreateImplementationFeature.
     * 
     * @param fp the feature provider
     * @param factory the factory creating the implementation
     * @param name the name of this feature
     * @param description the description for this feature
     * @param imageId the image ID for this feature
     */
    public CreateImplementationFeature(IFeatureProvider fp, IImplementationTypeFactory factory, String name,
            String description, String imageId) {
        super(fp, factory, Component.class, name, description);
        _imageId = imageId;
    }

    @Override
    public Object[] create(ICreateContext context) {
        if (getContainerObject(context).getImplementation() != null) {
            if (!MessageDialog
                    .openQuestion(
                            getShell(),
                            Messages.title_replaceCurrentImplementation,
                            Messages.description_replaceCurrentImplementation)) {
                return null;
            }
        }
        return super.create(context);
    }

    @Override
    protected Object[] updateContainer(ICreateContext context, Implementation newObject) {
        final List<Object> added = new ArrayList<Object>(3);
        final Component component = getContainerObject(context);

        // add the implementation
        component.setImplementation(newObject);
        added.add(newObject);

        final IImplementationTypeFactory factory = (IImplementationTypeFactory) getFactory();
        if (component.getService().size() == 0) {
            // add any new services
            final ComponentService service = factory.getImplementedService();
            if (service != null) {
                component.getService().add(service);
                added.add(service);
            }
        }

        if (component.getReference().size() == 0) {
            // add any new references
            final List<ComponentReference> references = factory.getImplementationReferences();
            if (references != null) {
                for (ComponentReference reference : references) {
                    component.getReference().add(reference);
                    added.add(reference);
                }
            }
        }

        return added.toArray();
    }

    @Override
    public String getCreateImageId() {
        return _imageId;
    }

    @Override
    protected Collection<String> getRequiredCapabilities(Implementation newObject) {
        return ComponentTypeExtensionManager.instance().getExtensionFor(newObject.getClass())
                .getRequiredCapabilities(newObject);
    }
}
