/*
 * JBoss, Home of Professional Open Source.
 *
 * See the LEGAL.txt file distributed with this work for information regarding copyright ownership and licensing.
 *
 * See the AUTHORS.txt file distributed with this work for a full listing of individual contributors.
 */
package org.teiid.designer.ui.common.dialog;

import java.io.File;

import org.eclipse.core.runtime.IPath;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.teiid.core.designer.util.CoreArgCheck;
import org.teiid.core.designer.util.CoreStringUtil;
import org.teiid.designer.ui.common.UiConstants;


/**
 * @since 8.0
 */
public class FileUiUtils implements UiConstants {

    public static FileUiUtils INSTANCE = new FileUiUtils();

    /**
     * Obtains the file name of an existing file whose name is the same as the specified input regardless of case. Leading and
     * trailing spaces are stripped from the input.
     * 
     * @param theFullPathName the file name being checked
     * @return the file name of an existing file having the same name but different case; otherwise the input parameter.
     * @throws AssertionError if input parameter is <code>null</code> or empty
     * @since 5.0.1
     */
    public String getExistingCaseVariantFileName( String theFullPathName ) {
        String result = theFullPathName;

        if (result != null) {
            result = theFullPathName.trim();
        }

        CoreArgCheck.isTrue(!CoreStringUtil.isEmpty(result), "The full path name cannot be empty"); //$NON-NLS-1$

        File file = new File(result);

        // file.exists() returns true even if case is different
        if (file.exists()) {
            String name = file.getName();
            File parentDir = file.getParentFile();

            if (parentDir == null) {
                File tempFile = file.getAbsoluteFile();
                parentDir = tempFile.getParentFile();
            }

            if (parentDir != null) {
                File[] kids = parentDir.listFiles();

                // Walk the parent directory looking for files that do not have the EXACT name,
                // but do have the same name with one or more letters of a different case.
                for (int i = 0; i < kids.length; ++i) {
                    String existingName = kids[i].getName();

                    if (existingName.equalsIgnoreCase(name)) {
                        result = kids[i].getAbsolutePath();
                        break;
                    }
                }
            }
        }

        return result;
    }

    /**
     * Search the directory of the specified file for files that have the exact same name, but with one or more letters of a
     * different case. For example, "myFile.txt" and "MyFile.txt". Return the filename of the case-variant file, if any, or return
     * null.
     * 
     * @param path The path to the file to be saved
     * @return The clashing file name or null
     */
    public String getExistingCaseVariantFileName( IPath path ) {
        return getExistingCaseVariantFileName(path.toOSString());
    }
    
    public void showFileContents(Shell shell, File file, String title, String displayedFileName) {
    	if( file.exists() ) {
    		new FileContentsDialog(shell, title, file, displayedFileName).open();
    	} else {
    		MessageDialog.openError(shell,  "File Not Found", "The file: " + displayedFileName + " does not exist");
    	}
    }

}
