(function(mod) {
    if (typeof exports == "object" && typeof module == "object") // CommonJS
        return mod(require("tern/lib/infer"), require("tern/lib/tern"));
    if (typeof define == "function" && define.amd) // AMD
        return define([ "tern/lib/infer", "tern/lib/tern" ], mod);
    mod(tern, tern);
})(function(infer, tern) {
    "use strict";
    
  function emptyObj(ctor) {
    var empty = Object.create(ctor.prototype);
    empty.props = Object.create(null);
    //empty.isShell = true;
    return empty;
  }
  
  function getModule(data, name) {
    return data.modules[name] || (data.modules[name] = emptyObj(infer.Obj));
  }
  
  function getSubModule(data, name) {   
    return data.submodules[name] || (data.submodules[name] = new infer.AVal);
  }
  
  function findModule(data, name) {   
    return data.modules[name];
  }
  
  function copyModules(modules, Y) {
    for (var name in modules) {
      copyModule(modules[name], Y);	  
    }
  }
  
  function copyModule(mod, Y) {
    var from = mod.getType();
    if (!from) return; // Unknown module
    from.forAllProps(function(prop, val, local) {
      if (local && prop != "<i>") {
        var t = new infer.PropHasSubset(prop, val);
        t.origin = from.origin;
        Y.propagate(t);
      }
    });	
  }  
  
  function injectModules(Y, name) {
    var cx = infer.cx(), server = cx.parent, data = server._yui;
    if (name == '*') {
      // inject local (YUI3) and contributed (AlloyUI) YUI modules
      copyModules(data.modules, Y);      
    } else {
        var module = findModule(data, name);
    }
  }
      
  function getFnIndex(argNodes) {
    for ( var i = 0; i < argNodes.length; i++) if (argNodes[i].type == "FunctionExpression") return i;
  }

  infer.registerFunction("yui_use", function(self, args, argNodes) {
    var yui = self.getType();
    if (yui && argNodes) {
      var index = getFnIndex(argNodes);
      if (index) {
        var fn = args[index];
        if (fn.argNames  && fn.argNames.length > 0) {  
          var cx = infer.cx(), paths = cx.paths;
          // Create instance of YUI because fn(modules: string, callback?: fn(Y: +yui.YUI)) -> !this support only signatures with one module
          // see https://github.com/angelozerr/tern-yui3/issues/12
          // var Y = fn.args[0];
          var Y = new infer.Obj(paths["yui.YUI.prototype"]);//fn.args[0];
          var deps = [];
          deps[0] = Y.getType();
          fn.getType().propagate(new infer.IsCallee(infer.cx().topScope, deps, null, infer.Null))
          
          // Inject local and contributed modules.
          injectModules(Y, '*');
          
          for ( var i = 0; i < argNodes.length - 1; i++) {
            var node = argNodes[i];
            if (node.type == "Literal" && typeof node.value == "string") {
              injectModules(Y, node.value);
            } else if (node.type == "ArrayExpression") for (var i = 0; i < node.elements.length; ++i) {
              var elt = node.elements[i];
              if (elt.type == "Literal" && typeof elt.value == "string") {
                injectModules(Y, elt.value);
              }
            }
          }
        }        
      }
    }
  });
  
  function registerLints() {
    if (!tern.registerLint) return;
    
    // validate existing modules for YUI().use(' 
    tern.registerLint("yui_use_lint", function(node, addMessage, getRule) {
      var rule = getRule("UnknownModule");
      if (rule && node.arguments) {
        var cx = infer.cx(), server = cx.parent, mods = server._yui.modules, submods = server._yui.submodules;
        for (var i = 0; i < node.arguments.length; i++) {
          var argNode = node.arguments[i];
          if (argNode.type == "Literal" && typeof argNode.value == "string") {
            var name = argNode.value;
            // check the module name exists for locals (YUI3) and custom (ex : AlloyUI) modules
            if (!mods[name] && !submods[name]) addMessage(argNode, "Unknown module '" + name + "'", rule.severity);
          } else {
            // the node is not a literal string, check if it's the last parameter which is a function type
            if (!(i == (node.arguments.length - 1) && argNode.type == "FunctionExpression")) addMessage(argNode, "Expected string type for YUI module", rule.severity);            
          }
        }
      }      
    });
    
  }

  tern.registerPlugin("yui3", function(server, options) {
    registerLints();
    server._yui = {
      modules: Object.create(null),
      submodules: Object.create(null)
    };
    
    server.on("reset", function() {
	  this._yui.modules = Object.create(null);
	  this._yui.submodules = Object.create(null); 
	});
    
    return {defs: defs,
      passes: {preLoadDef: preLoadDef,
               postLoadDef: postLoadDef,
               completion: findCompletions}};
  });
    
  function preLoadDef(data) {
    var cx = infer.cx(), localDefs = cx.localDefs;
    if (cx.definitions["yui3"] && data["!define"] && data["!define"]["_yui"]) {
      // set yui3 as local defs for AlloyUI tern plugin
      cx.localDefs["yui3"] = cx.definitions["yui3"];
    }
  }        
	
  function findClassByPath(paths, props, props2) {
    var clazz = null;
    for (var i = 0; i < paths.length; i++) {
      var path = paths[i];
      clazz = props[path] ? props[path] : props2[path];
      if (clazz) props = clazz.getType().props; else break;     
    }
    return clazz;
  }
  
  function postLoadDef(data) {
    var cx = infer.cx(), defName = data["!name"], mods = null;
    if (defName == "yui3") mods = cx.definitions[defName];
    else if (cx.definitions[defName]["_yui"]) mods = cx.definitions[defName]["_yui"].props;
    var _yui = cx.parent._yui;
    if (mods) for (var name in mods) {
      
      // loop for YUI modules
      var mod = mods[name].getType()
      if (mod && mod.name != 'config') {      
        var name = (mod.metaData && mod.metaData.module) ? mod.metaData.module : name, 
            submodules = (mod.metaData && mod.metaData.submodules) ? mod.metaData.submodules : null, 
            modToPropagate = getModule(_yui, name);
        if (!modToPropagate.origin) modToPropagate.origin = defName;
        
        // for AlloyUI, module is declared inside A
        var a = mod.hasProp("A");
        if (a) mod = a.getType();
        
        // Loop for YUI classes of the current module
        for(var className in mod.props) {
          var clazz = mod.props[className].getType(), metaData = clazz.metaData, 
              forClass = (metaData && metaData["for"]), augments = (metaData && metaData["augments"]), exts = (metaData && metaData["extends"]);
          if (!forClass) {
            var t = new infer.PropHasSubset(className, clazz);
            t.origin = clazz.origin;
            modToPropagate.propagate(t);                       
          } else {
            var classToUpdate = findClassByPath(forClass.split("."), mods, _yui.modules);
            if (classToUpdate) mix(clazz, classToUpdate.getType());
          }
          if (augments) mixall(augments, clazz, mods,  _yui.modules, true);
          if (exts) mixall(exts, clazz, mods,  _yui.modules, false);
        }
        
        // update submodules
        if (submodules) for(var subname in submodules) {
          var submodule = getSubModule(_yui, subname);
          submodule.origin = defName;
        }
      }
    }
  }
  
  function mixall(froms, to, mods, mods2, augments) {
    // update classes with extends or augments
    for (var i = 0; i < froms.length; i++) {
      var useClass = findClassByPath(froms[i].split("."), mods, mods2);
      if (useClass) {
        if (!augments) mix(useClass.getType(), to); else mix(to, useClass.getType());
      }
    }    
  }
  
  function mix(from, to) {
    from.forAllProps(function(prop, val, local) {
      if (local && prop != "<i>") {
        if (prop == "prototype") {          
          var fromProto = val.getType();
          var toProto = to.hasProp("prototype") && to.hasProp("prototype").getType() ? to.hasProp("prototype").getType() : null;
          if (toProto) mix(fromProto, toProto);          
        } else {
          var t = new infer.PropHasSubset(prop, val);
          t.origin = from.origin;
          to.propagate(t);  
        }
      }
    });
  }
  
  function findCompletions(file, query) {
    var wordEnd = tern.resolvePos(file, query.end);
    var callExpr = infer.findExpressionAround(file.ast, null, wordEnd, file.scope, "CallExpression");
    if (!callExpr) return;
    var callNode = callExpr.node;
    if (!callNode.callee.object || !callNode.callee.object.callee || !(callNode.callee.object.callee.name === "YUI" || callNode.callee.object.callee.name === "AUI") ||
        callNode.callee.type != "MemberExpression" || !callNode.callee.property || callNode.callee.property.name != "use" ||
        callNode.arguments.length < 1) return;
    // here completion for modules YUI().use('Ctrl+Space'
    var argNode = findNodeModule(callNode.arguments, wordEnd);
    if (!argNode) return;
    var word = argNode.raw.slice(1, wordEnd - argNode.start), quote = argNode.raw.charAt(0);
    if (word && word.charAt(word.length - 1) == quote)
      word = word.slice(0, word.length - 1);
    var completions = completeModuleName(query, file, word);
    if (argNode.end == wordEnd + 1 && file.text.charAt(wordEnd) == quote)
      ++wordEnd;
    return {
      start: tern.outputPos(query, file, argNode.start),
      end: tern.outputPos(query, file, wordEnd),
      isProperty: false,
      isObjectKey: false,
      completions: completions.map(function(rec) {
        var name = typeof rec == "string" ? rec : rec.name;
        var string = JSON.stringify(name);
        if (quote == "'") string = quote + string.slice(1, string.length -1).replace(/'/g, "\\'") + quote;
        if (typeof rec == "string") return string;
        rec.displayName = name;
        rec.name = string;
        return rec;
      })
    };
  }
  
  function findNodeModule(argsNode, wordEnd) {
    for (var i = 0; i < argsNode.length; i++) {
      var argNode = argsNode[i];
      if (argNode.type == "Literal" && typeof argNode.value == "string" &&
          argNode.start < wordEnd && argNode.end > wordEnd) return argNode;
    }
  }
  
  function completeModuleName(query, file, word) {
    var completions = [];
    var cx = infer.cx(), server = cx.parent, modules = server._yui.modules, submodules = server._yui.submodules;
    var wrapAsObjs = query.types || query.depths || query.docs || query.urls || query.origins;

    function maybeSet(obj, prop, val) {
      if (val != null) obj[prop] = val;
    }
    
    function gather(modules, isSubModule) {
      for (var name in modules) {
        
        var moduleName = name;
        if (moduleName &&
            !(query.filter !== false && word &&
              (query.caseInsensitive ? moduleName.toLowerCase() : moduleName).indexOf(word) !== 0)) {
          var rec = wrapAsObjs ? {name: moduleName} : moduleName;
          completions.push(rec);

          if (query.types || query.docs || query.urls || query.origins) {
            var val = modules[name];          
            if (query.types)
              rec.type = isSubModule ? "submodule" : "module";
            if (query.docs)
              maybeSet(rec, "doc", val.doc);
            if (query.urls)
              maybeSet(rec, "url", val.url);
            if (query.origins)
              maybeSet(rec, "origin", val.origin);
          }
        }
      }
    }

    if (query.caseInsensitive) word = word.toLowerCase();
    gather(modules, false);
    gather(submodules, true);
    return completions;
  }
  
  var defs = {
 "!name": "yui3",
 "!define": {
  "config": {
   "AnimConfig": {
    "!proto": "config.BaseConfig",
    "node": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_node",
     "!doc": "The object to be animated.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "duration": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_duration",
     "!doc": "The length of the animation.  Defaults to \"1\" (second).",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "easing": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_easing",
     "!doc": "The method that will provide values to the attribute(s) during the animation.\nDefaults to \"Easing.easeNone\".",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "from": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_from",
     "!doc": "The starting values for the animated properties.\n\nFields may be strings, numbers, or functions.\nIf a function is used, the return value becomes the from value.\nIf no from value is specified, the DEFAULT_GETTER will be used.\nSupports any unit, provided it matches the \"to\" (or default)\nunit (e.g. `{width: 10em, color: rgb(0, 0, 0), borderColor: #ccc}`).\n\nIf using the default (px for length-based units), the unit may be omitted\n(e.g. `{width: 100}, borderColor: ccc}`, which defaults to pixels\nand hex, respectively).",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "to": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_to",
     "!doc": "The ending values for the animated properties.\n\nFields may be strings, numbers, or functions.\nSupports any unit, provided it matches the \"from\" (or default)\nunit (e.g. `{width: 50%, color: red, borderColor: #ccc}`).\n\nIf using the default (px for length-based units), the unit may be omitted\n(e.g. `{width: 100, borderColor: ccc}`, which defaults to pixels\nand hex, respectively).",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "startTime": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_startTime",
     "!doc": "Date stamp for the first frame of the animation.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "elapsedTime": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_elapsedTime",
     "!doc": "Current time the animation has been running.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "running": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_running",
     "!doc": "Whether or not the animation is currently running.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "iterations": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_iterations",
     "!doc": "The number of times the animation should run",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "iterationCount": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_iterationCount",
     "!doc": "The number of iterations that have occurred.\nResets when an animation ends (reaches iteration count or stop() called).",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "direction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_direction",
     "!doc": "How iterations of the animation should behave.\nPossible values are \"normal\" and \"alternate\".\nNormal will repeat the animation, alternate will reverse on every other pass.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "paused": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_paused",
     "!doc": "Whether or not the animation is currently paused.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "reverse": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#attribute_reverse",
     "!doc": "If true, the `from` and `to` attributes are swapped,\nand the animation is then run starting from `from`.",
     "!data": {
      "submodule": "anim-base"
     }
    }
   },
   "AppConfig": {
    "!proto": "config.App.BaseConfig",
    "views": {
     "!type": "+Object",
     "!doc": "Hash of view-name to metadata used to\n    declaratively describe an applications views and their relationship with\n    the app and other views. The views specified here will override any defaults\n    provided by the `views` object on the `prototype`."
    }
   },
   "App": {
    "ContentShowContentConfig": {
     "view": {
      "!type": "+Object|string",
      "!doc": "The name of a view defined in this\n      apps `views`, or an object with the following properties:"
     }
    },
    "TransitionsConfig": {
     "transitions": {
      "!type": "bool|+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Transitions.html#attribute_transitions",
      "!doc": "Whether or not this application should use view transitions, and if so then\nwhich ones or `true` for the defaults which are specified by the\n`transitions` prototype property.\n\n**Note:** Transitions are an opt-in feature and will only be used in\nbrowsers which support native CSS3 transitions.",
      "!data": {
       "submodule": "app-transitions"
      }
     }
    },
    "TransitionsShowViewConfig": {
     "callback": {
      "!type": "fn()",
      "!doc": "Optional callback function to call\n    after new `activeView` is ready to use, the function will be passed:"
     },
     "prepend": {
      "!type": "bool",
      "!doc": "Whether the `view` should be\n    prepended instead of appended to the `viewContainer`."
     },
     "render": {
      "!type": "bool",
      "!doc": "Whether the `view` should be rendered.\n    **Note:** If no value is specified, a view instance will only be\n    rendered if its newly created by this method."
     },
     "transition": {
      "!type": "bool|string",
      "!doc": "Optional transition override.\n    A transition can be specified which will override the default, or\n    `false` for no transition."
     },
     "update": {
      "!type": "bool",
      "!doc": "Whether an existing view should\n    have its attributes updated by passing the `config` object to its\n    `setAttrs()` method. **Note:** This option does not have an effect if\n    the `view` instance is created as a result of calling this method."
     }
    },
    "BaseConfig": {
     "!proto": "config.BaseConfig",
     "views": {
      "!type": "+Object",
      "!doc": "Hash of view-name to metadata used to\n    declaratively describe an applications views and their relationship with\n    the app and other views. The views specified here will override any defaults\n    provided by the `views` object on the `prototype`."
     },
     "activeView": {
      "!type": "+app.View",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#attribute_activeView",
      "!doc": "The applications active/visible view.\n\nThis attribute is read-only, to set the `activeView` use the\n`showView()` method.",
      "!data": {
       "submodule": "app-base"
      }
     },
     "container": {
      "!type": "+HTMLElement|+node.Node|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#attribute_container",
      "!doc": "Container node which represents the applications bounding-box, into\nwhich this apps content will be rendered.\n\nThe container node serves as the host for all DOM events attached by the\napp. Delegation is used to handle events on children of the container,\nallowing the containers contents to be re-rendered at any time without\nlosing event subscriptions.\n\nThe default container is the `<body>` Node, but you can override this in\na subclass, or by passing in a custom `container` config value at\ninstantiation time.\n\nWhen `container` is overridden by a subclass or passed as a config\noption at instantiation time, it may be provided as a selector string, a\nDOM element, or a `Y.Node` instance. During initialization, this apps\n`create()` method will be called to convert the container into a\n`Y.Node` instance if it isnt one already and stamp it with the CSS\nclass: `\"yui3-app\"`.\n\nThe container is not added to the page automatically. This allows you to\nhave full control over how and when your app is actually rendered to\nthe page.",
      "!data": {
       "submodule": "app-base"
      }
     },
     "html5": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#attribute_html5",
      "!doc": "Whether or not this browser is capable of using HTML5 history.\n\nThis value is dependent on the value of `serverRouting` and will default\naccordingly.\n\nSetting this to `false` will force the use of hash-based history even on\nHTML5 browsers, but please dont do this unless you understand the\nconsequences.",
      "!data": {
       "submodule": "app-base"
      }
     },
     "linkSelector": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#attribute_linkSelector",
      "!doc": "CSS selector string used to filter link click events so that only the\nlinks which match it will have the enhanced-navigation behavior of pjax\napplied.\n\nWhen a link is clicked and that link matches this selector, navigating\nto the links `href` URL using the enhanced, pjax, behavior will be\nattempted; and the browsers default way to navigate to new pages will\nbe the fallback.\n\nBy default this selector will match _all_ links on the page.",
      "!data": {
       "submodule": "app-base"
      }
     },
     "serverRouting": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#attribute_serverRouting",
      "!doc": "Whether or not this applications server is capable of properly routing\nall requests and rendering the initial state in the HTML responses.\n\nThis can have three different values, each having particular\nimplications on how the app will handle routing and navigation:\n\n  * `undefined`: The best form of URLs will be chosen based on the\n    capabilities of the browser. Given no information about the server\n    environmentm a balanced approach to routing and navigation is\n    chosen.\n\n    The server should be capable of handling full-path requests, since\n    full-URLs will be generated by browsers using HTML5 history. If this\n    is a client-side-only app the server could handle full-URL requests\n    by sending a redirect back to the root with a hash-based URL, e.g:\n\n        Request:     http://example.com/users/1\n        Redirect to: http://example.com/#/users/1\n\n  * `true`: The server is *fully* capable of properly handling requests\n    to all full-path URLs the app can produce.\n\n    This is the best option for progressive-enhancement because it will\n    cause **all URLs to always have full-paths**, which means the server\n    will be able to accurately handle all URLs this app produces. e.g.\n\n        http://example.com/users/1\n\n    To meet this strict full-URL requirement, browsers which are not\n    capable of using HTML5 history will make requests to the server\n    resulting in full-page reloads.\n\n  * `false`: The server is *not* capable of properly handling requests\n    to all full-path URLs the app can produce, therefore all routing\n    will be handled by this App instance.\n\n    Be aware that this will cause **all URLs to always be hash-based**,\n    even in browsers that are capable of using HTML5 history. e.g.\n\n        http://example.com/#/users/1\n\n    A single-page or client-side-only app where the server sends a\n    \"shell\" page with JavaScript to the client might have this\n    restriction. If youre setting this to `false`, read the following:\n\n**Note:** When this is set to `false`, the server will *never* receive\nthe full URL because browsers do not send the fragment-part to the\nserver, that is everything after and including the \"#\".\n\nConsider the following example:\n\n    URL shown in browser: http://example.com/#/users/1\n    URL sent to server:   http://example.com/\n\nYou should feel bad about hurting our precious web if you forcefully set\neither `serverRouting` or `html5` to `false`, because youre basically\npunching the web in the face here with your lossy URLs! Please make sure\nyou know what youre doing and that you understand the implications.\n\nIdeally you should always prefer full-path URLs (not /#/foo/), and want\nfull-page reloads when the clients browser is not capable of enhancing\nthe experience using the HTML5 history APIs. Setting this to `true` is\nthe best option for progressive-enhancement (and graceful-degradation).",
      "!data": {
       "submodule": "app-base"
      }
     },
     "viewContainer": {
      "!type": "+HTMLElement|+node.Node|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#attribute_viewContainer",
      "!doc": "The node into which this apps `views` will be rendered when they become\nthe `activeView`.\n\nThe view container node serves as the container to hold the apps\n`activeView`. Each time the `activeView` is set via `showView()`, the\nprevious view will be removed from this node, and the new active views\n`container` node will be appended.\n\nThe default view container is a `<div>` Node, but you can override this\nin a subclass, or by passing in a custom `viewContainer` config value at\ninstantiation time. The `viewContainer` may be provided as a selector\nstring, DOM element, or a `Y.Node` instance (having the `viewContainer`\nand the `container` be the same node is also supported).\n\nThe apps `render()` method will stamp the view container with the CSS\nclass `\"yui3-app-views\"` and append it to the apps `container` node if\nit isnt already, and any `activeView` will be appended to this node if\nit isnt already.",
      "!data": {
       "submodule": "app-base"
      }
     }
    },
    "BaseNavigateConfig": {
     "replace": {
      "!type": "bool",
      "!doc": "Whether or not the current history\n      entry will be replaced, or a new entry will be created. Will default\n      to `true` if the specified `url` is the same as the current URL."
     },
     "force": {
      "!type": "bool",
      "!doc": "Whether the enhanced navigation\n      should occur even in browsers without HTML5 history. Will default to\n      `true` when `serverRouting` is falsy."
     }
    },
    "BaseShowViewConfig": {
     "callback": {
      "!type": "fn()",
      "!doc": "Optional callback function to call\n    after new `activeView` is ready to use, the function will be passed:"
     },
     "prepend": {
      "!type": "bool",
      "!doc": "Whether the `view` should be\n    prepended instead of appended to the `viewContainer`."
     },
     "render": {
      "!type": "bool",
      "!doc": "Whether the `view` should be rendered.\n    **Note:** If no value is specified, a view instance will only be\n    rendered if its newly created by this method."
     },
     "update": {
      "!type": "bool",
      "!doc": "Whether an existing view should\n    have its attributes updated by passing the `config` object to its\n    `setAttrs()` method. **Note:** This option does not have an effect if\n    the `view` instance is created as a result of calling this method."
     }
    }
   },
   "ModelSync": {
    "RESTSyncConfig": {
     "csrfToken": {
      "!type": "string",
      "!doc": "The authenticity token used by the\n    server to verify the validity of this request and protected against CSRF\n    attacks. This overrides the default value provided by the static\n    `CSRF_TOKEN` property."
     },
     "headers": {
      "!type": "+Object",
      "!doc": "The HTTP headers to mix with the default\n    headers specified by the static `HTTP_HEADERS` property."
     },
     "timeout": {
      "!type": "number",
      "!doc": "The number of milliseconds before the\n    request will timeout and be aborted. This overrides the default provided\n    by the static `HTTP_TIMEOUT` property."
     }
    }
   },
   "ModelListConfig": {
    "!proto": "config.BaseConfig",
    "items": {
     "!type": "+app.Model|[+app.Model]|+app.ModelList|+Object|[+Object]",
     "!doc": "Model\n        instance, array of model instances, or ModelList to add to this list on\n        init. The `add` event will not be fired for models added on init."
    }
   },
   "ModelListAddConfig": {
    "index": {
     "!type": "number",
     "!doc": "Index at which to insert the added\n        models. If not specified, the models will automatically be inserted\n        in the appropriate place according to the current sort order as\n        dictated by the `comparator()` method, if any."
    },
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, no `add` event(s)\n        will be fired."
    }
   },
   "ModelListCreateConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, no `add` event(s) will\n      be fired."
    }
   },
   "ModelListFilterConfig": {
    "asList": {
     "!type": "bool",
     "!doc": "If truthy, results will be\n        returned as a new ModelList instance rather than as an array."
    }
   },
   "ModelListRemoveConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, no `remove` event(s)\n        will be fired."
    }
   },
   "ModelListResetConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, no `reset` event will\n        be fired."
    }
   },
   "ModelListSortConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, no `reset` event will\n      be fired."
    },
    "descending": {
     "!type": "bool",
     "!doc": "If `true`, the sort is\n      performed in descending order."
    }
   },
   "ModelDestroyConfig": {
    "remove": {
     "!type": "bool",
     "!doc": "If `true`, the model will be\n    deleted via the sync layer in addition to the instance being destroyed."
    }
   },
   "ModelSetConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, no `change` event will\n      be fired."
    }
   },
   "ModelSetAttrsConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, no `change` event will\n      be fired."
    }
   },
   "ModelUndoConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, no `change` event will\n      be fired."
    }
   },
   "ModelConfig": {
    "!proto": "config.BaseConfig",
    "clientId": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#attribute_clientId",
     "!doc": "A client-only identifier for this model.\n\nLike the `id` attribute, `clientId` may be used to retrieve model\ninstances from lists. Unlike the `id` attribute, `clientId` is\nautomatically generated, and is only intended to be used on the client\nduring the current pageview.",
     "!data": {
      "submodule": "model"
     }
    },
    "id": {
     "!type": "string|number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#attribute_id",
     "!doc": "A unique identifier for this model. Among other things, this id may be\nused to retrieve model instances from lists, so it should be unique.\n\nIf the id is empty, this model instance is assumed to represent a new\nitem that hasnt yet been saved.\n\nIf you would prefer to use a custom attribute as this models id instead\nof using the `id` attribute (for example, maybe youd rather use `_id`\nor `uid` as the primary id), you may set the `idAttribute` property to\nthe name of your custom id attribute. The `id` attribute will then\nact as an alias for your custom attribute.",
     "!data": {
      "submodule": "model"
     }
    }
   },
   "RouterConfig": {
    "!proto": "config.BaseConfig",
    "html5": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#attribute_html5",
     "!doc": "Whether or not this browser is capable of using HTML5 history.\n\nSetting this to `false` will force the use of hash-based history even on\nHTML5 browsers, but please dont do this unless you understand the\nconsequences.",
     "!data": {
      "submodule": "router"
     }
    },
    "root": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#attribute_root",
     "!doc": "Absolute root path from which all routes should be evaluated.\n\nFor example, if your router is running on a page at\n`http://example.com/myapp/` and you add a route with the path `/`, your\nroute will never execute, because the path will always be preceded by\n`/myapp`. Setting `root` to `/myapp` would cause all routes to be\nevaluated relative to that root URL, so the `/` route would then execute\nwhen the user browses to `http://example.com/myapp/`.",
     "!data": {
      "submodule": "router"
     }
    },
    "routes": {
     "!type": "[+Object]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#attribute_routes",
     "!doc": "Array of route objects.\n\nEach item in the array must be an object with the following properties\nin order to be processed by the router:\n\n  * `path`: String or regex representing the path to match. See the docs\n    for the `route()` method for more details.\n\n  * `callbacks`: Function or a string representing the name of a\n    function on this router instance that should be called when the\n    route is triggered. An array of functions and/or strings may also be\n    provided. See the docs for the `route()` method for more details.\n\nIf a route object contains a `regex` or `regexp` property, or if its\n`path` is a regular express, then the route will be considered to be\nfully-processed. Any fully-processed routes may contain the following\nproperties:\n\n  * `regex`: The regular expression representing the path to match, this\n    property may also be named `regexp` for greater compatibility.\n\n  * `keys`: Array of named path parameters used to populate `req.params`\n    objects when dispatching to route handlers.\n\nAny additional data contained on these route objects will be retained.\nThis is useful to store extra metadata about a route; e.g., a `name` to\ngive routes logical names.\n\nThis attribute is intended to be used to set routes at init time, or to\ncompletely reset all routes after init. To add routes after init without\nresetting all existing routes, use the `route()` method.",
     "!data": {
      "submodule": "router"
     }
    },
    "params": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#attribute_params",
     "!doc": "Map of params handlers in the form: `name` -> RegExp | Function.\n\nIf a param handler regex or function returns a value of `false`, `null`,\n`undefined`, or `NaN`, the current route will not match and be skipped.\nAll other return values will be used in place of the original param\nvalue parsed from the URL.\n\nThis attribute is intended to be used to set params at init time, or to\ncompletely reset all params after init. To add params after init without\nresetting all existing params, use the `param()` method.",
     "!data": {
      "submodule": "router"
     }
    }
   },
   "param0Config": {
    "params": {
     "!type": "[?]|+Object",
     "!doc": "Captured parameters matched\n      by the route path specification. If a string path was used and\n      contained named parameters, then this will be a key/value hash mapping\n      parameter names to their matched values. If a regex path was used,\n      this will be an array of subpattern matches starting at index 0 for\n      the full match, then 1 for the first subpattern match, and so on."
    },
    "path": {
     "!type": "string",
     "!doc": "The current URL path."
    },
    "pendingCallbacks": {
     "!type": "number",
     "!doc": "Number of remaining\n      callbacks the route handler has after this one in the dispatch chain."
    },
    "pendingRoutes": {
     "!type": "number",
     "!doc": "Number of matching routes\n      after this one in the dispatch chain."
    },
    "query": {
     "!type": "+Object",
     "!doc": "Query hash representing the URL\n      query string, if any. Parameter names are keys, and are mapped to\n      parameter values."
    },
    "route": {
     "!type": "+Object",
     "!doc": "Reference to the current route\n      object whose callbacks are being dispatched."
    },
    "router": {
     "!type": "+Object",
     "!doc": "Reference to this router instance."
    },
    "src": {
     "!type": "string",
     "!doc": "What initiated the dispatch. In an\n      HTML5 browser, when the back/forward buttons are used, this property\n      will have a value of \"popstate\". When the `dispath()` method is\n      called, the `src` will be `\"dispatch\"`."
    },
    "url": {
     "!type": "string",
     "!doc": "The full URL."
    }
   },
   "param1Config": {
    "req": {
     "!type": "+Object",
     "!doc": "Reference to the request object."
    }
   },
   "ViewDestroyConfig": {
    "remove": {
     "!type": "bool",
     "!doc": "If `true`, this Views container\n        will be removed from the DOM and destroyed as well."
    }
   },
   "ViewConfig": {
    "!proto": "config.BaseConfig",
    "container": {
     "!type": "+HTMLElement|+node.Node|string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#attribute_container",
     "!doc": "Container node into which this views content will be rendered.\n\nThe container node serves as the host for all DOM events attached by the\nview. Delegation is used to handle events on children of the container,\nallowing the containers contents to be re-rendered at any time without\nlosing event subscriptions.\n\nThe default container is a `<div>` Node, but you can override this in\na subclass, or by passing in a custom `container` config value at\ninstantiation time. If you override the default container in a subclass\nusing `ATTRS`, you must use the `valueFn` property. The views constructor\nwill ignore any assignments using `value`.\n\nWhen `container` is overridden by a subclass or passed as a config\noption at instantiation time, you can provide it as a selector string, a\nDOM element, a `Y.Node` instance, or (if you are subclassing and modifying\nthe attribute), a `valueFn` function that returns a `Y.Node` instance.\nThe value will be converted into a `Y.Node` instance if it isnt one\nalready.\n\nThe container is not added to the page automatically. This allows you to\nhave full control over how and when your view is actually rendered to\nthe page.",
     "!data": {
      "submodule": "view"
     }
    }
   },
   "ArraySortNaturalCompareConfig": {
    "caseSensitive": {
     "!type": "bool",
     "!doc": "If `true`, a\n        case-sensitive comparison will be performed. By default the\n        comparison is case-insensitive."
    },
    "descending": {
     "!type": "bool",
     "!doc": "If `true`, the sort order\n        will be reversed so that larger values are sorted before smaller\n        values."
    }
   },
   "AutoCompleteBaseConfig": {
    "allowBrowserAutocomplete": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_allowBrowserAutocomplete",
     "!doc": "Whether or not to enable the browsers built-in autocomplete functionality\nfor input fields.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "allowTrailingDelimiter": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_allowTrailingDelimiter",
     "!doc": "When a `queryDelimiter` is set, trailing delimiters will automatically be\nstripped from the input value by default when the input node loses focus.\nSet this to `true` to allow trailing delimiters.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "enableCache": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_enableCache",
     "!doc": "Whether or not to enable in-memory caching in result sources that support\nit.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "inputNode": {
     "!type": "+node.Node|+HTMLElement|string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_inputNode",
     "!doc": "Node to monitor for changes, which will generate `query` events when\nappropriate. May be either an `<input>` or a `<textarea>`.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "maxResults": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_maxResults",
     "!doc": "Maximum number of results to return. A value of `0` or less will allow an\nunlimited number of results.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "minQueryLength": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_minQueryLength",
     "!doc": "Minimum number of characters that must be entered before a `query` event\nwill be fired. A value of `0` allows empty queries; a negative value will\neffectively disable all `query` events.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "query": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_query",
     "!doc": "Current query, or `null` if there is no current query.\n\nThe query might not be the same as the current value of the input node, both\nfor timing reasons (due to `queryDelay`) and because when one or more\n`queryDelimiter` separators are in use, only the last portion of the\ndelimited input string will be used as the query value.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "queryDelay": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_queryDelay",
     "!doc": "Number of milliseconds to delay after input before triggering a `query`\nevent. If new input occurs before this delay is over, the previous input\nevent will be ignored and a new delay will begin.\n\nThis can be useful both to throttle queries to a remote data source and to\navoid distracting the user by showing them less relevant results before\ntheyve paused their typing.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "queryDelimiter": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_queryDelimiter",
     "!doc": "Query delimiter string. When a delimiter is configured, the input value\nwill be split on the delimiter, and only the last portion will be used in\nautocomplete queries and updated when the `query` attribute is\nmodified.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "requestTemplate": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_requestTemplate",
     "!doc": "Source request template. This can be a function that accepts a query as a\nparameter and returns a request string, or it can be a string containing the\nplaceholder \"{query}\", which will be replaced with the actual URI-encoded\nquery. In either case, the resulting string will be appended to the request\nURL when the `source` attribute is set to a remote DataSource, JSONP URL, or\nXHR URL (it will not be appended to YQL URLs).\n\nWhile `requestTemplate` may be set to either a function or a string, it will\nalways be returned as a function that accepts a query argument and returns a\nstring.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "resultFilters": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_resultFilters",
     "!doc": "Array of local result filter functions. If provided, each filter will be\ncalled with two arguments when results are received: the query and an array\nof result objects. See the documentation for the `results` event for a list\nof the properties available on each result object.\n\nEach filter is expected to return a filtered or modified version of the\nresults array, which will then be passed on to subsequent filters, then the\n`resultHighlighter` function (if set), then the `resultFormatter` function\n(if set), and finally to subscribers to the `results` event.\n\nIf no `source` is set, result filters will not be called.\n\nPrepackaged result filters provided by the autocomplete-filters and\nautocomplete-filters-accentfold modules can be used by specifying the filter\nname as a string, such as `phraseMatch` (assuming the necessary filters\nmodule is loaded).",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "resultFormatter": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_resultFormatter",
     "!doc": "Function which will be used to format results. If provided, this function\nwill be called with two arguments after results have been received and\nfiltered: the query and an array of result objects. The formatter is\nexpected to return an array of HTML strings or Node instances containing the\ndesired HTML for each result.\n\nSee the documentation for the `results` event for a list of the properties\navailable on each result object.\n\nIf no `source` is set, the formatter will not be called.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "resultHighlighter": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_resultHighlighter",
     "!doc": "Function which will be used to highlight results. If provided, this function\nwill be called with two arguments after results have been received and\nfiltered: the query and an array of filtered result objects. The highlighter\nis expected to return an array of highlighted result text in the form of\nHTML strings.\n\nSee the documentation for the `results` event for a list of the properties\navailable on each result object.\n\nIf no `source` is set, the highlighter will not be called.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "resultListLocator": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_resultListLocator",
     "!doc": "Locator that should be used to extract an array of results from a non-array\nresponse.\n\nBy default, no locator is applied, and all responses are assumed to be\narrays by default. If all responses are already arrays, you dont need to\ndefine a locator.\n\nThe locator may be either a function (which will receive the raw response as\nan argument and must return an array) or a string representing an object\npath, such as \"foo.bar.baz\" (which would return the value of\n`result.foo.bar.baz` if the response is an object).\n\nWhile `resultListLocator` may be set to either a function or a string, it\nwill always be returned as a function that accepts a response argument and\nreturns an array.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "results": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_results",
     "!doc": "Current results, or an empty array if there are no results.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "resultTextLocator": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_resultTextLocator",
     "!doc": "Locator that should be used to extract a plain text string from a non-string\nresult item. The resulting text value will typically be the value that ends\nup being inserted into an input field or textarea when the user of an\nautocomplete implementation selects a result.\n\nBy default, no locator is applied, and all results are assumed to be plain\ntext strings. If all results are already plain text strings, you dont need\nto define a locator.\n\nThe locator may be either a function (which will receive the raw result as\nan argument and must return a string) or a string representing an object\npath, such as \"foo.bar.baz\" (which would return the value of\n`result.foo.bar.baz` if the result is an object).\n\nWhile `resultTextLocator` may be set to either a function or a string, it\nwill always be returned as a function that accepts a result argument and\nreturns a string.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "source": {
     "!type": "[?]|+DataSource|+node.Node|+Object|string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_source",
     "!doc": "Source for autocomplete results. The following source types are supported:\n\n<dl>\n  <dt>Array</dt>\n  <dd>\n    <p>\n    The full array will be provided to any configured filters for each\n    query. This is an easy way to create a fully client-side autocomplete\n    implementation.\n    </p>\n\n    <p>\n    Example: `[first result, second result, etc]`\n    </p>\n  </dd>\n\n  <dt>DataSource</dt>\n  <dd>\n    A `DataSource` instance or other object that provides a DataSource-like\n    `sendRequest` method. See the `DataSource` documentation for details.\n  </dd>\n\n  <dt>Function</dt>\n  <dd>\n    <p>\n    A function source will be called with the current query and a\n    callback function as parameters, and should either return an array of\n    results (for synchronous operation) or return nothing and pass an\n    array of results to the provided callback (for asynchronous\n    operation).\n    </p>\n\n    <p>\n    Example (synchronous):\n    </p>\n\n    <pre>\n    function (query) {\n        return [foo, bar];\n    }\n    </pre>\n\n    <p>\n    Example (async):\n    </p>\n\n    <pre>\n    function (query, callback) {\n        callback([foo, bar]);\n    }\n    </pre>\n  </dd>\n\n  <dt>Object</dt>\n  <dd>\n    <p>\n    An object will be treated as a query hashmap. If a property on the\n    object matches the current query, the value of that property will be\n    used as the response.\n    </p>\n\n    <p>\n    The response is assumed to be an array of results by default. If the\n    response is not an array, provide a `resultListLocator` to\n    process the response and return an array.\n    </p>\n\n    <p>\n    Example: `{foo: [foo result 1, foo result 2], bar: [bar result]}`\n    </p>\n  </dd>\n</dl>\n\nIf the optional `autocomplete-sources` module is loaded, then\nthe following additional source types will be supported as well:\n\n<dl>\n  <dt>&lt;select&gt; Node</dt>\n  <dd>\n    You may provide a YUI Node instance wrapping a &lt;select&gt;\n    element, and the options in the list will be used as results. You\n    will also need to specify a `resultTextLocator` of text\n    or value, depending on what you want to use as the text of the\n    result.\n\n    Each result will be an object with the following properties:\n\n    <dl>\n      <dt>html (String)</dt>\n      <dd>\n        <p>HTML content of the &lt;option&gt; element.</p>\n      </dd>\n\n      <dt>index (Number)</dt>\n      <dd>\n        <p>Index of the &lt;option&gt; element in the list.</p>\n      </dd>\n\n      <dt>node (Y.Node)</dt>\n      <dd>\n        <p>Node instance referring to the original &lt;option&gt; element.</p>\n      </dd>\n\n      <dt>selected (Boolean)</dt>\n      <dd>\n        <p>Whether or not this item is currently selected in the\n        &lt;select&gt; list.</p>\n      </dd>\n\n      <dt>text (String)</dt>\n      <dd>\n        <p>Text content of the &lt;option&gt; element.</p>\n      </dd>\n\n      <dt>value (String)</dt>\n      <dd>\n        <p>Value of the &lt;option&gt; element.</p>\n      </dd>\n    </dl>\n  </dd>\n\n  <dt>String (JSONP URL)</dt>\n  <dd>\n    <p>\n    If a URL with a `{callback}` placeholder is provided, it will be used to\n    make a JSONP request. The `{query}` placeholder will be replaced with\n    the current query, and the `{callback}` placeholder will be replaced\n    with an internally-generated JSONP callback name. Both placeholders must\n    appear in the URL, or the request will fail. An optional `{maxResults}`\n    placeholder may also be provided, and will be replaced with the value of\n    the maxResults attribute (or 1000 if the maxResults attribute is 0 or\n    less).\n    </p>\n\n    <p>\n    The response is assumed to be an array of results by default. If the\n    response is not an array, provide a `resultListLocator` to process the\n    response and return an array.\n    </p>\n\n    <p>\n    <strong>The `jsonp` module must be loaded in order for\n    JSONP URL sources to work.</strong> If the `jsonp` module\n    is not already loaded, it will be loaded on demand if possible.\n    </p>\n\n    <p>\n    Example: `http://example.com/search?q={query}&callback={callback}`\n    </p>\n  </dd>\n\n  <dt>String (XHR URL)</dt>\n  <dd>\n    <p>\n    If a URL without a `{callback}` placeholder is provided, it will be used\n    to make a same-origin XHR request. The `{query}` placeholder will be\n    replaced with the current query. An optional `{maxResults}` placeholder\n    may also be provided, and will be replaced with the value of the\n    maxResults attribute (or 1000 if the maxResults attribute is 0 or less).\n    </p>\n\n    <p>\n    The response is assumed to be a JSON array of results by default. If the\n    response is a JSON object and not an array, provide a\n    `resultListLocator` to process the response and return an array. If the\n    response is in some form other than JSON, you will need to use a custom\n    DataSource instance as the source.\n    </p>\n\n    <p>\n    <strong>The `io-base` and `json-parse` modules\n    must be loaded in order for XHR URL sources to work.</strong> If\n    these modules are not already loaded, they will be loaded on demand\n    if possible.\n    </p>\n\n    <p>\n    Example: `http://example.com/search?q={query}`\n    </p>\n  </dd>\n\n  <dt>String (YQL query)</dt>\n  <dd>\n    <p>\n    If a YQL query is provided, it will be used to make a YQL request. The\n    `{query}` placeholder will be replaced with the current autocomplete\n    query. This placeholder must appear in the YQL query, or the request\n    will fail. An optional `{maxResults}` placeholder may also be provided,\n    and will be replaced with the value of the maxResults attribute (or 1000\n    if the maxResults attribute is 0 or less).\n    </p>\n\n    <p>\n    <strong>The `yql` module must be loaded in order for YQL\n    sources to work.</strong> If the `yql` module is not\n    already loaded, it will be loaded on demand if possible.\n    </p>\n\n    <p>\n    Example: `select * from search.suggest where query=\"{query}\"`\n    </p>\n  </dd>\n</dl>\n\nAs an alternative to providing a source, you could simply listen for `query`\nevents and handle them any way you see fit. Providing a source is optional,\nbut will usually be simpler.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "sourceType": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_sourceType",
     "!doc": "May be used to force a specific source type, overriding the automatic source\ntype detection. It should almost never be necessary to do this, but as they\ntaught us in the Boy Scouts, one should always be prepared, so its here if\nyou need it. Be warned that if you set this attribute and something breaks,\nits your own fault.\n\nSupported `sourceType` values are: array, datasource, function, and\nobject.\n\nIf the `autocomplete-sources` module is loaded, the following additional\nsource types are supported: io, jsonp, select, string, yql",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "tokenInput": {
     "!type": "+Plugin.TokenInput",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_tokenInput",
     "!doc": "If the `inputNode` specified at instantiation time has a `node-tokeninput`\nplugin attached to it, this attribute will be a reference to the\n`Y.Plugin.TokenInput` instance.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "value": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_value",
     "!doc": "Current value of the input node.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    },
    "yqlEnv": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_yqlEnv",
     "!doc": "YQL environment file URL to load when the `source` is set to a YQL query.\nSet this to `null` to use the default Open Data Tables environment file\n(http://datatables.org/alltables.env).",
     "!data": {
      "submodule": "autocomplete-sources"
     }
    },
    "yqlProtocol": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#attribute_yqlProtocol",
     "!doc": "URL protocol to use when the `source` is set to a YQL query.",
     "!data": {
      "submodule": "autocomplete-sources"
     }
    }
   },
   "AutoCompleteListConfig": {
    "!proto": "config.WidgetConfig",
    "activateFirstItem": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_activateFirstItem",
     "!doc": "If `true`, the first item in the list will be activated by default when\nthe list is initially displayed and when results change.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    },
    "activeItem": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_activeItem",
     "!doc": "Item thats currently active, if any. When the user presses enter, this\nis the item that will be selected.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    },
    "alwaysShowList": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_alwaysShowList",
     "!doc": "If `true`, the list will remain visible even when there are no results\nto display.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    },
    "circular": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_circular",
     "!doc": "If `true`, keyboard navigation will wrap around to the opposite end of\nthe list when navigating past the first or last item.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    },
    "hoveredItem": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_hoveredItem",
     "!doc": "Item currently being hovered over by the mouse, if any.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    },
    "listNode": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_listNode",
     "!doc": "Node that will contain result items.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    },
    "scrollIntoView": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_scrollIntoView",
     "!doc": "If `true`, the viewport will be scrolled to ensure that the active list\nitem is visible when necessary.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    },
    "strings": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_strings",
     "!doc": "Translatable strings used by the AutoCompleteList widget.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    },
    "tabSelect": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#attribute_tabSelect",
     "!doc": "If `true`, pressing the tab key while the list is visible will select\nthe active item, if any.",
     "!data": {
      "submodule": "autocomplete-list"
     }
    }
   },
   "BaseCoreConfig": {
    "initialized": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#attribute_initialized",
     "!doc": "Flag indicating whether or not this object\nhas been through the init lifecycle phase.",
     "!data": {
      "submodule": "base-core"
     }
    },
    "destroyed": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#attribute_destroyed",
     "!doc": "Flag indicating whether or not this object\nhas been through the destroy lifecycle phase.",
     "!data": {
      "submodule": "base-core"
     }
    }
   },
   "ButtonConfig": {
    "!proto": "config.WidgetConfig",
    "label": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Button.html#attribute_label",
     "!doc": "The text of the buttons label"
    },
    "labelHTML": {
     "!type": "+HTML",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Button.html#attribute_labelHTML",
     "!doc": "The HTML of the buttons label\n\nThis attribute accepts HTML and inserts it into the DOM **without**\nsanitization.  This attribute should only be used with HTML that has\neither been escaped (using `Y.Escape.html`), or sanitized according to\nthe requirements of your application.\n\nIf all you need is support for text labels, please use the `label`\nattribute instead."
    }
   },
   "ToggleButtonConfig": {
    "!proto": "config.ButtonConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#attribute_type"
    }
   },
   "ButtonGroupConfig": {
    "!proto": "config.WidgetConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html#attribute_type"
    }
   },
   "CacheConfig": {
    "!proto": "config.BaseConfig",
    "max": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#attribute_max",
     "!doc": "Maximum number of entries the Cache can hold.\nSet to 0 to turn off caching.",
     "!data": {
      "submodule": "cache-base"
     }
    },
    "size": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#attribute_size",
     "!doc": "Number of entries currently cached.",
     "!data": {
      "submodule": "cache-base"
     }
    },
    "uniqueKeys": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#attribute_uniqueKeys",
     "!doc": "Validate uniqueness of stored keys. Default is false and\nis more performant.",
     "!data": {
      "submodule": "cache-base"
     }
    },
    "expires": {
     "!type": "+datatype_date.Date|number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#attribute_expires",
     "!doc": "Absolute Date when data expires or\nrelative number of milliseconds. Zero disables expiration.",
     "!data": {
      "submodule": "cache-base"
     }
    },
    "entries": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#attribute_entries",
     "!doc": "Cached entries.",
     "!data": {
      "submodule": "cache-base"
     }
    }
   },
   "CacheOfflineConfig": {
    "!proto": "config.CacheConfig",
    "sandbox": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html#attribute_sandbox",
     "!doc": "A string that must be passed in via the constructor.\nThis identifier is used to sandbox one cache instances entries\nfrom another. Calling the cache instances flush and length methods\nor get(\"entries\") will apply to only these sandboxed entries.",
     "!data": {
      "submodule": "cache-offline"
     }
    },
    "expires": {
     "!type": "+datatype_date.Date|number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html#attribute_expires",
     "!doc": "Absolute Date when data expires or\nrelative number of milliseconds. Zero disables expiration.",
     "!data": {
      "submodule": "cache-offline"
     }
    },
    "max": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html#attribute_max",
     "!doc": "Disabled.",
     "!data": {
      "submodule": "cache-offline"
     }
    },
    "uniqueKeys": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html#attribute_uniqueKeys",
     "!doc": "Always true for CacheOffline.",
     "!data": {
      "submodule": "cache-offline"
     }
    }
   },
   "CalendarBaseConfig": {
    "!proto": "config.WidgetConfig",
    "date": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#attribute_date",
     "!doc": "The date corresponding to the current calendar view. Always\nnormalized to the first of the month that contains the date\nat assignment time. Used as the first date visible in the\ncalendar.",
     "!data": {
      "submodule": "calendar-base"
     }
    },
    "showPrevMonth": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#attribute_showPrevMonth",
     "!doc": "A setting specifying whether to shows days from the previous\nmonth in the visible months grid, if there are empty preceding\ncells available.",
     "!data": {
      "submodule": "calendar-base"
     }
    },
    "showNextMonth": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#attribute_showNextMonth",
     "!doc": "A setting specifying whether to shows days from the next\nmonth in the visible months grid, if there are empty\ncells available at the end.",
     "!data": {
      "submodule": "calendar-base"
     }
    },
    "headerRenderer": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#attribute_headerRenderer",
     "!doc": "Custom header renderer for the calendar.",
     "!data": {
      "submodule": "calendar-base"
     }
    },
    "enabledDatesRule": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#attribute_enabledDatesRule",
     "!doc": "The name of the rule which all enabled dates should match.\nEither disabledDatesRule or enabledDatesRule should be specified,\nor neither, but not both.",
     "!data": {
      "submodule": "calendar-base"
     }
    },
    "disabledDatesRule": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#attribute_disabledDatesRule",
     "!doc": "The name of the rule which all disabled dates should match.\nEither disabledDatesRule or enabledDatesRule should be specified,\nor neither, but not both.",
     "!data": {
      "submodule": "calendar-base"
     }
    },
    "selectedDates": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#attribute_selectedDates",
     "!doc": "A read-only attribute providing a list of currently selected dates.",
     "!data": {
      "submodule": "calendar-base"
     }
    },
    "customRenderer": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#attribute_customRenderer",
     "!doc": "An object of the form {rules:Object, filterFunction:Function},\nproviding  set of rules and a custom rendering function for\ncustomizing specific calendar cells.",
     "!data": {
      "submodule": "calendar-base"
     }
    }
   },
   "CalendarConfig": {
    "!proto": "config.CalendarBaseConfig",
    "selectionMode": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#attribute_selectionMode",
     "!doc": "A setting specifying the type of selection the calendar allows.\nPossible values include:\n<ul>\n  <li>`single` - One date at a time</li>\n  <li>`multiple-sticky` - Multiple dates, selected one at a time (the dates \"stick\"). This option\n  is appropriate for mobile devices, where function keys from the keyboard are not available.</li>\n  <li>`multiple` - Multiple dates, selected with Ctrl/Meta keys for additional single\n  dates, and Shift key for date ranges.</li>"
    },
    "date": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#attribute_date",
     "!doc": "The date corresponding to the current calendar view. Always\nnormalized to the first of the month that contains the date\nat assignment time. Used as the first date visible in the\ncalendar."
    },
    "minimumDate": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#attribute_minimumDate",
     "!doc": "Unless minimumDate is null, it will not be possible to display and select dates earlier than this one."
    },
    "maximumDate": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#attribute_maximumDate",
     "!doc": "Unless maximumDate is null, it will not be possible to display and select dates later than this one."
    }
   },
   "Plugin": {
    "CalendarNavigatorConfig": {
     "!proto": "config.Plugin.BaseConfig",
     "shiftByMonths": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CalendarNavigator.html#attribute_shiftByMonths",
      "!doc": "The number of months to shift by when the control arrows are clicked."
     }
    },
    "ConsoleFiltersConfig": {
     "defaultVisibility": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#attribute_defaultVisibility",
      "!doc": "Default visibility applied to new categories and sources."
     },
     "category": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#attribute_category",
      "!doc": "<p>Map of entry categories to their visibility status.  Update a\nparticular categorys visibility by setting the subattribute to true\n(visible) or false (hidden).</p>\n\n<p>For example, yconsole.filter.set(category.info, false) to hide\nlog entries with the category/logLevel of info.</p>\n\n<p>Similarly, yconsole.filter.get(category.warn) will return a\nboolean indicating whether that category is currently being included\nin the UI.</p>\n\n<p>Unlike the YUI instance configurations logInclude and logExclude\nproperties, filtered entries are only hidden from the UI, but\ncan be made visible again.</p>"
     },
     "source": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#attribute_source",
      "!doc": "<p>Map of entry sources to their visibility status.  Update a\nparticular sourcess visibility by setting the subattribute to true\n(visible) or false (hidden).</p>\n\n<p>For example, yconsole.filter.set(sources.slider, false) to hide\nlog entries originating from Y.Slider.</p>"
     },
     "cacheLimit": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#attribute_cacheLimit",
      "!doc": "Maximum number of entries to store in the message cache.  Use this to\nlimit the memory footprint in environments with heavy log usage.\nBy default, there is no limit (Number.POSITIVE_INFINITY)."
     }
    },
    "DataTableDataSourceConfig": {
     "!proto": "config.Plugin.BaseConfig",
     "datasource": {
      "!type": "+DataSource",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DataTableDataSource.html#attribute_datasource",
      "!doc": "Pointer to DataSource instance.",
      "!data": {
       "submodule": "datatable-datasource"
      }
     },
     "initialRequest": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DataTableDataSource.html#attribute_initialRequest",
      "!doc": "Request sent to DataSource immediately upon initialization.",
      "!data": {
       "submodule": "datatable-datasource"
      }
     }
    },
    "DDConstrainedConfig": {
     "!proto": "config.BaseConfig",
     "stickX": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_stickX",
      "!doc": "Stick the drag movement to the X-Axis. Default: false",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "stickY": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_stickY",
      "!doc": "Stick the drag movement to the Y-Axis",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "tickX": {
      "!type": "number|bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_tickX",
      "!doc": "The X tick offset the drag node should snap to on each drag move. False for no ticks. Default: false",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "tickY": {
      "!type": "number|bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_tickY",
      "!doc": "The Y tick offset the drag node should snap to on each drag move. False for no ticks. Default: false",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "tickXArray": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_tickXArray",
      "!doc": "An array of page coordinates to use as X ticks for drag movement.",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "tickYArray": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_tickYArray",
      "!doc": "An array of page coordinates to use as Y ticks for drag movement.",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "gutter": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_gutter",
      "!doc": "CSS style string for the gutter of a region (supports negative values): 5 0\n(sets top and bottom to 5px, left and right to 0px), 1 2 3 4 (top 1px, right 2px, bottom 3px, left 4px)",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "constrain": {
      "!type": "string|+Object|+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_constrain",
      "!doc": "Will attempt to constrain the drag node to the boundaries. Arguments:<br>\nview: Contrain to Viewport<br>\n#selector_string: Constrain to this node<br>\n{Region Object}: An Object Literal containing a valid region (top, right, bottom, left) of page positions",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "constrain2region": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_constrain2region",
      "!doc": "An Object Literal containing a valid region (top, right, bottom, left) of page positions to constrain the drag node to.",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "constrain2node": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_constrain2node",
      "!doc": "Will attempt to constrain the drag node to the boundaries of this node.",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "constrain2view": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_constrain2view",
      "!doc": "Will attempt to constrain the drag node to the boundaries of the viewport region.",
      "!data": {
       "submodule": "dd-constrain"
      }
     },
     "cacheRegion": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#attribute_cacheRegion",
      "!doc": "Should the region be cached for performace. Default: true",
      "!data": {
       "submodule": "dd-constrain"
      }
     }
    },
    "DDProxyConfig": {
     "!proto": "config.BaseConfig",
     "moveOnEnd": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDProxy.html#attribute_moveOnEnd",
      "!doc": "Move the original node at the end of the drag. Default: true",
      "!data": {
       "submodule": "dd-proxy"
      }
     },
     "hideOnEnd": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDProxy.html#attribute_hideOnEnd",
      "!doc": "Hide the drag node at the end of the drag. Default: true",
      "!data": {
       "submodule": "dd-proxy"
      }
     },
     "resizeFrame": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDProxy.html#attribute_resizeFrame",
      "!doc": "Make the Proxy node assume the size of the original node. Default: true",
      "!data": {
       "submodule": "dd-proxy"
      }
     },
     "positionProxy": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDProxy.html#attribute_positionProxy",
      "!doc": "Make the Proxy node appear in the same place as the original node. Default: true",
      "!data": {
       "submodule": "dd-proxy"
      }
     },
     "borderStyle": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDProxy.html#attribute_borderStyle",
      "!doc": "The default border style for the border of the proxy. Default: 1px solid #808080",
      "!data": {
       "submodule": "dd-proxy"
      }
     },
     "cloneNode": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDProxy.html#attribute_cloneNode",
      "!doc": "Should the node be cloned into the proxy for you. Default: false",
      "!data": {
       "submodule": "dd-proxy"
      }
     }
    },
    "DDWindowScrollConfig": {
     "!proto": "config.ScrollConfig",
     "windowScroll": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDWindowScroll.html#attribute_windowScroll",
      "!doc": "Turn on window scroll support, default: true",
      "!data": {
       "submodule": "dd-scroll"
      }
     }
    },
    "DDNodeScrollConfig": {
     "!proto": "config.ScrollConfig",
     "node": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDNodeScroll.html#attribute_node",
      "!doc": "The node we want to scroll. Used to set the internal parentScroll attribute.",
      "!data": {
       "submodule": "dd-scroll"
      }
     }
    },
    "FlickConfig": {
     "deceleration": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#attribute_deceleration",
      "!doc": "Drag coefficent for inertial scrolling. The closer to 1 this\nvalue is, the less friction during scrolling."
     },
     "bounce": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#attribute_bounce",
      "!doc": "Drag coefficient for intertial scrolling at the upper\nand lower boundaries of the scrollview. Set to 0 to\ndisable \"rubber-banding\"."
     },
     "bounceDistance": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#attribute_bounceDistance",
      "!doc": "The bounce distance in pixels"
     },
     "minVelocity": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#attribute_minVelocity",
      "!doc": "The minimum flick gesture distance (px) for which to trigger the flick response"
     },
     "boundingBox": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#attribute_boundingBox",
      "!doc": "The constraining box relative to which the flick animation and bounds should be calculated."
     },
     "step": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#attribute_step",
      "!doc": "Time between flick animation frames."
     },
     "duration": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#attribute_duration",
      "!doc": "The custom duration to apply to the flick animation. By default,\nthe animation duration is controlled by the deceleration factor."
     },
     "easing": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#attribute_easing",
      "!doc": "The custom transition easing to use for the flick animation. If not\nprovided defaults to internally to Flick.EASING, or Flick.SNAP_EASING based\non whether or not were animating the flick or bounce step."
     }
    },
    "ScrollInfoConfig": {
     "!proto": "config.Plugin.BaseConfig",
     "scrollDelay": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollInfo.html#attribute_scrollDelay",
      "!doc": "Number of milliseconds to wait after a native `scroll` event before\nfiring local scroll events. If another native scroll event occurs during\nthis time, previous events will be ignored. This ensures that we dont\nfire thousands of events when the user is scrolling quickly."
     },
     "scrollMargin": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollInfo.html#attribute_scrollMargin",
      "!doc": "Additional margin in pixels beyond the onscreen region of the host node\nthat should be considered \"onscreen\".\n\nFor example, if set to 50, then a `scrollToBottom` event would be fired\nwhen the user scrolls to within 50 pixels of the bottom of the\nscrollable region, even if they dont actually scroll completely to the\nvery bottom pixel.\n\nThis margin also applies to the `getOffscreenNodes()` and\n`getOnscreenNodes()` methods by default."
     }
    },
    "BaseConfig": {
     "!proto": "config.BaseConfig",
     "host": {
      "!type": "+pluginhost_base.Plugin.Host",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Base.html#attribute_host",
      "!doc": "The plugins host object."
     }
    },
    "ResizeConstrainedConfig": {
     "constrain": {
      "!type": "string|+Object|+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeConstrained.html#attribute_constrain",
      "!doc": "Will attempt to constrain the resize node to the boundaries. Arguments:<br>\nview: Contrain to Viewport<br>\n#selector_string: Constrain to this node<br>\n{Region Object}: An Object Literal containing a valid region (top, right, bottom, left) of page positions",
      "!data": {
       "submodule": "resize-contrain"
      }
     },
     "minHeight": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeConstrained.html#attribute_minHeight",
      "!doc": "The minimum height of the element",
      "!data": {
       "submodule": "resize-contrain"
      }
     },
     "minWidth": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeConstrained.html#attribute_minWidth",
      "!doc": "The minimum width of the element",
      "!data": {
       "submodule": "resize-contrain"
      }
     },
     "maxHeight": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeConstrained.html#attribute_maxHeight",
      "!doc": "The maximum height of the element",
      "!data": {
       "submodule": "resize-contrain"
      }
     },
     "maxWidth": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeConstrained.html#attribute_maxWidth",
      "!doc": "The maximum width of the element",
      "!data": {
       "submodule": "resize-contrain"
      }
     },
     "preserveRatio": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeConstrained.html#attribute_preserveRatio",
      "!doc": "Maintain the elements ratio when resizing.",
      "!data": {
       "submodule": "resize-contrain"
      }
     },
     "tickX": {
      "!type": "number|bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeConstrained.html#attribute_tickX",
      "!doc": "The number of x ticks to span the resize to.",
      "!data": {
       "submodule": "resize-contrain"
      }
     },
     "tickY": {
      "!type": "number|bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeConstrained.html#attribute_tickY",
      "!doc": "The number of y ticks to span the resize to.",
      "!data": {
       "submodule": "resize-contrain"
      }
     }
    },
    "ResizeConfig": {
     "!proto": "config.ResizeConfig",
     "node": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Resize.html#attribute_node",
      "!doc": "Stores the node that is being resized",
      "!data": {
       "submodule": "resize-plugin"
      }
     },
     "widget": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Resize.html#attribute_widget",
      "!doc": "Stores the widget that the node belongs to, if one exists",
      "!data": {
       "submodule": "resize-plugin"
      }
     }
    },
    "ResizeProxyConfig": {
     "!proto": "config.Plugin.BaseConfig",
     "proxyNode": {
      "!type": "string|+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ResizeProxy.html#attribute_proxyNode",
      "!doc": "The Resize proxy element.",
      "!data": {
       "submodule": "resize-proxy"
      }
     }
    },
    "ScrollViewListConfig": {
     "!proto": "config.Plugin.BaseConfig",
     "isAttached": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollViewList.html#attribute_isAttached",
      "!doc": "Specifies whether the list elements (the immediate <ul>s and the\n immediate <li>s inside those <ul>s) have class names attached to\n them or not"
     }
    },
    "ScrollViewPaginatorConfig": {
     "!proto": "config.Plugin.BaseConfig",
     "axis": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollViewPaginator.html#attribute_axis",
      "!doc": "Specifies ability to scroll on x, y, or x and y axis/axes.\n If unspecified, it inherits from the host instance."
     },
     "selector": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollViewPaginator.html#attribute_selector",
      "!doc": "CSS selector for a page inside the scrollview. The scrollview\nwill snap to the closest page."
     },
     "index": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollViewPaginator.html#attribute_index",
      "!doc": "The active page number for a paged scrollview"
     },
     "total": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollViewPaginator.html#attribute_total",
      "!doc": "The total number of pages"
     }
    },
    "ScrollViewScrollbarsConfig": {
     "!proto": "config.Plugin.BaseConfig",
     "verticalNode": {
      "!type": "+Y.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollViewScrollbars.html#attribute_verticalNode",
      "!doc": "Vertical scrollbar node",
      "!data": {
       "submodule": "scrollview-scrollbars"
      }
     },
     "horizontalNode": {
      "!type": "+Y.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollViewScrollbars.html#attribute_horizontalNode",
      "!doc": "Horizontal scrollbar node",
      "!data": {
       "submodule": "scrollview-scrollbars"
      }
     }
    },
    "WidgetAnimConfig": {
     "!proto": "config.Plugin.BaseConfig",
     "duration": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.WidgetAnim.html#attribute_duration",
      "!doc": "Default duration in seconds. Used as the default duration for the default animation implementations"
     },
     "animShow": {
      "!type": "+anim.Anim",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.WidgetAnim.html#attribute_animShow",
      "!doc": "Default animation instance used for showing the widget (opacity fade-in)"
     },
     "animHide": {
      "!type": "+anim.Anim",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.WidgetAnim.html#attribute_animHide",
      "!doc": "Default animation instance used for hiding the widget (opacity fade-out)"
     }
    }
   },
   "AreaSeriesConfig": {
    "!proto": "config.CartesianSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AreaSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-area"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AreaSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing area fills. This attribute is inherited from `Renderer`. Below are the default values:\n\n <dl>\n     <dt>color</dt><dd>The color of the fill. The default value is determined by the order of the series on the graph. The color will be\n     retrieved from the following array:\n     `[\"#66007f\", \"#a86f41\", \"#295454\", \"#996ab2\", \"#e8cdb7\", \"#90bdbd\",\"#000000\",\"#c3b8ca\", \"#968373\", \"#678585\"]`\n     </dd>\n     <dt>alpha</dt><dd>Number between 0 and 1 that indicates the opacity of the fill. The default value is 1</dd>\n </dl>",
     "!data": {
      "submodule": "series-area"
     }
    }
   },
   "AreaSplineSeriesConfig": {
    "!proto": "config.AreaSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AreaSplineSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-areaspline"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AreaSplineSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing area fills. This attribute is inherited from `Renderer`. Below are the default values:\n\n <dl>\n     <dt>color</dt><dd>The color of the fill. The default value is determined by the order of the series on the graph. The color will be\n     retrieved from the following array:\n     `[\"#66007f\", \"#a86f41\", \"#295454\", \"#996ab2\", \"#e8cdb7\", \"#90bdbd\",\"#000000\",\"#c3b8ca\", \"#968373\", \"#678585\"]`\n     </dd>\n     <dt>alpha</dt><dd>Number between 0 and 1 that indicates the opacity of the fill. The default value is 1</dd>\n </dl>",
     "!data": {
      "submodule": "series-areaspline"
     }
    }
   },
   "AxisConfig": {
    "!proto": "config.WidgetConfig",
    "width": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_width",
     "!doc": "When set, defines the width of a vertical axis instance. By default, vertical axes automatically size based\non their contents. When the width attribute is set, the axis will not calculate its width. When the width\nattribute is explicitly set, axis labels will postion themselves off of the the inner edge of the axis and the\ntitle, if present, will position itself off of the outer edge. If a specified width is less than the sum of\nthe axis contents, excess content will overflow.",
     "!data": {
      "submodule": "axis"
     }
    },
    "height": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_height",
     "!doc": "When set, defines the height of a horizontal axis instance. By default, horizontal axes automatically size based\non their contents. When the height attribute is set, the axis will not calculate its height. When the height\nattribute is explicitly set, axis labels will postion themselves off of the the inner edge of the axis and the\ntitle, if present, will position itself off of the outer edge. If a specified height is less than the sum of\nthe axis contents, excess content will overflow.",
     "!data": {
      "submodule": "axis"
     }
    },
    "graphic": {
     "!type": "+graphics.Graphic",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_graphic",
     "!doc": "The graphic in which the axis line and ticks will be rendered.",
     "!data": {
      "submodule": "axis"
     }
    },
    "node": {
     "!type": "+HTMLElement",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_node",
     "!doc": "Contains the contents of the axis.",
     "!data": {
      "submodule": "axis"
     }
    },
    "position": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_position",
     "!doc": "Direction of the axis.",
     "!data": {
      "submodule": "axis"
     }
    },
    "topTickOffset": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_topTickOffset",
     "!doc": "Distance determined by the tick styles used to calculate the distance between the axis\nline in relation to the top of the axis.",
     "!data": {
      "submodule": "axis"
     }
    },
    "bottomTickOffset": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_bottomTickOffset",
     "!doc": "Distance determined by the tick styles used to calculate the distance between the axis\nline in relation to the bottom of the axis.",
     "!data": {
      "submodule": "axis"
     }
    },
    "leftTickOffset": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_leftTickOffset",
     "!doc": "Distance determined by the tick styles used to calculate the distance between the axis\nline in relation to the left of the axis.",
     "!data": {
      "submodule": "axis"
     }
    },
    "rightTickOffset": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_rightTickOffset",
     "!doc": "Distance determined by the tick styles used to calculate the distance between the axis\nline in relation to the right side of the axis.",
     "!data": {
      "submodule": "axis"
     }
    },
    "labels": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_labels",
     "!doc": "Collection of labels used to render the axis.",
     "!data": {
      "submodule": "axis"
     }
    },
    "tickPoints": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_tickPoints",
     "!doc": "Collection of points used for placement of labels and ticks along the axis.",
     "!data": {
      "submodule": "axis"
     }
    },
    "overlapGraph": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_overlapGraph",
     "!doc": "Indicates whether the axis overlaps the graph. If an axis is the inner most axis on a given\nposition and the tick position is inside or cross, the axis will need to overlap the graph.",
     "!data": {
      "submodule": "axis"
     }
    },
    "title": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_title",
     "!doc": "Title for the axis. When specified, the title will display. The position of the title is determined by the axis position.\n<dl>\n    <dt>top</dt><dd>Appears above the axis and it labels. The default rotation is 0.</dd>\n    <dt>right</dt><dd>Appears to the right of the axis and its labels. The default rotation is 90.</dd>\n    <dt>bottom</dt><dd>Appears below the axis and its labels. The default rotation is 0.</dd>\n    <dt>left</dt><dd>Appears to the left of the axis and its labels. The default rotation is -90.</dd>\n</dl>",
     "!data": {
      "submodule": "axis"
     }
    },
    "appendLabelFunction": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_appendLabelFunction",
     "!doc": "Function used to append an axis value to an axis label. This function has the following signature:\n <dl>\n     <dt>textField</dt><dd>The axis label to be appended. (`HTMLElement`)</dd>\n     <dt>val</dt><dd>The value to attach to the text field. This method will accept an `HTMLELement`\n     or a `String`. This method does not use (`HTMLElement` | `String`)</dd>\n </dl>\nThe default method appends a value to the `HTMLElement` using the `appendChild` method. If the given\nvalue is a `String`, the method will convert the the value to a `textNode` before appending to the\n`HTMLElement`. This method will not convert an `HTMLString` to an `HTMLElement`.",
     "!data": {
      "submodule": "axis"
     }
    },
    "appendTitleFunction": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_appendTitleFunction",
     "!doc": "Function used to append a title value to the title object. This function has the following signature:\n <dl>\n     <dt>textField</dt><dd>The title text field to be appended. (`HTMLElement`)</dd>\n     <dt>val</dt><dd>The value to attach to the text field. This method will accept an `HTMLELement`\n     or a `String`. This method does not use (`HTMLElement` | `String`)</dd>\n </dl>\nThe default method appends a value to the `HTMLElement` using the `appendChild` method. If the given\nvalue is a `String`, the method will convert the the value to a `textNode` before appending to the\n`HTMLElement` element. This method will not convert an `HTMLString` to an `HTMLElement`.",
     "!data": {
      "submodule": "axis"
     }
    },
    "labelValues": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_labelValues",
     "!doc": "An array containing the unformatted values of the axis labels. By default, TimeAxis, NumericAxis and\nStackedAxis labelValues are determined by the majorUnit style. By default, CategoryAxis labels are\ndetermined by the values of the dataProvider.\n<p>When the labelValues attribute is explicitly set, the labelValues are dictated by the set value and\nthe position of ticks and labels are determined by where those values would fall on the axis. </p>",
     "!data": {
      "submodule": "axis"
     }
    },
    "hideFirstMajorUnit": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_hideFirstMajorUnit",
     "!doc": "Suppresses the creation of the the first visible label and tick.",
     "!data": {
      "submodule": "axis"
     }
    },
    "hideLastMajorUnit": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_hideLastMajorUnit",
     "!doc": "Suppresses the creation of the the last visible label and tick.",
     "!data": {
      "submodule": "axis"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#attribute_styles",
     "!doc": "Style properties used for drawing an axis. This attribute is inherited from `Renderer`. Below are the default values:\n <dl>\n     <dt>majorTicks</dt><dd>Properties used for drawing ticks.\n         <dl>\n             <dt>display</dt><dd>Position of the tick. Possible values are `inside`, `outside`, `cross` and `none`.\n             The default value is `inside`.</dd>\n             <dt>length</dt><dd>The length (in pixels) of the tick. The default value is 4.</dd>\n             <dt>color</dt><dd>The color of the tick. The default value is `#dad8c9`</dd>\n             <dt>weight</dt><dd>Number indicating the width of the tick. The default value is 1.</dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the tick. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>line</dt><dd>Properties used for drawing the axis line.\n         <dl>\n             <dt>weight</dt><dd>Number indicating the width of the axis line. The default value is 1.</dd>\n             <dt>color</dt><dd>The color of the axis line. The default value is `#dad8c9`.</dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the tick. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>majorUnit</dt><dd>Properties used to calculate the `majorUnit` for the axis.\n         <dl>\n             <dt>determinant</dt><dd>The algorithm used for calculating distance between ticks. The possible options are\n             `count` and `distance`. If the `determinant` is `count`, the axis ticks will spaced so that a specified number\n             of ticks appear on the axis. If the `determinant` is `distance`, the axis ticks will spaced out according to\n             the specified distance. The default value is `count`.</dd>\n             <dt>count</dt><dd>Number of ticks to appear on the axis when the `determinant` is `count`. The default value is 11.</dd>\n             <dt>distance</dt><dd>The distance (in pixels) between ticks when the `determinant` is `distance`. The default\n             value is 75.</dd>\n         </dl>\n     </dd>\n     <dt>label</dt><dd>Properties and styles applied to the axis labels.\n         <dl>\n             <dt>color</dt><dd>The color of the labels. The default value is `#808080`.</dd>\n             <dt>alpha</dt><dd>Number between 0 and 1 indicating the opacity of the labels. The default value is 1.</dd>\n             <dt>fontSize</dt><dd>The font-size of the labels. The default value is 85%</dd>\n             <dt>rotation</dt><dd>The rotation, in degrees (between -90 and 90) of the labels. The default value is 0.</dd>\n             <dt>offset</td><dd>A number between 0 and 1 indicating the relationship of the label to a tick. For a horizontal axis\n             label, a value of 0 will position the labels left side even to the the tick. A position of 1 would position the\n             right side of the label with the tick. A position of 0.5 would center the label horizontally with the tick. For a\n             vertical axis, a value of 0 would position the top of the label with the tick, a value of 1 would position the bottom\n             of the label with the tick and a value 0 would center the label vertically with the tick. The default value is 0.5.</dd>\n             <dt>margin</dt><dd>The distance between the label and the axis/tick. Depending on the position of the `Axis`,\n             only one of the properties used.\n                 <dl>\n                     <dt>top</dt><dd>Pixel value used for an axis with a `position` of `bottom`. The default value is 4.</dd>\n                     <dt>right</dt><dd>Pixel value used for an axis with a `position` of `left`. The default value is 4.</dd>\n                     <dt>bottom</dt><dd>Pixel value used for an axis with a `position` of `top`. The default value is 4.</dd>\n                     <dt>left</dt><dd>Pixel value used for an axis with a `position` of `right`. The default value is 4.</dd>\n                 </dl>\n             </dd>\n         </dl>\n     </dd>\n </dl>",
     "!data": {
      "submodule": "axis"
     }
    }
   },
   "AxisBaseConfig": {
    "!proto": "config.BaseConfig",
    "calculateEdgeOffset": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_calculateEdgeOffset",
     "!doc": "Determines whether and offset is automatically calculated for the edges of the axis.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "keys": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_keys",
     "!doc": "Hash of array identifed by a string value.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_type",
     "!doc": "Returns the type of axis data\n  <dl>\n      <dt>time</dt><dd>Manages time data</dd>\n      <dt>stacked</dt><dd>Manages stacked numeric data</dd>\n      <dt>numeric</dt><dd>Manages numeric data</dd>\n      <dt>category</dt><dd>Manages categorical data</dd>\n  </dl>",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "dataProvider": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_dataProvider",
     "!doc": "Instance of `ChartDataProvider` that the class uses\nto build its own data.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "dataMaximum": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_dataMaximum",
     "!doc": "The maximum value contained in the `data` array. Used for\n`maximum` when `autoMax` is true.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "maximum": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_maximum",
     "!doc": "The maximum value that will appear on an axis.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "dataMinimum": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_dataMinimum",
     "!doc": "The minimum value contained in the `data` array. Used for\n`minimum` when `autoMin` is true.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "minimum": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_minimum",
     "!doc": "The minimum value that will appear on an axis.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "setMax": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_setMax",
     "!doc": "Determines whether the maximum is calculated or explicitly\nset by the user.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "setMin": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_setMin",
     "!doc": "Determines whether the minimum is calculated or explicitly\nset by the user.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "data": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_data",
     "!doc": "Array of axis data",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "keyCollection": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_keyCollection",
     "!doc": "Array containing all the keys in the axis.",
     "!data": {
      "submodule": "axis-base"
     }
    },
    "labelFunctionScope": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#attribute_labelFunctionScope",
     "!doc": "Object which should have by the labelFunction",
     "!data": {
      "submodule": "axis-base"
     }
    }
   },
   "BarSeriesConfig": {
    "!proto": "config.MarkerSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BarSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-bar"
     }
    },
    "direction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BarSeries.html#attribute_direction",
     "!doc": "Indicates the direction of the category axis that the bars are plotted against.",
     "!data": {
      "submodule": "series-bar"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BarSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing markers. This attribute is inherited from `MarkerSeries`. Below are the default values:\n <dl>\n     <dt>fill</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the fill. The default value is determined by the order of the series on the graph. The color\n             will be retrieved from the below array:<br/>\n             `[\"#66007f\", \"#a86f41\", \"#295454\", \"#996ab2\", \"#e8cdb7\", \"#90bdbd\",\"#000000\",\"#c3b8ca\", \"#968373\", \"#678585\"]`\n             </dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>border</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the border. The default value is determined by the order of the series on the graph. The color\n             will be retrieved from the below array:<br/>\n             `[\"#205096\", \"#b38206\", \"#000000\", \"#94001e\", \"#9d6fa0\", \"#e55b00\", \"#5e85c9\", \"#adab9e\", \"#6ac291\", \"#006457\"]`\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n             <dt>weight</dt><dd>Number indicating the width of the border. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>height</dt><dd>indicates the width of the marker. The default value is 12.</dd>\n     <dt>over</dt><dd>hash containing styles for markers when highlighted by a `mouseover` event. The default\n     values for each style is null. When an over style is not set, the non-over value will be used. For example,\n     the default value for `marker.over.fill.color` is equivalent to `marker.fill.color`.</dd>\n </dl>",
     "!data": {
      "submodule": "series-bar"
     }
    }
   },
   "CandlestickSeriesConfig": {
    "!proto": "config.RangeSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CandlestickSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-candlestick"
     }
    },
    "graphic": {
     "!type": "+graphics.Graphic",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CandlestickSeries.html#attribute_graphic",
     "!doc": "The graphic in which drawings will be rendered.",
     "!data": {
      "submodule": "series-candlestick"
     }
    },
    "upcandle": {
     "!type": "+graphics.Path",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CandlestickSeries.html#attribute_upcandle",
     "!doc": "Reference to the candlestick used when the close value is higher than the open value.",
     "!data": {
      "submodule": "series-candlestick"
     }
    },
    "downcandle": {
     "!type": "+graphics.Path",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CandlestickSeries.html#attribute_downcandle",
     "!doc": "Reference to the candlestick used when the open value is higher than the close value.",
     "!data": {
      "submodule": "series-candlestick"
     }
    },
    "wick": {
     "!type": "+graphics.Path",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CandlestickSeries.html#attribute_wick",
     "!doc": "Reference to the line drawn between the high and low values.",
     "!data": {
      "submodule": "series-candlestick"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CandlestickSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing candles and wicks. This attribute is inherited from `RangeSeries`. Below are the default values:\n <dl>\n     <dt>upcandle</dt><dd>Properties for a candle representing a period that closes higher than it opens.\n         <dl>\n             <dt>fill</dt><dd>A hash containing the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the fill. The default value is \"#00aa00\".</dd>\n                     </dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n                 </dl>\n             </dd>\n             <dt>border</dt><dd>A hash containing the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the border. The default value is \"#000000\".</dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n                     <dt>weight</dt><dd>Number indicating the width of the border. The default value is 0.</dd>\n                 </dl>\n             </dd>\n         </dl>\n     </dd>\n     <dt>downcandle</dt><dd>Properties for a candle representing a period that opens higher than it closes.\n         <dl>\n             <dt>fill</dt><dd>A hash containing the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the fill. The default value is \"#aa0000\".</dd>\n                     </dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n                 </dl>\n             </dd>\n             <dt>border</dt><dd>A hash containing the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the border. The default value is \"#000000\".</dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n                     <dt>weight</dt><dd>Number indicating the width of the border. The default value is 0.</dd>\n                 </dl>\n             </dd>\n         </dl>\n     </dd>\n     <dt>wick</dt><dd>Properties for the wick, which is a line drawn from the high point of the period to the low point of the period.\n         <dl>\n             <dt>color</dt><dd>The color of the wick. The default value is \"#000000\".</dd>\n             <dt>weight</dt><dd>The weight of the wick. The default value is 1.</dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the wick. The default value is 1.</dd>\n         </dl>\n     </dd>\n </dl>",
     "!data": {
      "submodule": "series-candlestick"
     }
    }
   },
   "CartesianChartConfig": {
    "!proto": "config.ChartBaseConfig",
    "allowContentOverflow": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_allowContentOverflow",
     "!doc": "Indicates whether axis labels are allowed to overflow beyond the bounds of the charts content box.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_styles",
     "!doc": "Style properties for the chart. Contains a key indexed hash of the following:\n <dl>\n     <dt>series</dt><dd>A key indexed hash containing references to the `styles` attribute for each series in the chart.\n     Specific style attributes vary depending on the series:\n     <ul>\n         <li><a href=\"AreaSeries.html#attr_styles\">AreaSeries</a></li>\n         <li><a href=\"BarSeries.html#attr_styles\">BarSeries</a></li>\n         <li><a href=\"ColumnSeries.html#attr_styles\">ColumnSeries</a></li>\n         <li><a href=\"ComboSeries.html#attr_styles\">ComboSeries</a></li>\n         <li><a href=\"LineSeries.html#attr_styles\">LineSeries</a></li>\n         <li><a href=\"MarkerSeries.html#attr_styles\">MarkerSeries</a></li>\n         <li><a href=\"SplineSeries.html#attr_styles\">SplineSeries</a></li>\n     </ul>\n     </dd>\n     <dt>axes</dt><dd>A key indexed hash containing references to the `styles` attribute for each axes in the chart. Specific\n     style attributes can be found in the <a href=\"Axis.html#attr_styles\">Axis</a> class.</dd>\n     <dt>graph</dt><dd>A reference to the `styles` attribute in the chart. Specific style attributes can be found in the\n     <a href=\"Graph.html#attr_styles\">Graph</a> class.</dd>\n </dl>",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "axes": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_axes",
     "!doc": "Axes to appear in the chart. This can be a key indexed hash of axis instances or object literals\nused to construct the appropriate axes.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "seriesCollection": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_seriesCollection",
     "!doc": "Collection of series to appear on the chart. This can be an array of Series instances or object literals\nused to construct the appropriate series.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "stacked": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_stacked",
     "!doc": "Indicates whether or not the chart is stacked.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "direction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_direction",
     "!doc": "Direction of charts category axis when there is no series collection specified. Charts can\nbe horizontal or vertical. When the chart type is column, the chart is horizontal.\nWhen the chart type is bar, the chart is vertical.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "showAreaFill": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_showAreaFill",
     "!doc": "Indicates whether or not an area is filled in a combo chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "showMarkers": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_showMarkers",
     "!doc": "Indicates whether to display markers in a combo chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "showLines": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_showLines",
     "!doc": "Indicates whether to display lines in a combo chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "categoryAxisName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_categoryAxisName",
     "!doc": "Indicates the key value used to identify a category axis in the `axes` hash. If\nnot specified, the categoryKey attribute value will be used.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "valueAxisName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_valueAxisName",
     "!doc": "Indicates the key value used to identify a the series axis when an axis not generated.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "horizontalGridlines": {
     "!type": "+charts.Gridlines",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_horizontalGridlines",
     "!doc": "Reference to the horizontalGridlines for the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "verticalGridlines": {
     "!type": "+charts.Gridlines",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_verticalGridlines",
     "!doc": "Reference to the verticalGridlines for the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_type",
     "!doc": "Type of chart when there is no series collection specified.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "categoryAxis": {
     "!type": "+charts.Axis",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#attribute_categoryAxis",
     "!doc": "Reference to the category axis used by the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    }
   },
   "CartesianSeriesConfig": {
    "!proto": "config.SeriesBaseConfig",
    "seriesTypeCollection": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_seriesTypeCollection",
     "!doc": "An array of all series of the same type used within a chart application.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "xDisplayName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_xDisplayName",
     "!doc": "Name used for for displaying data related to the x-coordinate.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "yDisplayName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_yDisplayName",
     "!doc": "Name used for for displaying data related to the y-coordinate.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "categoryDisplayName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_categoryDisplayName",
     "!doc": "Name used for for displaying category data",
     "!data": {
      "submodule": "series-base"
     }
    },
    "valueDisplayName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_valueDisplayName",
     "!doc": "Name used for for displaying value data",
     "!data": {
      "submodule": "series-base"
     }
    },
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "order": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_order",
     "!doc": "Order of this instance of this `type`.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "graphOrder": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_graphOrder",
     "!doc": "Order of the instance",
     "!data": {
      "submodule": "series-base"
     }
    },
    "xcoords": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_xcoords",
     "!doc": "x coordinates for the series.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "ycoords": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_ycoords",
     "!doc": "y coordinates for the series",
     "!data": {
      "submodule": "series-base"
     }
    },
    "xAxis": {
     "!type": "+charts.Axis",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_xAxis",
     "!doc": "Reference to the `Axis` instance used for assigning\nx-values to the graph.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "yAxis": {
     "!type": "+charts.Axis",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_yAxis",
     "!doc": "Reference to the `Axis` instance used for assigning\ny-values to the graph.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "xKey": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_xKey",
     "!doc": "Indicates which array to from the hash of value arrays in\nthe x-axis `Axis` instance.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "yKey": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_yKey",
     "!doc": "Indicates which array to from the hash of value arrays in\nthe y-axis `Axis` instance.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "xData": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_xData",
     "!doc": "Array of x values for the series.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "yData": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_yData",
     "!doc": "Array of y values for the series.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "xMarkerPlane": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_xMarkerPlane",
     "!doc": "Collection of area maps along the xAxis. Used to determine mouseover for multiple\nseries.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "yMarkerPlane": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_yMarkerPlane",
     "!doc": "Collection of area maps along the yAxis. Used to determine mouseover for multiple\nseries.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "xMarkerPlaneOffset": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_xMarkerPlaneOffset",
     "!doc": "Distance from a data coordinate to the left/right for setting a hotspot.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "yMarkerPlaneOffset": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_yMarkerPlaneOffset",
     "!doc": "Distance from a data coordinate to the top/bottom for setting a hotspot.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "direction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianSeries.html#attribute_direction",
     "!doc": "Direction of the series",
     "!data": {
      "submodule": "series-base"
     }
    }
   },
   "CategoryImplConfig": {
    "labelFormat": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryImpl.html#attribute_labelFormat",
     "!doc": "Pattern used by the `labelFunction` to format a label. The default `labelFunction` values for\n`CategoryAxis` and `CategoryAxisBase` do not accept a format object. This value can be used by\na custom method.",
     "!data": {
      "submodule": "axis-category-base"
     }
    },
    "calculateEdgeOffset": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryImpl.html#attribute_calculateEdgeOffset",
     "!doc": "Determines whether and offset is automatically calculated for the edges of the axis.",
     "!data": {
      "submodule": "axis-category-base"
     }
    },
    "labelFunction": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryImpl.html#attribute_labelFunction",
     "!doc": "Method used for formatting a label. This attribute allows for the default label formatting method to overridden.\nThe method use would need to implement the arguments below and return a `String` or `HTMLElement`.\n<dl>\n     <dt>val</dt><dd>Label to be formatted. (`String`)</dd>\n     <dt>format</dt><dd>Template for formatting label. (optional)</dd>\n</dl>",
     "!data": {
      "submodule": "axis-category-base"
     }
    }
   },
   "ChartBaseConfig": {
    "dataProvider": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_dataProvider",
     "!doc": "Data used to generate the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "seriesKeys": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_seriesKeys",
     "!doc": "A collection of keys that map to the series axes. If no keys are set,\nthey will be generated automatically depending on the data structure passed into\nthe chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "ariaLabel": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_ariaLabel",
     "!doc": "Sets the `aria-label` for the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "ariaDescription": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_ariaDescription",
     "!doc": "Sets the aria description for the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "tooltip": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_tooltip",
     "!doc": "Reference to the default tooltip available for the chart.\n<p>Contains the following properties:</p>\n <dl>\n     <dt>node</dt><dd>Reference to the actual dom node</dd>\n     <dt>showEvent</dt><dd>Event that should trigger the tooltip</dd>\n     <dt>hideEvent</dt><dd>Event that should trigger the removal of a tooltip (can be an event or an array of events)</dd>\n     <dt>styles</dt><dd>A hash of style properties that will be applied to the tooltip node</dd>\n     <dt>show</dt><dd>Indicates whether or not to show the tooltip</dd>\n     <dt>markerEventHandler</dt><dd>Displays and hides tooltip based on marker events</dd>\n     <dt>planarEventHandler</dt><dd>Displays and hides tooltip based on planar events</dd>\n     <dt>markerLabelFunction</dt><dd>Reference to the function used to format a marker event triggered tooltips text.\n     The method contains the following arguments:\n <dl>\n     <dt>categoryItem</dt><dd>An object containing the following:\n <dl>\n     <dt>axis</dt><dd>The axis to which the category is bound.</dd>\n     <dt>displayName</dt><dd>The display name set to the category (defaults to key if not provided).</dd>\n     <dt>key</dt><dd>The key of the category.</dd>\n     <dt>value</dt><dd>The value of the category.</dd>\n </dl>\n </dd>\n <dt>valueItem</dt><dd>An object containing the following:\n     <dl>\n         <dt>axis</dt><dd>The axis to which the items series is bound.</dd>\n         <dt>displayName</dt><dd>The display name of the series. (defaults to key if not provided)</dd>\n         <dt>key</dt><dd>The key for the series.</dd>\n         <dt>value</dt><dd>The value for the series item.</dd>\n     </dl>\n </dd>\n <dt>itemIndex</dt><dd>The index of the item within the series.</dd>\n <dt>series</dt><dd> The `CartesianSeries` instance of the item.</dd>\n <dt>seriesIndex</dt><dd>The index of the series in the `seriesCollection`.</dd>\n </dl>\n The method returns an `HTMLElement` which is written into the DOM using `appendChild`. If you override this method and choose\n to return an html string, you will also need to override the tooltips `setTextFunction` method to accept an html string.\n </dd>\n <dt>planarLabelFunction</dt><dd>Reference to the function used to format a planar event triggered tooltips text\n <dl>\n     <dt>categoryAxis</dt><dd> `CategoryAxis` Reference to the categoryAxis of the chart.\n     <dt>valueItems</dt><dd>Array of objects for each series that has a data point in the coordinate plane of the event. Each\n     object contains the following data:\n <dl>\n     <dt>axis</dt><dd>The value axis of the series.</dd>\n     <dt>key</dt><dd>The key for the series.</dd>\n     <dt>value</dt><dd>The value for the series item.</dd>\n     <dt>displayName</dt><dd>The display name of the series. (defaults to key if not provided)</dd>\n </dl>\n </dd>\n     <dt>index</dt><dd>The index of the item within its series.</dd>\n     <dt>seriesArray</dt><dd>Array of series instances for each value item.</dd>\n     <dt>seriesIndex</dt><dd>The index of the series in the `seriesCollection`.</dd>\n </dl>\n </dd>\n </dl>\n The method returns an `HTMLElement` which is written into the DOM using `appendChild`. If you override this method and choose\n to return an html string, you will also need to override the tooltips `setTextFunction` method to accept an html string.\n </dd>\n <dt>setTextFunction</dt><dd>Method that writes content returned from `planarLabelFunction` or `markerLabelFunction` into the\n the tooltip node. Has the following signature:\n <dl>\n     <dt>label</dt><dd>The `HTMLElement` that the content is to be added.</dd>\n     <dt>val</dt><dd>The content to be rendered into tooltip. This can be a `String` or `HTMLElement`. If an HTML string is used,\n     it will be rendered as a string.</dd>\n </dl>\n </dd>\n </dl>",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "categoryKey": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_categoryKey",
     "!doc": "The key value used for the charts category axis.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "categoryType": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_categoryType",
     "!doc": "Indicates the type of axis to use for the category axis.\n\n <dl>\n     <dt>category</dt><dd>Specifies a `CategoryAxis`.</dd>\n     <dt>time</dt><dd>Specifies a `TimeAxis</dd>\n </dl>",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "interactionType": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_interactionType",
     "!doc": "Indicates the the type of interactions that will fire events.\n\n <dl>\n     <dt>marker</dt><dd>Events will be broadcasted when the mouse interacts with individual markers.</dd>\n     <dt>planar</dt><dd>Events will be broadcasted when the mouse intersects the plane of any markers on the chart.</dd>\n     <dt>none</dt><dd>No events will be broadcasted.</dd>\n </dl>",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "axesCollection": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_axesCollection",
     "!doc": "Reference to all the axes in the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "graph": {
     "!type": "+charts.Graph",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_graph",
     "!doc": "Reference to graph instance.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "groupMarkers": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#attribute_groupMarkers",
     "!doc": "Indicates whether or not markers for a series will be grouped and rendered in a single complex shape instance.",
     "!data": {
      "submodule": "charts-base"
     }
    }
   },
   "ChartLegendConfig": {
    "!proto": "config.WidgetConfig",
    "chart": {
     "!type": "+charts.Chart",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_chart",
     "!doc": "Reference to the `Chart` instance.",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "direction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_direction",
     "!doc": "Indicates the direction in relation of the legends layout. The `direction` of the legend is determined by its\n`position` value.",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "position": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_position",
     "!doc": "Indicates the position and direction of the legend. Possible values are `left`, `top`, `right` and `bottom`.\nValues of `left` and `right` values have a `direction` of `vertical`. Values of `top` and `bottom` values have\na `direction` of `horizontal`.",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "width": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_width",
     "!doc": "The width of the legend. Depending on the implementation of the ChartLegend, this value is `readOnly`.\nBy default, the legend is included in the layout of the `Chart` that it references. Under this circumstance,\n`width` is always `readOnly`. When the legend is rendered in its own dom element, the `readOnly` status is\ndetermined by the direction of the legend. If the `position` is `left` or `right` or the `direction` is\n`vertical`, width is `readOnly`. If the position is `top` or `bottom` or the `direction` is `horizontal`,\nwidth can be explicitly set. If width is not explicitly set, the width will be determined by the width of the\nlegends parent element.",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "height": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_height",
     "!doc": "The height of the legend. Depending on the implementation of the ChartLegend, this value is `readOnly`.\nBy default, the legend is included in the layout of the `Chart` that it references. Under this circumstance,\n`height` is always `readOnly`. When the legend is rendered in its own dom element, the `readOnly` status is\ndetermined by the direction of the legend. If the `position` is `top` or `bottom` or the `direction` is\n`horizontal`, height is `readOnly`. If the position is `left` or `right` or the `direction` is `vertical`,\nheight can be explicitly set. If height is not explicitly set, the height will be determined by the width of the\nlegends parent element.",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "x": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_x",
     "!doc": "Indicates the x position of legend.",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "y": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_y",
     "!doc": "Indicates the y position of legend.",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "items": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_items",
     "!doc": "Array of items contained in the legend. Each item is an object containing the following properties:\n\n<dl>\n     <dt>node</dt><dd>Node containing text for the legend item.</dd>\n     <dt>marker</dt><dd>Shape for the legend item.</dd>\n</dl>",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "background": {
     "!type": "+graphics.Rect",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_background",
     "!doc": "Background for the legend.",
     "!data": {
      "submodule": "charts-legend"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartLegend.html#attribute_styles",
     "!doc": "Properties used to display and style the ChartLegend.  This attribute is inherited from `Renderer`.\nBelow are the default values:\n\n <dl>\n     <dt>gap</dt><dd>Distance, in pixels, between the `ChartLegend` instance and the charts content. When `ChartLegend`\n     is rendered within a `Chart` instance this value is applied.</dd>\n     <dt>hAlign</dt><dd>Defines the horizontal alignment of the `items` in a `ChartLegend` rendered in a horizontal direction.\n     This value is applied when the instances `position` is set to top or bottom. This attribute can be set to left, center\n     or right. The default value is center.</dd>\n     <dt>vAlign</dt><dd>Defines the vertical alignment of the `items` in a `ChartLegend` rendered in vertical direction. This\n     value is applied when the instances `position` is set to left or right. The attribute can be set to top, middle or\n     bottom. The default value is middle.</dd>\n     <dt>item</dt><dd>Set of style properties applied to the `items` of the `ChartLegend`.\n         <dl>\n             <dt>hSpacing</dt><dd>Horizontal distance, in pixels, between legend `items`.</dd>\n             <dt>vSpacing</dt><dd>Vertical distance, in pixels, between legend `items`.</dd>\n             <dt>label</dt><dd>Properties for the text of an `item`.\n                 <dl>\n                     <dt>color</dt><dd>Color of the text. The default values is \"#808080\".</dd>\n                     <dt>fontSize</dt><dd>Font size for the text. The default value is \"85%\".</dd>\n                 </dl>\n             </dd>\n             <dt>marker</dt><dd>Properties for the `item` markers.\n                 <dl>\n                     <dt>width</dt><dd>Specifies the width of the markers.</dd>\n                     <dt>height</dt><dd>Specifies the height of the markers.</dd>\n                 </dl>\n             </dd>\n         </dl>\n     </dd>\n     <dt>background</dt><dd>Properties for the `ChartLegend` background.\n         <dl>\n             <dt>fill</dt><dd>Properties for the background fill.\n                 <dl>\n                     <dt>color</dt><dd>Color for the fill. The default value is \"#faf9f2\".</dd>\n                 </dl>\n             </dd>\n             <dt>border</dt><dd>Properties for the background border.\n                 <dl>\n                     <dt>color</dt><dd>Color for the border. The default value is \"#dad8c9\".</dd>\n                     <dt>weight</dt><dd>Weight of the border. The default values is 1.</dd>\n                 </dl>\n             </dd>\n         </dl>\n     </dd>\n</dl>",
     "!data": {
      "submodule": "charts-legend"
     }
    }
   },
   "ColumnSeriesConfig": {
    "!proto": "config.MarkerSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ColumnSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-column"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ColumnSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing markers. This attribute is inherited from `MarkerSeries`. Below are the default values:\n <dl>\n     <dt>fill</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the fill. The default value is determined by the order of the series on the graph. The color\n             will be retrieved from the below array:<br/>\n             `[\"#66007f\", \"#a86f41\", \"#295454\", \"#996ab2\", \"#e8cdb7\", \"#90bdbd\",\"#000000\",\"#c3b8ca\", \"#968373\", \"#678585\"]`\n             </dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>border</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the border. The default value is determined by the order of the series on the graph. The color\n             will be retrieved from the below array:<br/>\n             `[\"#205096\", \"#b38206\", \"#000000\", \"#94001e\", \"#9d6fa0\", \"#e55b00\", \"#5e85c9\", \"#adab9e\", \"#6ac291\", \"#006457\"]`\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n             <dt>weight</dt><dd>Number indicating the width of the border. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>width</dt><dd>indicates the width of the marker. The default value is 12.</dd>\n     <dt>over</dt><dd>hash containing styles for markers when highlighted by a `mouseover` event. The default\n     values for each style is null. When an over style is not set, the non-over value will be used. For example,\n     the default value for `marker.over.fill.color` is equivalent to `marker.fill.color`.</dd>\n </dl>",
     "!data": {
      "submodule": "series-column"
     }
    }
   },
   "ComboSeriesConfig": {
    "!proto": "config.CartesianSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-combo"
     }
    },
    "showAreaFill": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSeries.html#attribute_showAreaFill",
     "!doc": "Indicates whether a fill is displayed.",
     "!data": {
      "submodule": "series-combo"
     }
    },
    "showLines": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSeries.html#attribute_showLines",
     "!doc": "Indicates whether lines are displayed.",
     "!data": {
      "submodule": "series-combo"
     }
    },
    "showMarkers": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSeries.html#attribute_showMarkers",
     "!doc": "Indicates whether markers are displayed.",
     "!data": {
      "submodule": "series-combo"
     }
    },
    "marker": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSeries.html#attribute_marker",
     "!doc": "Reference to the styles of the markers. These styles can also\nbe accessed through the `styles` attribute. Below are default\nvalues:\n <dl>\n     <dt>fill</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the fill. The default value is determined by the order of the series on the\n             graph. The color will be retrieved from the below array:<br/>\n             `[\"#6084d0\", \"#eeb647\", \"#6c6b5f\", \"#d6484f\", \"#ce9ed1\", \"#ff9f3b\", \"#93b7ff\", \"#e0ddd0\", \"#94ecba\", \"#309687\"]`\n             </dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>border</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the border. The default value is determined by the order of the series on the graph.\n             The color will be retrieved from the below array:<br/>\n             `[\"#205096\", \"#b38206\", \"#000000\", \"#94001e\", \"#9d6fa0\", \"#e55b00\", \"#5e85c9\", \"#adab9e\", \"#6ac291\", \"#006457\"]`\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n             <dt>weight</dt><dd>Number indicating the width of the border. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>width</dt><dd>indicates the width of the marker. The default value is 10.</dd>\n     <dt>height</dt><dd>indicates the height of the marker The default value is 10.</dd>\n     <dt>over</dt><dd>hash containing styles for markers when highlighted by a `mouseover` event. The default\n     values for each style is null. When an over style is not set, the non-over value will be used. For example,\n     the default value for `marker.over.fill.color` is equivalent to `marker.fill.color`.</dd>\n </dl>",
     "!data": {
      "submodule": "series-combo"
     }
    },
    "line": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSeries.html#attribute_line",
     "!doc": "Reference to the styles of the lines. These styles can also be accessed through the `styles` attribute.\nBelow are the default values:\n <dl>\n     <dt>color</dt><dd>The color of the line. The default value is determined by the order of the series on the graph. The color\n     will be retrieved from the following array:\n     `[\"#426ab3\", \"#d09b2c\", \"#000000\", \"#b82837\", \"#b384b5\", \"#ff7200\", \"#779de3\", \"#cbc8ba\", \"#7ed7a6\", \"#007a6c\"]`\n     <dt>weight</dt><dd>Number that indicates the width of the line. The default value is 6.</dd>\n     <dt>alpha</dt><dd>Number between 0 and 1 that indicates the opacity of the line. The default value is 1.</dd>\n     <dt>lineType</dt><dd>Indicates whether the line is solid or dashed. The default value is solid.</dd>\n     <dt>dashLength</dt><dd>When the `lineType` is dashed, indicates the length of the dash. The default value is 10.</dd>\n     <dt>gapSpace</dt><dd>When the `lineType` is dashed, indicates the distance between dashes. The default value is 10.</dd>\n     <dt>connectDiscontinuousPoints</dt><dd>Indicates whether or not to connect lines when there is a missing or null value\n     between points. The default value is true.</dd>\n     <dt>discontinuousType</dt><dd>Indicates whether the line between discontinuous points is solid or dashed. The default\n     value is solid.</dd>\n     <dt>discontinuousDashLength</dt><dd>When the `discontinuousType` is dashed, indicates the length of the dash. The default\n     value is 10.</dd>\n     <dt>discontinuousGapSpace</dt><dd>When the `discontinuousType` is dashed, indicates the distance between dashes. The default\n     value is 10.</dd>\n </dl>",
     "!data": {
      "submodule": "series-combo"
     }
    },
    "area": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSeries.html#attribute_area",
     "!doc": "Reference to the styles of the area fills. These styles can also be accessed through the `styles` attribute.\nBelow are the default values:\n\n <dl>\n     <dt>color</dt><dd>The color of the fill. The default value is determined by the order of the series on the\n     graph. The color will be retrieved from the following array:\n     `[\"#66007f\", \"#a86f41\", \"#295454\", \"#996ab2\", \"#e8cdb7\", \"#90bdbd\",\"#000000\",\"#c3b8ca\", \"#968373\", \"#678585\"]`\n     </dd>\n     <dt>alpha</dt><dd>Number between 0 and 1 that indicates the opacity of the fill. The default value is 1</dd>\n </dl>",
     "!data": {
      "submodule": "series-combo"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSeries.html#attribute_styles",
     "!doc": "Style properties for the series. Contains a key indexed hash of the following:\n <dl>\n     <dt>marker</dt><dd>Style properties for the markers in the series. Specific style attributes are listed\n     <a href=\"#attr_marker\">here</a>.</dd>\n     <dt>line</dt><dd>Style properties for the lines in the series. Specific\n     style attributes are listed <a href=\"#attr_line\">here</a>.</dd>\n     <dt>area</dt><dd>Style properties for the area fills in the series. Specific style attributes are listed\n     <a href=\"#attr_area\">here</a>.</dd>\n </dl>",
     "!data": {
      "submodule": "series-combo"
     }
    }
   },
   "ComboSplineSeriesConfig": {
    "!proto": "config.ComboSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ComboSplineSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-combospline"
     }
    }
   },
   "GraphConfig": {
    "!proto": "config.WidgetConfig",
    "chart": {
     "!type": "+charts.ChartBase",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_chart",
     "!doc": "Reference to the chart instance using the graph.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "seriesCollection": {
     "!type": "+charts.CartesianSeries",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_seriesCollection",
     "!doc": "Collection of series. When setting the `seriesCollection` the array can contain a combination of either\n`CartesianSeries` instances or object literals with properties that will define a series.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "showBackground": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_showBackground",
     "!doc": "Indicates whether the `Graph` has a background.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "seriesDictionary": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_seriesDictionary",
     "!doc": "Read-only hash lookup for all series on in the `Graph`.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "horizontalGridlines": {
     "!type": "+charts.Gridlines",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_horizontalGridlines",
     "!doc": "Reference to the horizontal `Gridlines` instance.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "verticalGridlines": {
     "!type": "+charts.Gridlines",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_verticalGridlines",
     "!doc": "Reference to the vertical `Gridlines` instance.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "background": {
     "!type": "+graphics.Graphic",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_background",
     "!doc": "Reference to graphic instance used for the background.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "gridlines": {
     "!type": "+graphics.Graphic",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_gridlines",
     "!doc": "Reference to graphic instance used for gridlines.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "graphic": {
     "!type": "+graphics.Graphic",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_graphic",
     "!doc": "Reference to graphic instance used for series.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "groupMarkers": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_groupMarkers",
     "!doc": "Indicates whether or not markers for a series will be grouped and rendered in a single complex shape instance.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#attribute_styles",
     "!doc": "Style properties used for drawing a background. Below are the default values:\n <dl>\n     <dt>background</dt><dd>An object containing the following values:\n         <dl>\n             <dt>fill</dt><dd>Defines the style properties for the fill. Contains the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the fill. The default value is #faf9f2.</dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the background fill.\n                     The default value is 1.</dd>\n                 </dl>\n             </dd>\n             <dt>border</dt><dd>Defines the style properties for the border. Contains the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the border. The default value is #dad8c9.</dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the background border.\n                     The default value is 1.</dd>\n                     <dt>weight</dt><dd>Number indicating the width of the border. The default value is 1.</dd>\n                 </dl>\n             </dd>\n         </dl>\n     </dd>\n </dl>",
     "!data": {
      "submodule": "charts-base"
     }
    }
   },
   "GridlinesConfig": {
    "!proto": "config.BaseConfig",
    "direction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Gridlines.html#attribute_direction",
     "!doc": "Indicates the direction of the gridline.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "axis": {
     "!type": "+charts.Axis",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Gridlines.html#attribute_axis",
     "!doc": "Indicate the `Axis` in which to bind\nthe gridlines.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "graph": {
     "!type": "+charts.Graph",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Gridlines.html#attribute_graph",
     "!doc": "Indicates the `Graph` in which the gridlines\nare drawn.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "count": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Gridlines.html#attribute_count",
     "!doc": "Indicates the number of gridlines to display. If no value is set, gridlines will equal the number of ticks in\nthe corresponding axis.",
     "!data": {
      "submodule": "charts-base"
     }
    }
   },
   "LineSeriesConfig": {
    "!proto": "config.CartesianSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/LineSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-line"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/LineSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing lines. This attribute is inherited from `Renderer`. Below are the\ndefault values:\n <dl>\n     <dt>color</dt><dd>The color of the line. The default value is determined by the order of the series\n     on the graph. The color will be retrieved from the following array:\n     `[\"#426ab3\", \"#d09b2c\", \"#000000\", \"#b82837\", \"#b384b5\", \"#ff7200\", \"#779de3\", \"#cbc8ba\", \"#7ed7a6\", \"#007a6c\"]`\n     <dt>weight</dt><dd>Number that indicates the width of the line. The default value is 6.</dd>\n     <dt>alpha</dt><dd>Number between 0 and 1 that indicates the opacity of the line. The default value is 1.</dd>\n     <dt>lineType</dt><dd>Indicates whether the line is solid or dashed. The default value is solid.</dd>\n     <dt>dashLength</dt><dd>When the `lineType` is dashed, indicates the length of the dash. The default\n     value is 10.</dd>\n     <dt>gapSpace</dt><dd>When the `lineType` is dashed, indicates the distance between dashes. The default\n     value is 10.</dd>\n     <dt>connectDiscontinuousPoints</dt><dd>Indicates whether or not to connect lines when there is a missing\n     or null value between points. The default value is true.</dd>\n     <dt>discontinuousType</dt><dd>Indicates whether the line between discontinuous points is solid or dashed.\n     The default value is solid.</dd>\n     <dt>discontinuousDashLength</dt><dd>When the `discontinuousType` is dashed, indicates the length of the\n     dash. The default value is 10.</dd>\n     <dt>discontinuousGapSpace</dt><dd>When the `discontinuousType` is dashed, indicates the distance between\n     dashes. The default value is 10.</dd>\n </dl>",
     "!data": {
      "submodule": "series-line"
     }
    }
   },
   "MarkerSeriesConfig": {
    "!proto": "config.CartesianSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/MarkerSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-marker"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/MarkerSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing markers. This attribute is inherited from `Renderer`. Below are the default\nvalues:\n <dl>\n     <dt>fill</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the fill. The default value is determined by the order of the series on\n             the graph. The color will be retrieved from the below array:<br/>\n             `[\"#6084d0\", \"#eeb647\", \"#6c6b5f\", \"#d6484f\", \"#ce9ed1\", \"#ff9f3b\", \"#93b7ff\", \"#e0ddd0\", \"#94ecba\", \"#309687\"]`\n             </dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>border</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the border. The default value is determined by the order of the series on\n             the graph. The color will be retrieved from the below array:<br/>\n             `[\"#205096\", \"#b38206\", \"#000000\", \"#94001e\", \"#9d6fa0\", \"#e55b00\", \"#5e85c9\", \"#adab9e\", \"#6ac291\", \"#006457\"]`\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n             <dt>weight</dt><dd>Number indicating the width of the border. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>width</dt><dd>indicates the width of the marker. The default value is 10.</dd>\n     <dt>height</dt><dd>indicates the height of the marker The default value is 10.</dd>\n     <dt>over</dt><dd>hash containing styles for markers when highlighted by a `mouseover` event. The default\n     values for each style is null. When an over style is not set, the non-over value will be used. For example,\n     the default value for `marker.over.fill.color` is equivalent to `marker.fill.color`.</dd>\n </dl>",
     "!data": {
      "submodule": "series-marker"
     }
    }
   },
   "NumericImplConfig": {
    "alwaysShowZero": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html#attribute_alwaysShowZero",
     "!doc": "Indicates whether 0 should always be displayed.",
     "!data": {
      "submodule": "axis-numeric-base"
     }
    },
    "labelFunction": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html#attribute_labelFunction",
     "!doc": "Method used for formatting a label. This attribute allows for the default label formatting method to overridden.\nThe method use would need to implement the arguments below and return a `String` or an `HTMLElement`. The default\nimplementation of the method returns a `String`. The output of this method will be rendered to the DOM using\n`appendChild`. If you override the `labelFunction` method and return an html string, you will also need to override\nthe Data `appendLabelFunction` to accept html as a `String`.\n<dl>\n     <dt>val</dt><dd>Label to be formatted. (`String`)</dd>\n     <dt>format</dt><dd>Object containing properties used to format the label. (optional)</dd>\n</dl>",
     "!data": {
      "submodule": "axis-numeric-base"
     }
    },
    "labelFormat": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html#attribute_labelFormat",
     "!doc": "Object containing properties used by the `labelFunction` to format a\nlabel.",
     "!data": {
      "submodule": "axis-numeric-base"
     }
    },
    "roundingMethod": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html#attribute_roundingMethod",
     "!doc": "Indicates how to round unit values.\n  <dl>\n      <dt>niceNumber</dt><dd>Units will be smoothed based on the number of ticks and data range.</dd>\n      <dt>auto</dt><dd>If the range is greater than 1, the units will be rounded.</dd>\n      <dt>numeric value</dt><dd>Units will be equal to the numeric value.</dd>\n      <dt>null</dt><dd>No rounding will occur.</dd>\n  </dl>",
     "!data": {
      "submodule": "axis-numeric-base"
     }
    },
    "": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html",
     "!doc": "Indicates the scaling for the chart. The default value is `linear`. For a logarithmic axis, set the value\nto `logarithmic`.",
     "!data": {
      "submodule": "axis-numeric-base"
     }
    }
   },
   "OHLCSeriesConfig": {
    "!proto": "config.RangeSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/OHLCSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-ohlc"
     }
    },
    "graphic": {
     "!type": "+graphics.Graphic",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/OHLCSeries.html#attribute_graphic",
     "!doc": "The graphic in which drawings will be rendered.",
     "!data": {
      "submodule": "series-ohlc"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/OHLCSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing markers. This attribute is inherited from `RangeSeries`. Below are the default values:\n <dl>\n     <dt>upmarker</dt><dd>Properties for a marker representing a period that closes higher than it opens.\n         <dl>\n             <dt>fill</dt><dd>A hash containing the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the fill. The default value is \"#00aa00\".</dd>\n                     </dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n                 </dl>\n             </dd>\n             <dt>border</dt><dd>A hash containing the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the border. The default value is \"#000000\".</dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n                     <dt>weight</dt><dd>Number indicating the width of the border. The default value is 0.</dd>\n                 </dl>\n             </dd>\n         </dl>\n     </dd>\n     <dt>downmarker</dt><dd>Properties for a marker representing a period that opens higher than it closes.\n         <dl>\n             <dt>fill</dt><dd>A hash containing the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the fill. The default value is \"#aa0000\".</dd>\n                     </dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n                 </dl>\n             </dd>\n             <dt>border</dt><dd>A hash containing the following values:\n                 <dl>\n                     <dt>color</dt><dd>Color of the border. The default value is \"#000000\".</dd>\n                     <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n                     <dt>weight</dt><dd>Number indicating the width of the border. The default value is 0.</dd>\n                 </dl>\n             </dd>\n         </dl>\n     </dd>\n </dl>",
     "!data": {
      "submodule": "series-ohlc"
     }
    }
   },
   "PieChartConfig": {
    "!proto": "config.ChartBaseConfig",
    "ariaDescription": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieChart.html#attribute_ariaDescription",
     "!doc": "Sets the aria description for the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "axes": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieChart.html#attribute_axes",
     "!doc": "Axes to appear in the chart.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "seriesCollection": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieChart.html#attribute_seriesCollection",
     "!doc": "Collection of series to appear on the chart. This can be an array of Series instances or object literals\nused to describe a Series instance.",
     "!data": {
      "submodule": "charts-base"
     }
    },
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieChart.html#attribute_type",
     "!doc": "Type of chart when there is no series collection specified.",
     "!data": {
      "submodule": "charts-base"
     }
    }
   },
   "PieSeriesConfig": {
    "!proto": "config.SeriesBaseConfig",
    "": {
     "!type": "+Legend",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html",
     "!doc": "The legend for the chart."
    },
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-pie"
     }
    },
    "order": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_order",
     "!doc": "Order of this instance of this `type`.",
     "!data": {
      "submodule": "series-pie"
     }
    },
    "graph": {
     "!type": "+charts.Graph",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_graph",
     "!doc": "Reference to the `Graph` in which the series is drawn into.",
     "!data": {
      "submodule": "series-pie"
     }
    },
    "categoryAxis": {
     "!type": "+charts.Axis",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_categoryAxis",
     "!doc": "Reference to the `Axis` instance used for assigning\nseries values to the graph.",
     "!data": {
      "submodule": "series-pie"
     }
    },
    "categoryKey": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_categoryKey",
     "!doc": "Indicates which array to from the hash of value arrays in\nthe category `Axis` instance.",
     "!data": {
      "submodule": "series-pie"
     }
    },
    "valueKey": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_valueKey",
     "!doc": "Indicates which array to from the hash of value arrays in\nthe value `Axis` instance.",
     "!data": {
      "submodule": "series-pie"
     }
    },
    "categoryDisplayName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_categoryDisplayName",
     "!doc": "Name used for for displaying category data",
     "!data": {
      "submodule": "series-pie"
     }
    },
    "valueDisplayName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_valueDisplayName",
     "!doc": "Name used for for displaying value data",
     "!data": {
      "submodule": "series-pie"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing markers. This attribute is inherited from `MarkerSeries`. Below are  the default\nvalues:\n <dl>\n     <dt>fill</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>colors</dt><dd>An array of colors to be used for the marker fills. The color for each marker  is\n             retrieved from the array below:<br/>\n             `[\"#66007f\", \"#a86f41\", \"#295454\", \"#996ab2\", \"#e8cdb7\", \"#90bdbd\",\"#000000\",\"#c3b8ca\", \"#968373\", \"#678585\"]`\n             </dd>\n             <dt>alphas</dt><dd>An array of alpha references (Number from 0 to 1) indicating the opacity of each marker\n             fill. The default value is [1].</dd>\n         </dl>\n     </dd>\n     <dt>border</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>An array of colors to be used for the marker borders. The color for each marker is\n             retrieved from the array below:<br/>\n             `[\"#205096\", \"#b38206\", \"#000000\", \"#94001e\", \"#9d6fa0\", \"#e55b00\", \"#5e85c9\", \"#adab9e\", \"#6ac291\", \"#006457\"]`\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n             <dt>weight</dt><dd>Number indicating the width of the border. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>over</dt><dd>hash containing styles for markers when highlighted by a `mouseover` event. The default\n     values for each style is null. When an over style is not set, the non-over value will be used. For example,\n     the default value for `marker.over.fill.color` is equivalent to `marker.fill.color`.</dd>\n </dl>",
     "!data": {
      "submodule": "series-pie"
     }
    }
   },
   "RangeSeriesConfig": {
    "!proto": "config.CartesianSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/RangeSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-range"
     }
    },
    "ohlc": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/RangeSeries.html#attribute_ohlc",
     "!doc": "Values to be used for open, high, low and close keys.",
     "!data": {
      "submodule": "series-range"
     }
    }
   },
   "RendererConfig": {
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Renderer.html#attribute_styles",
     "!doc": "Style properties for class"
    },
    "graphic": {
     "!type": "+graphics.Graphic",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Renderer.html#attribute_graphic",
     "!doc": "The graphic in which drawings will be rendered."
    }
   },
   "SeriesBaseConfig": {
    "!proto": "config.BaseConfig",
    "height": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html#attribute_height",
     "!doc": "Returns the height of the parent graph",
     "!data": {
      "submodule": "series-base"
     }
    },
    "graphic": {
     "!type": "+graphics.Graphic",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html#attribute_graphic",
     "!doc": "The graphic in which drawings will be rendered.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "chart": {
     "!type": "+charts.ChartBase",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html#attribute_chart",
     "!doc": "Reference to the `Chart` application. If no `Chart` application is present,\na reference to the `Graphic` instance that the series is drawn into will be returned.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "graph": {
     "!type": "+charts.Graph",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html#attribute_graph",
     "!doc": "Reference to the `Graph` in which the series is drawn into.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "rendered": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html#attribute_rendered",
     "!doc": "Indicates whether the Series has been through its initial set up.",
     "!data": {
      "submodule": "series-base"
     }
    },
    "visible": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html#attribute_visible",
     "!doc": "Indicates whether to show the series",
     "!data": {
      "submodule": "series-base"
     }
    },
    "groupMarkers": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html#attribute_groupMarkers",
     "!doc": "Indicates whether or not markers for a series will be grouped and rendered in a single complex shape instance.",
     "!data": {
      "submodule": "series-base"
     }
    }
   },
   "SplineSeriesConfig": {
    "!proto": "config.LineSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SplineSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-spline"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SplineSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing lines. This attribute is inherited from `Renderer`.\nBelow are the default values:\n <dl>\n     <dt>color</dt><dd>The color of the line. The default value is determined by the order of the series on\n     the graph. The color will be retrieved from the following array:\n     `[\"#426ab3\", \"#d09b2c\", \"#000000\", \"#b82837\", \"#b384b5\", \"#ff7200\", \"#779de3\", \"#cbc8ba\", \"#7ed7a6\", \"#007a6c\"]`\n     <dt>weight</dt><dd>Number that indicates the width of the line. The default value is 6.</dd>\n     <dt>alpha</dt><dd>Number between 0 and 1 that indicates the opacity of the line. The default value is 1.</dd>\n     <dt>lineType</dt><dd>Indicates whether the line is solid or dashed. The default value is solid.</dd>\n     <dt>dashLength</dt><dd>When the `lineType` is dashed, indicates the length of the dash. The default value\n     is 10.</dd>\n     <dt>gapSpace</dt><dd>When the `lineType` is dashed, indicates the distance between dashes. The default value is\n     10.</dd>\n     <dt>connectDiscontinuousPoints</dt><dd>Indicates whether or not to connect lines when there is a missing or null\n     value between points. The default value is true.</dd>\n     <dt>discontinuousType</dt><dd>Indicates whether the line between discontinuous points is solid or dashed. The\n     default value is solid.</dd>\n     <dt>discontinuousDashLength</dt><dd>When the `discontinuousType` is dashed, indicates the length of the dash.\n     The default value is 10.</dd>\n     <dt>discontinuousGapSpace</dt><dd>When the `discontinuousType` is dashed, indicates the distance between dashes.\n     The default value is 10.</dd>\n </dl>",
     "!data": {
      "submodule": "series-spline"
     }
    }
   },
   "StackedAreaSeriesConfig": {
    "!proto": "config.AreaSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedAreaSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-area-stacked"
     }
    }
   },
   "StackedAreaSplineSeriesConfig": {
    "!proto": "config.AreaSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedAreaSplineSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-areaspline-stacked"
     }
    }
   },
   "StackedBarSeriesConfig": {
    "!proto": "config.BarSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedBarSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-bar-stacked"
     }
    },
    "direction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedBarSeries.html#attribute_direction",
     "!doc": "Direction of the series",
     "!data": {
      "submodule": "series-bar-stacked"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedBarSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing markers. This attribute is inherited from `BarSeries`. Below are the default values:\n <dl>\n     <dt>fill</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the fill. The default value is determined by the order of the series on the graph. The color\n             will be retrieved from the below array:<br/>\n             `[\"#66007f\", \"#a86f41\", \"#295454\", \"#996ab2\", \"#e8cdb7\", \"#90bdbd\",\"#000000\",\"#c3b8ca\", \"#968373\", \"#678585\"]`\n             </dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>border</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the border. The default value is determined by the order of the series on the graph. The color\n             will be retrieved from the below array:<br/>\n             `[\"#205096\", \"#b38206\", \"#000000\", \"#94001e\", \"#9d6fa0\", \"#e55b00\", \"#5e85c9\", \"#adab9e\", \"#6ac291\", \"#006457\"]`\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n             <dt>weight</dt><dd>Number indicating the width of the border. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>height</dt><dd>indicates the width of the marker. The default value is 24.</dd>\n     <dt>over</dt><dd>hash containing styles for markers when highlighted by a `mouseover` event. The default\n     values for each style is null. When an over style is not set, the non-over value will be used. For example,\n     the default value for `marker.over.fill.color` is equivalent to `marker.fill.color`.</dd>\n </dl>",
     "!data": {
      "submodule": "series-bar-stacked"
     }
    }
   },
   "StackedColumnSeriesConfig": {
    "!proto": "config.ColumnSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedColumnSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-column-stacked"
     }
    },
    "styles": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedColumnSeries.html#attribute_styles",
     "!doc": "Style properties used for drawing markers. This attribute is inherited from `ColumnSeries`. Below are the default values:\n <dl>\n     <dt>fill</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the fill. The default value is determined by the order of the series on the graph. The color\n             will be retrieved from the below array:<br/>\n             `[\"#66007f\", \"#a86f41\", \"#295454\", \"#996ab2\", \"#e8cdb7\", \"#90bdbd\",\"#000000\",\"#c3b8ca\", \"#968373\", \"#678585\"]`\n             </dd>\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker fill. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>border</dt><dd>A hash containing the following values:\n         <dl>\n             <dt>color</dt><dd>Color of the border. The default value is determined by the order of the series on the graph. The color\n             will be retrieved from the below array:<br/>\n             `[\"#205096\", \"#b38206\", \"#000000\", \"#94001e\", \"#9d6fa0\", \"#e55b00\", \"#5e85c9\", \"#adab9e\", \"#6ac291\", \"#006457\"]`\n             <dt>alpha</dt><dd>Number from 0 to 1 indicating the opacity of the marker border. The default value is 1.</dd>\n             <dt>weight</dt><dd>Number indicating the width of the border. The default value is 1.</dd>\n         </dl>\n     </dd>\n     <dt>width</dt><dd>indicates the width of the marker. The default value is 24.</dd>\n     <dt>over</dt><dd>hash containing styles for markers when highlighted by a `mouseover` event. The default\n     values for each style is null. When an over style is not set, the non-over value will be used. For example,\n     the default value for `marker.over.fill.color` is equivalent to `marker.fill.color`.</dd>\n </dl>",
     "!data": {
      "submodule": "series-column-stacked"
     }
    }
   },
   "StackedComboSeriesConfig": {
    "!proto": "config.ComboSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedComboSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-combo-stacked"
     }
    },
    "showAreaFill": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedComboSeries.html#attribute_showAreaFill",
     "!doc": "Indicates whether a fill is displayed.",
     "!data": {
      "submodule": "series-combo-stacked"
     }
    }
   },
   "StackedComboSplineSeriesConfig": {
    "!proto": "config.StackedComboSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedComboSplineSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-combospline-stacked"
     }
    },
    "showAreaFill": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedComboSplineSeries.html#attribute_showAreaFill",
     "!doc": "Indicates whether a fill is displayed.",
     "!data": {
      "submodule": "series-combospline-stacked"
     }
    }
   },
   "StackedLineSeriesConfig": {
    "!proto": "config.LineSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedLineSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-line-stacked"
     }
    }
   },
   "StackedMarkerSeriesConfig": {
    "!proto": "config.MarkerSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedMarkerSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-marker-stacked"
     }
    }
   },
   "StackedSplineSeriesConfig": {
    "!proto": "config.SplineSeriesConfig",
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/StackedSplineSeries.html#attribute_type",
     "!doc": "Read-only attribute indicating the type of series.",
     "!data": {
      "submodule": "series-spline-stacked"
     }
    }
   },
   "TimeImplConfig": {
    "labelFunction": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/TimeImpl.html#attribute_labelFunction",
     "!doc": "Method used for formatting a label. This attribute allows for the default label formatting method to overridden.\nThe method use would need to implement the arguments below and return a `String` or an `HTMLElement`. The default\nimplementation of the method returns a `String`. The output of this method will be rendered to the DOM using\n`appendChild`. If you override the `labelFunction` method and return an html string, you will also need to override\nthe Axis `appendLabelFunction` to accept html as a `String`.\n<dl>\n     <dt>val</dt><dd>Label to be formatted. (`String`)</dd>\n     <dt>format</dt><dd>STRFTime string used to format the label. (optional)</dd>\n</dl>",
     "!data": {
      "submodule": "axis-time-base"
     }
    },
    "labelFormat": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/TimeImpl.html#attribute_labelFormat",
     "!doc": "Pattern used by the `labelFunction` to format a label.",
     "!data": {
      "submodule": "axis-time-base"
     }
    }
   },
   "Color": {
    "HarmonyGetOffsetConfig": {
     "h": {
      "!type": "number"
     },
     "s": {
      "!type": "number"
     },
     "l": {
      "!type": "number"
     }
    }
   },
   "ConsoleConfig": {
    "!proto": "config.WidgetConfig",
    "logEvent": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_logEvent",
     "!doc": "Name of the custom event that will communicate log messages."
    },
    "logSource": {
     "!type": "+event_custom.EventTarget",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_logSource",
     "!doc": "Object that will emit the log events.  By default the YUI instance.\nTo have a single Console capture events from all YUI instances, set\nthis to the Y.Global object."
    },
    "strings": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_strings",
     "!doc": "Collection of strings used to label elements in the Console UI.\nDefault collection contains the following name:value pairs:\n\n<ul>\n  <li>title : &quot;Log Console&quot;</li>\n  <li>pause : &quot;Pause&quot;</li>\n  <li>clear : &quot;Clear&quot;</li>\n  <li>collapse : &quot;Collapse&quot;</li>\n  <li>expand : &quot;Expand&quot;</li>\n</ul>"
    },
    "paused": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_paused",
     "!doc": "Boolean to pause the outputting of new messages to the console.\nWhen paused, messages will accumulate in the buffer."
    },
    "defaultCategory": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_defaultCategory",
     "!doc": "If a category is not specified in the Y.log(..) statement, this\ncategory will be used. Categories &quot;info&quot;,\n&quot;warn&quot;, and &quot;error&quot; are also called log level."
    },
    "defaultSource": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_defaultSource",
     "!doc": "If a source is not specified in the Y.log(..) statement, this\nsource will be used."
    },
    "entryTemplate": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_entryTemplate",
     "!doc": "Markup template used to create the DOM structure for Console entries."
    },
    "logLevel": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_logLevel",
     "!doc": "Minimum entry log level to render into the Console.  The initial\nlogLevel value for all Console instances defaults from the\nY.config.logLevel YUI configuration, or Console.LOG_LEVEL_INFO if\nthat configuration is not set.\n\nPossible values are &quot;info&quot;, &quot;warn&quot;,\n&quot;error&quot; (case insensitive), or their corresponding statics\nConsole.LOG_LEVEL_INFO and so on."
    },
    "printTimeout": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_printTimeout",
     "!doc": "Millisecond timeout between iterations of the print loop, moving\nentries from the buffer to the UI."
    },
    "printLimit": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_printLimit",
     "!doc": "Maximum number of entries printed in each iteration of the print\nloop. This is used to prevent excessive logging locking the page UI."
    },
    "consoleLimit": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_consoleLimit",
     "!doc": "Maximum number of Console entries allowed in the Console body at one\ntime.  This is used to keep acquired messages from exploding the\nDOM tree and impacting page performance."
    },
    "newestOnTop": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_newestOnTop",
     "!doc": "New entries should display at the top of the Console or the bottom?"
    },
    "scrollIntoView": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_scrollIntoView",
     "!doc": "When new entries are added to the Console UI, should they be\nscrolled into view?"
    },
    "startTime": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_startTime",
     "!doc": "The baseline time for this Console instance, used to measure elapsed\ntime from the moment the console module is <code>use</code>d to the\nmoment each new entry is logged (not rendered).\n\nThis value is reset by the instance method myConsole.reset()."
    },
    "lastTime": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_lastTime",
     "!doc": "The precise time the last entry was logged.  Used to measure elapsed\ntime between log messages."
    },
    "collapsed": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_collapsed",
     "!doc": "Controls the collapsed state of the Console"
    },
    "height": {
     "!type": "string|number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_height",
     "!doc": "String with units, or number, representing the height of the Console,\ninclusive of header and footer. If a number is provided, the default\nunit, defined by Widgets DEF_UNIT, property is used."
    },
    "width": {
     "!type": "string|number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_width",
     "!doc": "String with units, or number, representing the width of the Console.\nIf a number is provided, the default unit, defined by Widgets\nDEF_UNIT, property is used."
    },
    "useBrowserConsole": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_useBrowserConsole",
     "!doc": "Pass through to the YUI instance useBrowserConsole configuration.\nBy default this is set to false, which will disable logging to the\nbrowser console when a Console instance is created.  If the\nlogSource is not a YUI instance, this has no effect."
    },
    "style": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#attribute_style",
     "!doc": "Allows the Console to flow in the document.  Available values are\ninline, block, and separate (the default)."
    }
   },
   "DataSchema": {
    "ArrayApplyConfig": {
     "resultFields": {
      "!type": "[?]",
      "!doc": "Field identifiers to\n      locate/assign values in the response records. See above for\n      details."
     }
    },
    "JSONApplyConfig": {
     "resultListLocator": {
      "!type": "string",
      "!doc": "Path locator for the\n      location of the array of records to flatten into `response.results`"
     },
     "resultFields": {
      "!type": "[?]",
      "!doc": "Field identifiers to\n      locate/assign values in the response records. See above for\n      details."
     },
     "metaFields": {
      "!type": "[?]",
      "!doc": "Path locators to extract extra\n      non-record related information from the data object."
     }
    },
    "TextApplyConfig": {
     "resultDelimiter": {
      "!type": "string",
      "!doc": "Character or character\n      sequence that marks the end of one record and the start of\n      another."
     },
     "fieldDelimiter": {
      "!type": "string",
      "!doc": "Character or character\n      sequence that marks the end of a field and the start of\n      another within the same record."
     },
     "resultFields": {
      "!type": "[?]",
      "!doc": "Field identifiers to\n      assign values in the response records. See above for details."
     }
    },
    "XMLApplyConfig": {
     "resultListLocator": {
      "!type": "string",
      "!doc": "XPath locator for the\n      XML nodes that contain the data to flatten into `response.results`"
     },
     "resultFields": {
      "!type": "[?]",
      "!doc": "Field identifiers to\n      locate/assign values in the response records. See above for\n      details."
     },
     "metaFields": {
      "!type": "[?]",
      "!doc": "XPath locators to extract extra\n      non-record related information from the XML data"
     }
    }
   },
   "DataSource": {
    "FunctionConfig": {
     "!proto": "config.DataSource.LocalConfig",
     "source": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Function.html#attribute_source",
      "!doc": "Stores the function that will serve the response data.",
      "!data": {
       "submodule": "datasource-function"
      }
     }
    },
    "GetConfig": {
     "!proto": "config.DataSource.LocalConfig",
     "get": {
      "!type": "+Y.Get",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Get.html#attribute_get",
      "!doc": "Pointer to Get Utility.",
      "!data": {
       "submodule": "datasource-get"
      }
     },
     "asyncMode": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Get.html#attribute_asyncMode",
      "!doc": "Defines request/response management in the following manner:\n<dl>\n    <!--<dt>queueRequests</dt>\n    <dd>If a request is already in progress, wait until response is\n    returned before sending the next request.</dd>\n    <dt>cancelStaleRequests</dt>\n    <dd>If a request is already in progress, cancel it before\n    sending the next request.</dd>-->\n    <dt>ignoreStaleResponses</dt>\n    <dd>Send all requests, but handle only the response for the most\n    recently sent request.</dd>\n    <dt>allowAll</dt>\n    <dd>Send all requests and handle all responses.</dd>\n</dl>",
      "!data": {
       "submodule": "datasource-get"
      }
     },
     "scriptCallbackParam": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Get.html#attribute_scriptCallbackParam",
      "!doc": "Callback string parameter name sent to the remote script. By default,\nrequests are sent to\n&#60;URI&#62;?&#60;scriptCallbackParam&#62;=callbackFunction",
      "!data": {
       "submodule": "datasource-get"
      }
     },
     "generateRequestCallback": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Get.html#attribute_generateRequestCallback",
      "!doc": "Accepts the DataSource instance and a callback ID, and returns a callback\nparam/value string that gets appended to the script URI. Implementers\ncan customize this string to match their servers query syntax.",
      "!data": {
       "submodule": "datasource-get"
      }
     }
    },
    "IOConfig": {
     "!proto": "config.DataSource.LocalConfig",
     "io": {
      "!type": "+Y.io",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.IO.html#attribute_io",
      "!doc": "Pointer to IO Utility.",
      "!data": {
       "submodule": "datasource-io"
      }
     },
     "ioConfig": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.IO.html#attribute_ioConfig",
      "!doc": "Default IO Config.",
      "!data": {
       "submodule": "datasource-io"
      }
     }
    },
    "LocalConfig": {
     "!proto": "config.BaseConfig",
     "source": {
      "!type": "+MIXED",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Local.html#attribute_source",
      "!doc": "Pointer to live data.",
      "!data": {
       "submodule": "datasource-local"
      }
     }
    }
   },
   "DataTable": {
    "BaseConfig": {
     "!proto": "config.WidgetConfig",
     "view": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#attribute_view",
      "!doc": "The View class used to render the `<table>` into the Widgets\n`contentBox`.  This View can handle the entire table rendering itself\nor delegate to other Views.\n\nIt is not strictly necessary that the class function assigned here be\na View subclass.  It must however have a `render()` method.\n\nWhen the DataTable is rendered, an instance of this View will be\ncreated and its `render()` method called.  The View instance will be\nassigned to the DataTable instances `view` property.",
      "!data": {
       "submodule": "datatable-base"
      }
     },
     "headerView": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#attribute_headerView",
      "!doc": "If the View class assigned to the DataTables `view` attribute supports\nit, this class will be used for rendering the contents of the\n`<thead>`&mdash;the column headers for the table.\n\nSimilar to `view`, the instance of this View will be assigned to the\nDataTable instances `head` property.\n\nIt is not strictly necessary that the class function assigned here be\na View subclass.  It must however have a `render()` method.",
      "!data": {
       "submodule": "datatable-base"
      }
     },
     "footerView": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#attribute_footerView",
      "!doc": "If the View class assigned to the DataTables `view` attribute supports\nit, this class will be used for rendering the contents of the `<tfoot>`.\n\nSimilar to `view`, the instance of this View will be assigned to the\nDataTable instances `foot` property.\n\nIt is not strictly necessary that the class function assigned here be\na View subclass.  It must however have a `render()` method.",
      "!data": {
       "submodule": "datatable-base"
      }
     },
     "bodyView": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#attribute_bodyView",
      "!doc": "If the View class assigned to the DataTables `view` attribute supports\nit, this class will be used for rendering the contents of the `<tbody>`\nincluding all data rows.\n\nSimilar to `view`, the instance of this View will be assigned to the\nDataTable instances `body` property.\n\nIt is not strictly necessary that the class function assigned here be\na View subclass.  It must however have a `render()` method.",
      "!data": {
       "submodule": "datatable-base"
      }
     }
    },
    "HighlightConfig": {
     "highlightRows": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Highlight.html#attribute_highlightRows",
      "!doc": "Setting this to true will create a delegate on the DataTable adding the\ndefault classname to the row when the mouse is over the row.",
      "!data": {
       "submodule": "datatable-highlight"
      }
     },
     "highlightCols": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Highlight.html#attribute_highlightCols",
      "!doc": "Setting this to true will create a delegate on the DataTable adding the\ndefault classname to the column when the mouse is over the column.",
      "!data": {
       "submodule": "datatable-highlight"
      }
     },
     "highlightCells": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Highlight.html#attribute_highlightCells",
      "!doc": "Setting this to true will create a delegate on the DataTable adding the\ndefault classname to the cell when the mouse is over it.",
      "!data": {
       "submodule": "datatable-highlight"
      }
     }
    },
    "Paginator": {
     "ViewConfig": {
      "!proto": "config.ViewConfig",
      "pageSizes": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.View.html#attribute_pageSizes",
       "!doc": "Array of values used to populate the drop down for items per page",
       "!data": {
        "submodule": "datatable-paginator"
       }
      },
      "model": {
       "!type": "+app.Model",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.View.html#attribute_model",
       "!doc": "Model used for this view",
       "!data": {
        "submodule": "datatable-paginator"
       }
      }
     }
    },
    "PaginatorConfig": {
     "paginatorModel": {
      "!type": "+app.Model|+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#attribute_paginatorModel",
      "!doc": "A model instance or a configuration object for the Model.",
      "!data": {
       "submodule": "datatable-paginator"
      }
     },
     "paginatorModelType": {
      "!type": "+app.Model|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#attribute_paginatorModelType",
      "!doc": "A pointer to a Model object to be instantiated, or a String off of the\n`Y` namespace.\n\nThis is only used if the `paginatorModel` is a configuration object or\nis null.",
      "!data": {
       "submodule": "datatable-paginator"
      }
     },
     "paginatorView": {
      "!type": "+app.View|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#attribute_paginatorView",
      "!doc": "A pointer to a `Y.View` object to be instantiated. A new view will be\ncreated for each location provided. Each view created will be given the\nsame model instance.",
      "!data": {
       "submodule": "datatable-paginator"
      }
     },
     "pageSizes": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#attribute_pageSizes",
      "!doc": "Array of values used to populate the values in the Paginator UI allowing\nthe end user to select the number of items to display per page.",
      "!data": {
       "submodule": "datatable-paginator"
      }
     },
     "rowsPerPage": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#attribute_rowsPerPage",
      "!doc": "Number of rows to display per page. As the UI changes the number of pages\nto display, this will update to reflect the value selected in the UI",
      "!data": {
       "submodule": "datatable-paginator"
      }
     },
     "paginatorLocation": {
      "!type": "string|[?]|+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#attribute_paginatorLocation",
      "!doc": "String of `footer` or `header`, a Y.Node, or an Array or any combination\nof those values.",
      "!data": {
       "submodule": "datatable-paginator"
      }
     }
    },
    "TableViewConfig": {
     "!proto": "config.ViewConfig",
     "summary": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_summary",
      "!doc": "Content for the `<table summary=\"ATTRIBUTE VALUE HERE\">`.  Values\nassigned to this attribute will be HTML escaped for security.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "caption": {
      "!type": "+HTML",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_caption",
      "!doc": "HTML content of an optional `<caption>` element to appear above the\ntable.  Leave this config unset or set to a falsy value to remove the\ncaption.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "columns": {
      "!type": "[+Object]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_columns",
      "!doc": "Columns to include in the rendered table.\n\nThis attribute takes an array of objects. Each object is considered a\ndata column or header cell to be rendered.  How the objects are\ntranslated into markup is delegated to the `headerView`, `bodyView`,\nand `footerView`.\n\nThe raw value is passed to the `headerView` and `footerView`.  The\n`bodyView` receives the instances `displayColumns` array, which is\nparsed from the columns array.  If there are no nested columns (columns\nconfigured with a `children` array), the `displayColumns` is the same\nas the raw value.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "width": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_width",
      "!doc": "Width of the table including borders.  This value requires units, so\n`200` is invalid, but `200px` is valid.  Setting the empty string\n(the default) will allow the browser to set the table width.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "headerView": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_headerView",
      "!doc": "An instance of this class is used to render the contents of the\n`<thead>`&mdash;the column headers for the table.\n\nThe instance of this View will be assigned to the instances `head`\nproperty.\n\nIt is not strictly necessary that the class function assigned here be\na View subclass.  It must however have a `render()` method.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "headerConfig": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_headerConfig",
      "!doc": "Configuration overrides used when instantiating the `headerView`\ninstance.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "footerView": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_footerView",
      "!doc": "An instance of this class is used to render the contents of the\n`<tfoot>` (if appropriate).\n\nThe instance of this View will be assigned to the instances `foot`\nproperty.\n\nIt is not strictly necessary that the class function assigned here be\na View subclass.  It must however have a `render()` method.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "footerConfig": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_footerConfig",
      "!doc": "Configuration overrides used when instantiating the `footerView`\ninstance.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "bodyView": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_bodyView",
      "!doc": "An instance of this class is used to render the contents of the tables\n`<tbody>`&mdash;the data cells in the table.\n\nThe instance of this View will be assigned to the instances `body`\nproperty.\n\nIt is not strictly necessary that the class function assigned here be\na View subclass.  It must however have a `render()` method.",
      "!data": {
       "submodule": "datatable-table"
      }
     },
     "bodyConfig": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#attribute_bodyConfig",
      "!doc": "Configuration overrides used when instantiating the `bodyView`\ninstance.",
      "!data": {
       "submodule": "datatable-table"
      }
     }
    }
   },
   "DataTableConfig": {
    "!proto": "config.DataTable.BaseConfig",
    "columns": {
     "!type": "[+Object]|[string]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_columns",
     "!doc": "Columns to include in the rendered table.\n\nIf omitted, the attributes on the configured `recordType` or the first item\nin the `data` collection will be used as a source.\n\nThis attribute takes an array of strings or objects (mixing the two is\nfine).  Each string or object is considered a column to be rendered.\nStrings are converted to objects, so `columns: [first, last]` becomes\n`columns: [{ key: first }, { key: last }]`.\n\nDataTable.Core only concerns itself with a few properties of columns.\nThese properties are:\n\n* `key` - Used to identify the record field/attribute containing content for\n  this column.  Also used to create a default Model if no `recordType` or\n  `data` are provided during construction.  If `name` is not specified, this\n  is assigned to the `_id` property (with added incrementer if the key is\n  used by multiple columns).\n* `children` - Traversed to initialize nested column objects\n* `name` - Used in place of, or in addition to, the `key`.  Useful for\n  columns that arent bound to a field/attribute in the record data.  This\n  is assigned to the `_id` property.\n* `id` - For backward compatibility.  Implementers can specify the id of\n  the header cell.  This should be avoided, if possible, to avoid the\n  potential for creating DOM elements with duplicate IDs.\n* `field` - For backward compatibility.  Implementers should use `name`.\n* `_id` - Assigned unique-within-this-instance id for a column.  By order\n  of preference, assumes the value of `name`, `key`, `id`, or `_yuid`.\n  This is used by the rendering views as well as feature module\n  as a means to identify a specific column without ambiguity (such as\n  multiple columns using the same `key`.\n* `_yuid` - Guid stamp assigned to the column object.\n* `_parent` - Assigned to all child columns, referencing their parent\n  column.",
     "!data": {
      "submodule": "datatable-core"
     }
    },
    "recordType": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_recordType",
     "!doc": "Model subclass to use as the `model` for the ModelList stored in the `data`\nattribute.\n\nIf not provided, it will try really hard to figure out what to use.  The\nfollowing attempts will be made to set a default value:\n\n1. If the `data` attribute is set with a ModelList instance and its `model`\n   property is set, that will be used.\n2. If the `data` attribute is set with a ModelList instance, and its\n   `model` property is unset, but it is populated, the `ATTRS` of the\n   `constructor of the first item will be used.\n3. If the `data` attribute is set with a non-empty array, a Model subclass\n   will be generated using the keys of the first item as its `ATTRS` (see\n   the `_createRecordClass` method).\n4. If the `columns` attribute is set, a Model subclass will be generated\n   using the columns defined with a `key`. This is least desirable because\n   columns can be duplicated or nested in a way thats not parsable.\n5. If neither `data` nor `columns` is set or populated, a change event\n   subscriber will listen for the first to be changed and try all over\n   again.",
     "!data": {
      "submodule": "datatable-core"
     }
    },
    "data": {
     "!type": "+app.ModelList|[+Object]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_data",
     "!doc": "The collection of data records to display.  This attribute is a pass\nthrough to a `data` property, which is a ModelList instance.\n\nIf this attribute is passed a ModelList or subclass, it will be assigned to\nthe property directly.  If an array of objects is passed, a new ModelList\nwill be created using the configured `recordType` as its `model` property\nand seeded with the array.\n\nRetrieving this attribute will return the ModelList stored in the `data`\nproperty.",
     "!data": {
      "submodule": "datatable-core"
     }
    },
    "summary": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_summary",
     "!doc": "Content for the `<table summary=\"ATTRIBUTE VALUE HERE\">`.  Values assigned\nto this attribute will be HTML escaped for security.",
     "!data": {
      "submodule": "datatable-core"
     }
    },
    "caption": {
     "!type": "+HTML",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_caption",
     "!doc": "HTML content of an optional `<caption>` element to appear above the table.\nLeave this config unset or set to a falsy value to remove the caption.",
     "!data": {
      "submodule": "datatable-core"
     }
    },
    "recordset": {
     "!type": "[+Object]|+recordset.Recordset",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_recordset",
     "!doc": "Deprecated as of 3.5.0. Passes through to the `data` attribute.\n\nWARNING: `get(recordset)` will NOT return a Recordset instance as of\n3.5.0.  This is a break in backward compatibility.",
     "!data": {
      "submodule": "datatable-core"
     }
    },
    "columnset": {
     "!type": "[+Object]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_columnset",
     "!doc": "Deprecated as of 3.5.0. Passes through to the `columns` attribute.\n\nWARNING: `get(columnset)` will NOT return a Columnset instance as of\n3.5.0.  This is a break in backward compatibility.",
     "!data": {
      "submodule": "datatable-core"
     }
    },
    "booleanLabels": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_booleanLabels",
     "!doc": "Determines the texts to be shown to represent Boolean values when the\n[boolean](DataTable.BodyView.Formatters.html#method_boolean) formatter\nis used on any column.\n\nIt works like the column-specific\n[booleanLabels](DataTable.Column.html#property_booleanLabels) but\nfor all columns using the\n[boolean](DataTable.BodyView.Formatters.html#method_boolean) formatter at once.\nThe values are often retrieved from a resource of localized texts."
    },
    "currencyFormat": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_currencyFormat",
     "!doc": "Format specification for columns using the\n[currency](DataTable.BodyView.Formatters.html#method_currency) formatter.\nIt contains an object as described in\n[Number.format](Number.html#method_format).\n\nIt is similar to\n[currencyFormat](DataTable.Column.html#property_currencyFormat)\nbut it applies to any column using the\n[currency](DataTable.BodyView.Formatters.html#method_currency) formatter.\n The values are often retrieved from a resource of localized configuration."
    },
    "dateFormat": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_dateFormat",
     "!doc": "Format specification for columns using the\n[date](DataTable.BodyView.Formatters.html#method_date) formatter.\nIt contains an object as described in\n[Date.format](Date.html#method_format).\n\nIt is similar to\n[dateFormat](DataTable.Column.html#property_dateFormat)\nbut it applies to any column using the\n[date](DataTable.BodyView.Formatters.html#method_date) formatter.\n The values are often retrieved from a resource of localized configuration."
    },
    "numberFormat": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_numberFormat",
     "!doc": "Format specification for columns using the\n[number](DataTable.BodyView.Formatters.html#method_number) formatter.\nIt contains an object as described in\n[Number.format](Number.html#method_format).\n\nIt is similar to\n[numberFormat](DataTable.Column.html#property_numberFormat)\nbut it applies to any column using the\n[number](DataTable.BodyView.Formatters.html#method_number) formatter.\n The values are often retrieved from a resource of localized configuration."
    },
    "focusedCell": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_focusedCell",
     "!doc": "Cell thats currently either focused or\nfocusable when the DataTable gets the focus.",
     "!data": {
      "submodule": "datatable-keynav"
     }
    },
    "keyIntoHeaders": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_keyIntoHeaders",
     "!doc": "Determines whether it is possible to navigate into the header area.\nThe examples referenced in the document show both behaviors so it seems\nit is optional.",
     "!data": {
      "submodule": "datatable-keynav"
     }
    },
    "showMessages": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_showMessages",
     "!doc": "Enables the display of messages in the table.  Setting this to false will\nprevent the message Node from being created and `showMessage` from doing\nanything.",
     "!data": {
      "submodule": "datatable-message"
     }
    },
    "autoSync": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_autoSync",
     "!doc": "Controls whether `addRow`, `removeRow`, and `modifyRow` should trigger the\nunderlying Models sync layer by default.\n\nWhen `true`, it is unnecessary to pass the \"sync\" configuration property to\nthose methods to trigger per-operation sync.",
     "!data": {
      "submodule": "datatable-mutable"
     }
    },
    "scrollable": {
     "!type": "string|bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_scrollable",
     "!doc": "Activates or deactivates scrolling in the table.  Acceptable values are:\n\n * `false` - (default) Scrolling is disabled.\n * `true` or xy - If `height` is set, vertical scrolling will be\n   activated, if `width` is set, horizontal scrolling will be activated.\n * x - Activate horizontal scrolling only. Requires the `width` attribute\n   is also set.\n * y - Activate vertical scrolling only. Requires the `height` attribute\n   is also set."
    },
    "sortable": {
     "!type": "string|[string]|bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_sortable",
     "!doc": "Controls which column headers can trigger sorting by user clicks.\n\nAcceptable values are:\n\n * \"auto\" - (default) looks for `sortable: true` in the column configurations\n * `true` - all columns are enabled\n * `false - no UI sortable is enabled\n * {String[]} - array of key names to give sortable headers",
     "!data": {
      "submodule": "datatable-sort"
     }
    },
    "sortBy": {
     "!type": "string|[string]|+Object|[+Object]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_sortBy",
     "!doc": "The current sort configuration to maintain in the data.\n\nAccepts column `key` strings or objects with a single property, the column\n`key`, with a value of 1, -1, \"asc\", or \"desc\".  E.g. `{ username: asc\n}`.  String values are assumed to be ascending.\n\nExample values would be:\n\n * `\"username\"` - sort by the datas `username` field or the `key`\n   associated to a column with that `name`.\n * `{ username: \"desc\" }` - sort by `username` in descending order.\n   Alternately, use values \"asc\", 1 (same as \"asc\"), or -1 (same as \"desc\").\n * `[\"lastName\", \"firstName\"]` - ascending sort by `lastName`, but for\n   records with the same `lastName`, ascending subsort by `firstName`.\n   Array can have as many items as you want.\n * `[{ lastName: -1 }, \"firstName\"]` - descending sort by `lastName`,\n   ascending subsort by `firstName`. Mixed types are ok.",
     "!data": {
      "submodule": "datatable-sort"
     }
    },
    "strings": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#attribute_strings",
     "!doc": "Strings containing language for sorting tooltips.",
     "!data": {
      "submodule": "datatable-sort"
     }
    }
   },
   "DD": {
    "DDMConfig": {
     "!proto": "config.BaseConfig",
     "dragCursor": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#attribute_dragCursor",
      "!doc": "The cursor to apply when dragging, if shimmed the shim will get the cursor.",
      "!data": {
       "submodule": "dd-ddm-base"
      }
     },
     "clickPixelThresh": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#attribute_clickPixelThresh",
      "!doc": "The number of pixels to move to start a drag operation, default is 3.",
      "!data": {
       "submodule": "dd-ddm-base"
      }
     },
     "clickTimeThresh": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#attribute_clickTimeThresh",
      "!doc": "The number of milliseconds a mousedown has to pass to start a drag operation, default is 1000.",
      "!data": {
       "submodule": "dd-ddm-base"
      }
     },
     "throttleTime": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#attribute_throttleTime",
      "!doc": "The number of milliseconds to throttle the mousemove event. Default: 150",
      "!data": {
       "submodule": "dd-ddm-base"
      }
     },
     "dragMode": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#attribute_dragMode",
      "!doc": "This attribute only works if the dd-drop module is active. It will set the dragMode (point, intersect, strict) of all future Drag instances.",
      "!data": {
       "submodule": "dd-ddm-base"
      }
     }
    },
    "DelegateConfig": {
     "!proto": "config.BaseConfig",
     "container": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_container",
      "!doc": "A selector query to get the container to listen for mousedown events on. All \"nodes\" should be a child of this container.",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "nodes": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_nodes",
      "!doc": "A selector query to get the children of the \"container\" to make draggable elements from.",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "invalid": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_invalid",
      "!doc": "A selector query to test a node to see if its an invalid item.",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "lastNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_lastNode",
      "!doc": "Y.Node instance of the last item dragged.",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "currentNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_currentNode",
      "!doc": "Y.Node instance of the dd node.",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "dragNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_dragNode",
      "!doc": "Y.Node instance of the dd dragNode.",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "over": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_over",
      "!doc": "Is the mouse currently over the container",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "target": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_target",
      "!doc": "Should the items also be a drop target.",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "dragConfig": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_dragConfig",
      "!doc": "The default config to be used when creating the DD instance.",
      "!data": {
       "submodule": "dd-delegate"
      }
     },
     "handles": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#attribute_handles",
      "!doc": "The handles config option added to the temp DD instance.",
      "!data": {
       "submodule": "dd-delegate"
      }
     }
    },
    "DragConfig": {
     "!proto": "config.BaseConfig",
     "node": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_node",
      "!doc": "Y.Node instance to use as the element to initiate a drag operation",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "dragNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_dragNode",
      "!doc": "Y.Node instance to use as the draggable element, defaults to node",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "offsetNode": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_offsetNode",
      "!doc": "Offset the drag element by the difference in cursor position: default true",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "startCentered": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_startCentered",
      "!doc": "Center the dragNode to the mouse position on drag:start: default false",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "clickPixelThresh": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_clickPixelThresh",
      "!doc": "The number of pixels to move to start a drag operation, default is 3.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "clickTimeThresh": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_clickTimeThresh",
      "!doc": "The number of milliseconds a mousedown has to pass to start a drag operation, default is 1000.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "lock": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_lock",
      "!doc": "Set to lock this drag element so that it cant be dragged: default false.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "data": {
      "!type": "+Mixed",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_data",
      "!doc": "A payload holder to store arbitrary data about this drag object, can be used to store any value.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "move": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_move",
      "!doc": "If this is false, the drag element will not move with the cursor: default true. Can be used to \"resize\" the element.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "useShim": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_useShim",
      "!doc": "Use the protective shim on all drag operations: default true. Only works with dd-ddm, not dd-ddm-base.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "activeHandle": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_activeHandle",
      "!doc": "Config option is set by Drag to inform you of which handle fired the drag event (in the case that there are several handles): default false.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "primaryButtonOnly": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_primaryButtonOnly",
      "!doc": "By default a drag operation will only begin if the mousedown occurred with the primary mouse button.\nSetting this to false will allow for all mousedown events to trigger a drag.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "dragging": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_dragging",
      "!doc": "This attribute is not meant to be used by the implementor, it is meant to be used as an Event tracker so you can listen for it to change.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "target": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_target",
      "!doc": "This attribute only works if the dd-drop module has been loaded. It will make this node a drop target as well as draggable.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "dragMode": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_dragMode",
      "!doc": "This attribute only works if the dd-drop module is active. It will set the dragMode (point, intersect, strict) of this Drag instance.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "groups": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_groups",
      "!doc": "Array of groups to add this drag into.",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "handles": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_handles",
      "!doc": "Array of valid handles to add. Adding something here will set all handles, even if previously added with addHandle",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "bubbles": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_bubbles",
      "!doc": "Controls the default bubble parent for this Drag instance. Default: Y.DD.DDM. Set to false to disable bubbling. Use bubbleTargets in config",
      "!data": {
       "submodule": "dd-drag"
      }
     },
     "haltDown": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#attribute_haltDown",
      "!doc": "Should the mousedown event be halted. Default: true",
      "!data": {
       "submodule": "dd-drag"
      }
     }
    },
    "DropConfig": {
     "!proto": "config.BaseConfig",
     "node": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#attribute_node",
      "!doc": "Y.Node instance to use as the element to make a Drop Target",
      "!data": {
       "submodule": "dd-drop"
      }
     },
     "groups": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#attribute_groups",
      "!doc": "Array of groups to add this drop into.",
      "!data": {
       "submodule": "dd-drop"
      }
     },
     "padding": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#attribute_padding",
      "!doc": "CSS style padding to make the Drop Target bigger than the node.",
      "!data": {
       "submodule": "dd-drop"
      }
     },
     "lock": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#attribute_lock",
      "!doc": "Set to lock this drop element.",
      "!data": {
       "submodule": "dd-drop"
      }
     },
     "bubbles": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#attribute_bubbles",
      "!doc": "Controls the default bubble parent for this Drop instance. Default: Y.DD.DDM. Set to false to disable bubbling.\nUse bubbleTargets in config.",
      "!data": {
       "submodule": "dd-drop"
      }
     },
     "useShim": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#attribute_useShim",
      "!doc": "Use the Drop shim. Default: true",
      "!data": {
       "submodule": "dd-drop"
      }
     }
    },
    "ScrollConfig": {
     "!proto": "config.BaseConfig",
     "buffer": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#attribute_buffer",
      "!doc": "The number of pixels from the edge of the screen to turn on scrolling. Default: 30",
      "!data": {
       "submodule": "dd-scroll"
      }
     },
     "scrollDelay": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#attribute_scrollDelay",
      "!doc": "The number of milliseconds delay to pass to the auto scroller. Default: 235",
      "!data": {
       "submodule": "dd-scroll"
      }
     },
     "host": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#attribute_host",
      "!doc": "The host we are plugged into.",
      "!data": {
       "submodule": "dd-scroll"
      }
     },
     "windowScroll": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#attribute_windowScroll",
      "!doc": "Turn on window scroll support, default: false",
      "!data": {
       "submodule": "dd-scroll"
      }
     },
     "vertical": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#attribute_vertical",
      "!doc": "Allow vertical scrolling, default: true.",
      "!data": {
       "submodule": "dd-scroll"
      }
     },
     "horizontal": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#attribute_horizontal",
      "!doc": "Allow horizontal scrolling, default: true.",
      "!data": {
       "submodule": "dd-scroll"
      }
     }
    }
   },
   "DialConfig": {
    "!proto": "config.WidgetConfig",
    "min": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_min",
     "!doc": "minimum value allowed"
    },
    "max": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_max",
     "!doc": "maximum value allowed"
    },
    "diameter": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_diameter",
     "!doc": "diameter of the circular background object.\nOther objects scale accordingly.\nSet this only before rendering."
    },
    "handleDiameter": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_handleDiameter",
     "!doc": "diameter of the handle object which users drag to change the value.\nDial sets the pixel dimension of the handle equal to handleDiameter * diameter.\nSet this only before rendering."
    },
    "markerDiameter": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_markerDiameter",
     "!doc": "diameter of the marker object which follows the angle of the handle during value changes.\nDial sets the pixel dimension of the marker equal to markerDiameter * diameter.\nSet this only before rendering."
    },
    "centerButtonDiameter": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_centerButtonDiameter",
     "!doc": "diameter of the center button object.\nDial sets the pixel dimension of the centerButton equal to centerButtonDiameter * diameter.\nSet this only before rendering."
    },
    "value": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_value",
     "!doc": "initial value of the Dial"
    },
    "minorStep": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_minorStep",
     "!doc": "amount to increment/decrement the dial value\nwhen the arrow up/down/left/right keys are pressed"
    },
    "majorStep": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_majorStep",
     "!doc": "amount to increment/decrement the dial value\nwhen the page up/down keys are pressed"
    },
    "stepsPerRevolution": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_stepsPerRevolution",
     "!doc": "number of value increments in one 360 degree revolution\nof the handle around the dial"
    },
    "decimalPlaces": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_decimalPlaces",
     "!doc": "number of decimal places of accuracy in the value"
    },
    "strings": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_strings",
     "!doc": "visible strings for the dial UI. This attribute is\ndefined by the base Widget class but has an empty value. The\nDial is simply providing a default value for the attribute.\nGets localized strings in the current language"
    },
    "handleDistance": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#attribute_handleDistance",
     "!doc": "distance from the center of the dial to the\ncenter of the marker and handle, when at rest.\nThe value is a percent of the radius of the dial."
    }
   },
   "ContentEditableConfig": {
    "!proto": "config.Y.Plugin.BaseConfig",
    "dir": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_dir",
     "!doc": "The default text direction for this ContentEditable element. Default: ltr",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "container": {
     "!type": "string|+HTMLElement|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_container",
     "!doc": "The container to set contentEditable=true or to create on render.",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "content": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_content",
     "!doc": "The string to inject as Editor content. Default <br>",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "defaultblock": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_defaultblock",
     "!doc": "The default tag to use for block level items, defaults to: p",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "extracss": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_extracss",
     "!doc": "A string of CSS to add to the Head of the Editor",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "id": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_id",
     "!doc": "Set the id of the new Node. (optional)",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "lang": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_lang",
     "!doc": "The default language. Default: en-US",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "linkedcss": {
     "!type": "string|[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_linkedcss",
     "!doc": "An array of urls to external linked style sheets",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "node": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_node",
     "!doc": "The Node instance of the container.",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "use": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#attribute_use",
     "!doc": "Array of modules to include in the scoped YUI instance at render time. Default: [node-base, editor-selection, stylesheet]",
     "!data": {
      "submodule": "content-editable"
     }
    }
   },
   "EditorBaseConfig": {
    "!proto": "config.BaseConfig",
    "content": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#attribute_content",
     "!doc": "The content to load into the Editor Frame",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "dir": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#attribute_dir",
     "!doc": "The value of the dir attribute on the HTML element of the frame. Default: ltr",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "linkedcss": {
     "!type": "string|[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#attribute_linkedcss",
     "!doc": "An array of urls to external linked style sheets",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "extracss": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#attribute_extracss",
     "!doc": "A string of CSS to add to the Head of the Editor",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "defaultblock": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#attribute_defaultblock",
     "!doc": "The default tag to use for block level items, defaults to: p",
     "!data": {
      "submodule": "editor-base"
     }
    }
   },
   "FrameConfig": {
    "!proto": "config.BaseConfig",
    "title": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_title",
     "!doc": "The title to give the blank page.",
     "!data": {
      "submodule": "frame"
     }
    },
    "dir": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_dir",
     "!doc": "The default text direction for this new frame. Default: ltr",
     "!data": {
      "submodule": "frame"
     }
    },
    "lang": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_lang",
     "!doc": "The default language. Default: en-US",
     "!data": {
      "submodule": "frame"
     }
    },
    "src": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_src",
     "!doc": "The src of the iframe/window. Defaults to javascript:;",
     "!data": {
      "submodule": "frame"
     }
    },
    "designMode": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_designMode",
     "!doc": "Should designMode be turned on after creation.",
     "!data": {
      "submodule": "frame"
     }
    },
    "content": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_content",
     "!doc": "The string to inject into the body of the new frame/window.",
     "!data": {
      "submodule": "frame"
     }
    },
    "basehref": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_basehref",
     "!doc": "The base href to use in the iframe.",
     "!data": {
      "submodule": "frame"
     }
    },
    "use": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_use",
     "!doc": "Array of modules to include in the scoped YUI instance at render time. Default: [none, selector-css2]",
     "!data": {
      "submodule": "frame"
     }
    },
    "container": {
     "!type": "string|+HTMLElement|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_container",
     "!doc": "The container to append the iFrame to on render.",
     "!data": {
      "submodule": "frame"
     }
    },
    "node": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_node",
     "!doc": "The Node instance of the iframe.",
     "!data": {
      "submodule": "frame"
     }
    },
    "id": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_id",
     "!doc": "Set the id of the new Node. (optional)",
     "!data": {
      "submodule": "frame"
     }
    },
    "linkedcss": {
     "!type": "string|[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_linkedcss",
     "!doc": "An array of urls to external linked style sheets",
     "!data": {
      "submodule": "frame"
     }
    },
    "extracss": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_extracss",
     "!doc": "A string of CSS to add to the Head of the Editor",
     "!data": {
      "submodule": "frame"
     }
    },
    "defaultblock": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#attribute_defaultblock",
     "!doc": "The default tag to use for block level items, defaults to: p",
     "!data": {
      "submodule": "frame"
     }
    }
   },
   "EventTargetPublishConfig": {
    "broadcast": {
     "!type": "bool",
     "!doc": "whether or not the YUI instance and YUI global\n     are notified when the event is fired."
    },
    "bubbles": {
     "!type": "bool",
     "!doc": "Whether or not this event bubbles. Events can\n     only bubble if `emitFacade` is true."
    },
    "context": {
     "!type": "+Object",
     "!doc": "the default execution context for the listeners."
    },
    "defaultFn": {
     "!type": "fn()",
     "!doc": "the default function to execute when this event\n     fires if preventDefault was not called."
    },
    "emitFacade": {
     "!type": "bool",
     "!doc": "whether or not this event emits a facade."
    },
    "prefix": {
     "!type": "string",
     "!doc": "the prefix for this targets events, e.g., menu in menu:click."
    },
    "fireOnce": {
     "!type": "bool",
     "!doc": "if an event is configured to fire once,\n     new subscribers after the fire will be notified immediately."
    },
    "async": {
     "!type": "bool",
     "!doc": "fireOnce event listeners will fire synchronously\n     if the event has already fired unless `async` is `true`."
    },
    "preventable": {
     "!type": "bool",
     "!doc": "whether or not `preventDefault()` has an effect."
    },
    "preventedFn": {
     "!type": "fn()",
     "!doc": "a function that is executed when `preventDefault()` is called."
    },
    "queuable": {
     "!type": "bool",
     "!doc": "whether or not this event can be queued during bubbling."
    },
    "silent": {
     "!type": "bool",
     "!doc": "if silent is true, debug messages are not provided for this event."
    },
    "stoppedFn": {
     "!type": "fn()",
     "!doc": "a function that is executed when stopPropagation is called."
    },
    "monitored": {
     "!type": "bool",
     "!doc": "specifies whether or not this event should send notifications about\n     when the event has been attached, detached, or published."
    },
    "type": {
     "!type": "string",
     "!doc": "the event type (valid option if not provided as the first parameter to publish)."
    }
   },
   "FileFlashConfig": {
    "!proto": "config.BaseConfig",
    "id": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#attribute_id",
     "!doc": "A String containing the unique id of the file wrapped by the FileFlash instance.\nThe id is supplied by the Flash player uploader."
    },
    "size": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#attribute_size",
     "!doc": "The size of the file wrapped by FileFlash. This value is supplied by the Flash player uploader."
    },
    "name": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#attribute_name",
     "!doc": "The name of the file wrapped by FileFlash. This value is supplied by the Flash player uploader."
    },
    "dateCreated": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#attribute_dateCreated",
     "!doc": "The date that the file wrapped by FileFlash was created on. This value is supplied by the Flash player uploader."
    },
    "dateModified": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#attribute_dateModified",
     "!doc": "The date that the file wrapped by FileFlash was last modified on. This value is supplied by the Flash player uploader."
    },
    "bytesUploaded": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#attribute_bytesUploaded",
     "!doc": "The number of bytes of the file that has been uploaded to the server. This value is\nnon-zero only while a file is being uploaded."
    },
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#attribute_type",
     "!doc": "The type of the file wrapped by FileFlash. This value is provided by the Flash player\nuploader."
    },
    "uploder": {
     "!type": "+swf.SWF",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#attribute_uploder",
     "!doc": "The instance of Y.SWF wrapping the Flash player uploader associated with this file."
    }
   },
   "FileHTML5Config": {
    "!proto": "config.BaseConfig",
    "id": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_id",
     "!doc": "A String containing the unique id of the file wrapped by the FileFlash instance.\nThe id is supplied by the Flash player uploader."
    },
    "size": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_size",
     "!doc": "The size of the file wrapped by FileHTML5. This value is supplied by the instance of File()."
    },
    "name": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_name",
     "!doc": "The name of the file wrapped by FileHTML5. This value is supplied by the instance of File()."
    },
    "dateCreated": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_dateCreated",
     "!doc": "The date that the file wrapped by FileHTML5 was created on. This value is supplied by the instance of File()."
    },
    "dateModified": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_dateModified",
     "!doc": "The date that the file wrapped by FileHTML5 was last modified on. This value is supplied by the instance of File()."
    },
    "bytesUploaded": {
     "!type": "+datatype_date.Date",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_bytesUploaded",
     "!doc": "The number of bytes of the file that has been uploaded to the server. This value is\nnon-zero only while a file is being uploaded."
    },
    "type": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_type",
     "!doc": "The type of the file wrapped by FileHTML. This value is provided by the instance of File()"
    },
    "file": {
     "!type": "+file.File",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_file",
     "!doc": "The pointer to the instance of File() wrapped by FileHTML5."
    },
    "xhr": {
     "!type": "+XMLHttpRequest",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_xhr",
     "!doc": "The pointer to the instance of XMLHttpRequest used by FileHTML5 to upload the file."
    },
    "xhrHeaders": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_xhrHeaders",
     "!doc": "The dictionary of headers that should be set on the XMLHttpRequest object before\nsending it."
    },
    "xhrWithCredentials": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_xhrWithCredentials",
     "!doc": "A Boolean indicating whether the XMLHttpRequest should be sent with user credentials.\nThis does not affect same-site requests."
    },
    "boundEventHandler": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#attribute_boundEventHandler",
     "!doc": "The bound event handler used to handle events from XMLHttpRequest."
    }
   },
   "EventSimulateGestureConfig": {
    "point": {
     "!type": "[?]",
     "!doc": "(Optional) Indicates the [x, y] coordinates\n       where the flick should be simulated. Default is the center of the\n       node element."
    },
    "hold": {
     "!type": "number",
     "!doc": "(Optional) The hold time in milliseconds.\n       This is the time between `touchstart` and `touchend` event generation.\n       Default is 3000ms (3 seconds).\n\n     Valid options properties for the `move` gesture:"
    },
    "times": {
     "!type": "number",
     "!doc": "(Optional) Indicates the number of taps."
    },
    "delay": {
     "!type": "number",
     "!doc": "(Optional) The number of milliseconds\n       before the next tap simulation happens. This is valid only when `times`\n       is more than 1.\n\n     Valid options properties for the `doubletap` gesture:"
    },
    "path": {
     "!type": "+Object",
     "!doc": "(Optional) Indicates the path of the finger\n       movement. Its an object with three optional properties: `point`,\n       `xdist` and  `ydist`."
    },
    "duration": {
     "!type": "number",
     "!doc": "(Optional) The duration of the\n       gesture in milliseconds."
    },
    "axis": {
     "!type": "string",
     "!doc": "(Optional) Valid values are either\n       \"x\" or \"y\". Indicates axis to move along. The flick can move to one of\n       4 directions(left, right, up and down)."
    },
    "distance": {
     "!type": "number",
     "!doc": "(Optional) Distance to move in pixels"
    },
    "center": {
     "!type": "[?]",
     "!doc": "(Optional) The center of the circle where\n       two fingers are placed. Default is the center of the node element."
    },
    "r1": {
     "!type": "number",
     "!doc": "(Optional) Pixel radius of the start circle\n       where 2 fingers will be on when the gesture starts. The circles are\n       centered at the center of the element. Default is a fourth of the node\n       element width or height, whichever is smaller."
    },
    "r2": {
     "!type": "number",
     "!doc": "(Optional) Pixel radius of the end circle\n       when this gesture ends. Default is a fourth of the node element width or\n       height, whichever is smaller."
    },
    "start": {
     "!type": "number",
     "!doc": "(Optional) Starting degree of the first\n       finger. The value is relative to the path of the north. Default is 0\n       (i.e., 12:00 on a clock)."
    },
    "rotation": {
     "!type": "number",
     "!doc": "(Required) Degrees to rotate from\n       the starting degree. A negative value means rotation to the\n       counter-clockwise direction."
    }
   },
   "HandlebarsRenderConfig": {
    "helpers": {
     "!type": "+Object",
     "!doc": "Helper functions."
    },
    "partials": {
     "!type": "+Object",
     "!doc": "Partials."
    },
    "data": {
     "!type": "+Object",
     "!doc": "Data."
    }
   },
   "HighlightAllFoldConfig": {
    "startsWith": {
     "!type": "bool",
     "!doc": "If `true`, matches must be\n    anchored to the beginning of the string."
    }
   },
   "HighlightAllConfig": {
    "caseSensitive": {
     "!type": "bool",
     "!doc": "If `true`, matching will\n    be case-sensitive."
    },
    "startsWith": {
     "!type": "bool",
     "!doc": "If `true`, matches must be\n    anchored to the beginning of the string."
    }
   },
   "HighlightStartConfig": {
    "caseSensitive": {
     "!type": "bool",
     "!doc": "If `true`, matching will\n    be case-sensitive."
    }
   },
   "HighlightWordsConfig": {
    "caseSensitive": {
     "!type": "bool",
     "!doc": "If `true`, matching will\n    be case-sensitive."
    }
   },
   "ImgLoadGroupConfig": {
    "!proto": "config.BaseConfig",
    "name": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#attribute_name",
     "!doc": "Name for the group. Only used to identify the group in logging statements."
    },
    "timeLimit": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#attribute_timeLimit",
     "!doc": "Time limit, in seconds, after which images are fetched regardless of trigger events."
    },
    "foldDistance": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#attribute_foldDistance",
     "!doc": "Distance below the fold for which images are loaded. Images are not loaded until they are at most this distance away from (or above) the fold.\nThis check is performed at page load (domready) and after any window scroll or window resize event (until all images are loaded)."
    },
    "className": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#attribute_className",
     "!doc": "Class name that will identify images belonging to the group. This class name will be removed from each element in order to fetch images.\nThis class should have, in its CSS style definition, \"<code>background:none !important;</code>\"."
    },
    "classNameAction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#attribute_classNameAction",
     "!doc": "Determines how to act when className is used as the way to delay load images. The \"default\" action is to just\nremove the class name. The \"enhanced\" action is to remove the class name and also set the src attribute if\nthe element is an img."
    }
   },
   "ImgLoadImgObjConfig": {
    "!proto": "config.BaseConfig",
    "domId": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_domId",
     "!doc": "HTML DOM id of the image element."
    },
    "bgUrl": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_bgUrl",
     "!doc": "Background URL for the image.\nFor an image whose URL is specified by \"<code>background-image</code>\" in the elements style."
    },
    "srcUrl": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_srcUrl",
     "!doc": "Source URL for the image.\nFor an image whose URL is specified by a \"<code>src</code>\" attribute in the DOM element."
    },
    "width": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_width",
     "!doc": "Pixel width of the image. Will be set as a <code>width</code> attribute on the DOM element after the image is fetched.\nDefaults to the natural width of the image (no <code>width</code> attribute will be set).\nUsually only used with src images."
    },
    "height": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_height",
     "!doc": "Pixel height of the image. Will be set as a <code>height</code> attribute on the DOM element after the image is fetched.\nDefaults to the natural height of the image (no <code>height</code> attribute will be set).\nUsually only used with src images."
    },
    "setVisible": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_setVisible",
     "!doc": "Whether the images <code>style.visibility</code> should be set to <code>visible</code> after the image is fetched.\nUsed when setting images as <code>visibility:hidden</code> prior to image fetching."
    },
    "isPng": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_isPng",
     "!doc": "Whether the image is a PNG.\nPNG images get special treatment in that the URL is specified through AlphaImageLoader for IE, versions 6 and earlier.\nOnly used with background images."
    },
    "sizingMethod": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_sizingMethod",
     "!doc": "AlphaImageLoader <code>sizingMethod</code> property to be set for the image.\nOnly set if <code>isPng</code> value for this image is set to <code>true</code>.\nDefaults to <code>scale</code>."
    },
    "enabled": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#attribute_enabled",
     "!doc": "AlphaImageLoader <code>enabled</code> property to be set for the image.\nOnly set if <code>isPng</code> value for this image is set to <code>true</code>.\nDefaults to <code>true</code>."
    }
   },
   "IOStringifyConfig": {
    "useDisabled": {
     "!type": "bool",
     "!doc": "Whether to include disabled fields."
    },
    "extra": {
     "!type": "+Object|string",
     "!doc": "Extra values to include. May be a query string or an object with key/value pairs."
    }
   },
   "LoaderConfig": {
    "base": {
     "!type": "string",
     "!doc": "The base dir which to fetch this module from"
    },
    "comboBase": {
     "!type": "string",
     "!doc": "The Combo service base path. Ex: `http://yui.yahooapis.com/combo?`"
    },
    "root": {
     "!type": "string",
     "!doc": "The root path to prepend to module names for the combo service. Ex: `2.5.2/build/`"
    },
    "filter": {
     "!type": "string|+Object",
     "!doc": "A filter to apply to result urls. <a href=\"#property_filter\">See filter property</a>"
    },
    "filters": {
     "!type": "+Object",
     "!doc": "Per-component filter specification.  If specified for a given component, this overrides the filter config."
    },
    "combine": {
     "!type": "bool",
     "!doc": "Use a combo service to reduce the number of http connections required to load your dependencies"
    },
    "async": {
     "!type": "bool",
     "!doc": "Fetch files in async"
    },
    "ignore:": {
     "!type": "[?]",
     "!doc": "A list of modules that should never be dynamically loaded"
    },
    "force": {
     "!type": "[?]",
     "!doc": "A list of modules that should always be loaded when required, even if already present on the page"
    },
    "insertBefore": {
     "!type": "+HTMLElement|string",
     "!doc": "Node or id for a node that should be used as the insertion point for new nodes"
    },
    "jsAttributes": {
     "!type": "+Object",
     "!doc": "Object literal containing attributes to add to script nodes"
    },
    "cssAttributes": {
     "!type": "+Object",
     "!doc": "Object literal containing attributes to add to link nodes"
    },
    "timeout": {
     "!type": "number",
     "!doc": "The number of milliseconds before a timeout occurs when dynamically loading nodes.  If not set, there is no timeout"
    },
    "context": {
     "!type": "+Object",
     "!doc": "Execution context for all callbacks"
    },
    "onSuccess": {
     "!type": "fn()",
     "!doc": "Callback for the success event"
    },
    "onFailure": {
     "!type": "fn()",
     "!doc": "Callback for the failure event"
    },
    "onTimeout": {
     "!type": "fn()",
     "!doc": "Callback for the timeout event"
    },
    "onProgress": {
     "!type": "fn()",
     "!doc": "Callback executed each time a script or css file is loaded"
    },
    "modules": {
     "!type": "+Object",
     "!doc": "A list of module definitions.  See <a href=\"#method_addModule\">Loader.addModule</a> for the supported module metadata"
    },
    "groups": {
     "!type": "+Object",
     "!doc": "A list of group definitions.  Each group can contain specific definitions for `base`, `comboBase`, `combine`, and accepts a list of `modules`."
    },
    "2in3": {
     "!type": "string",
     "!doc": "The version of the YUI 2 in 3 wrapper to use.  The intrinsic support for YUI 2 modules in YUI 3 relies on versions of the YUI 2 components inside YUI 3 module wrappers.  These wrappers change over time to accomodate the issues that arise from running YUI 2 in a YUI 3 sandbox."
    },
    "yui2": {
     "!type": "string",
     "!doc": "When using the 2in3 project, you can select the version of YUI 2 to use.  Valid values are `2.2.2`, `2.3.1`, `2.4.1`, `2.5.2`, `2.6.0`, `2.7.0`, `2.8.0`, `2.8.1` and `2.9.0` [default] -- plus all versions of YUI 2 going forward."
    },
    "doBeforeLoader": {
     "!type": "fn()",
     "!doc": "An optional hook that allows for the patching of the loader instance. The `Y` instance is available as `this.context` and the only argument to the function is the Loader configuration object."
    }
   },
   "LoaderAddGroupConfig": {
    "name": {
     "!type": "string",
     "!doc": "required, the group name"
    },
    "base": {
     "!type": "string",
     "!doc": "The base directory for this module group"
    },
    "root": {
     "!type": "string",
     "!doc": "The root path to add to each combo resource path"
    },
    "combine": {
     "!type": "bool",
     "!doc": "Should the request be combined"
    },
    "comboBase": {
     "!type": "string",
     "!doc": "Combo service base path"
    },
    "modules": {
     "!type": "+Object",
     "!doc": "The group of modules"
    }
   },
   "LoaderAddModuleConfig": {
    "name": {
     "!type": "string",
     "!doc": "Required, the component name"
    },
    "type": {
     "!type": "string",
     "!doc": "Required, the component type (js or css)"
    },
    "path": {
     "!type": "string",
     "!doc": "Required, the path to the script from `base`"
    },
    "requires": {
     "!type": "[?]",
     "!doc": "Array of modules required by this component"
    },
    "optional": {
     "!type": "[?]",
     "!doc": "Array of optional modules for this component"
    },
    "supersedes": {
     "!type": "[?]",
     "!doc": "Array of the modules this component replaces"
    },
    "after": {
     "!type": "[?]",
     "!doc": "Array of modules the components which, if present, should be sorted above this one"
    },
    "after_map": {
     "!type": "+Object",
     "!doc": "Faster alternative to after -- supply a hash instead of an array"
    },
    "rollup": {
     "!type": "number",
     "!doc": "The number of superseded modules required for automatic rollup"
    },
    "fullpath": {
     "!type": "string",
     "!doc": "If `fullpath` is specified, this is used instead of the configured `base + path`"
    },
    "skinnable": {
     "!type": "bool",
     "!doc": "Flag to determine if skin assets should automatically be pulled in"
    },
    "submodules": {
     "!type": "+Object",
     "!doc": "Hash of submodules"
    },
    "group": {
     "!type": "string",
     "!doc": "The group the module belongs to -- this is set automatically when it is added as part of a group configuration."
    },
    "lang": {
     "!type": "[?]",
     "!doc": "Array of BCP 47 language tags of languages for which this module has localized resource bundles, e.g., `[\"en-GB\", \"zh-Hans-CN\"]`"
    },
    "condition": {
     "!type": "+Object",
     "!doc": "Specifies that the module should be loaded automatically if a condition is met. This is an object with up to four fields:"
    },
    "testresults": {
     "!type": "+Object",
     "!doc": "A hash of test results from `Y.Features.all()`"
    },
    "configFn": {
     "!type": "fn()",
     "!doc": "A function to exectute when configuring this module"
    },
    "optionalRequires": {
     "!type": "[string]",
     "!doc": "List of dependencies that\n       may optionally be loaded by this loader. This is targeted mostly at\n       polyfills, since they should not be in the list of requires because\n       polyfills are assumed to be available in the global scope."
    },
    "test": {
     "!type": "fn()",
     "!doc": "Test to be called when this module is\n       added as an optional dependency of another module. If the test function\n       returns `false`, the module will be ignored and will not be attached to\n       this YUI instance."
    }
   },
   "plugin": {
    "NodeFocusManagerConfig": {
     "focused": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#attribute_focused",
      "!doc": "Boolean indicating that one of the descendants is focused."
     },
     "descendants": {
      "!type": "+Y.NodeList",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#attribute_descendants",
      "!doc": "String representing the CSS selector used to define the descendant Nodes\nwhose focus should be managed."
     },
     "activeDescendant": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#attribute_activeDescendant",
      "!doc": "<p>Node, or index of the Node, representing the descendant that is either\nfocused or is focusable (<code>tabIndex</code> attribute is set to 0).\nThe value cannot represent a disabled descendant Node.  Use a value of -1\nto remove all descendant Nodes from the default tab flow.\nIf no value is specified, the active descendant will be inferred using\nthe following criteria:</p>\n<ol>\n<li>Examining the <code>tabIndex</code> attribute of each descendant and\nusing the first descendant whose <code>tabIndex</code> attribute is set\nto 0</li>\n<li>If no default can be inferred then the value is set to either 0 or\nthe index of the first enabled descendant.</li>\n</ol>"
     },
     "keys": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#attribute_keys",
      "!doc": "Object literal representing the keys to be used to navigate between the\nnext/previous descendant.  The format for the attributes value is\n<code>{ next: \"down:40\", previous: \"down:38\" }</code>.  The value for the\n\"next\" and \"previous\" properties are used to attach\n<a href=\"event/#keylistener\"><code>key</code></a> event listeners. See\nthe <a href=\"event/#keylistener\">Using the key Event</a> section of\nthe Event documentation for more information on \"key\" event listeners."
     },
     "focusClass": {
      "!type": "string|+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#attribute_focusClass",
      "!doc": "String representing the name of class applied to the focused active\ndescendant Node.  Can also be an object literal used to define both the\nclass name, and the Node to which the class should be applied.  If using\nan object literal, the format is:\n<code>{ className: \"focus\", fn: myFunction }</code>.  The function\nreferenced by the <code>fn</code> property in the object literal will be\npassed a reference to the currently focused active descendant Node."
     },
     "circular": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#attribute_circular",
      "!doc": "Boolean indicating if focus should be set to the first/last descendant\nwhen the end or beginning of the descendants has been reached."
     }
    },
    "NodeMenuNavConfig": {
     "useARIA": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeMenuNav.html#attribute_useARIA",
      "!doc": "Boolean indicating if use of the WAI-ARIA Roles and States should be\nenabled for the menu."
     },
     "autoSubmenuDisplay": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeMenuNav.html#attribute_autoSubmenuDisplay",
      "!doc": "Boolean indicating if submenus are automatically made visible when the\nuser mouses over the menus items."
     },
     "submenuShowDelay": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeMenuNav.html#attribute_submenuShowDelay",
      "!doc": "Number indicating the time (in milliseconds) that should expire before a\nsubmenu is made visible when the user mouses over the menus label."
     },
     "submenuHideDelay": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeMenuNav.html#attribute_submenuHideDelay",
      "!doc": "Number indicating the time (in milliseconds) that should expire before a\nsubmenu is hidden when the user mouses out of a menu label heading in the\ndirection of a submenu."
     },
     "mouseOutHideDelay": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeMenuNav.html#attribute_mouseOutHideDelay",
      "!doc": "Number indicating the time (in milliseconds) that should expire before a\nsubmenu is hidden when the user mouses out of it."
     }
    }
   },
   "NodeSimulateGestureConfig": {
    "point": {
     "!type": "[?]",
     "!doc": "(Optional) Indicates the [x, y] coordinates\n       where the flick should be simulated. Default is the center of the\n       node element."
    },
    "hold": {
     "!type": "number",
     "!doc": "(Optional) The hold time in milliseconds.\n       This is the time between `touchstart` and `touchend` event generation.\n       Default is 3000ms (3 seconds).\n\n     Valid options properties for the `move` gesture:"
    },
    "times": {
     "!type": "number",
     "!doc": "(Optional) Indicates the number of taps."
    },
    "delay": {
     "!type": "number",
     "!doc": "(Optional) The number of milliseconds\n       before the next tap simulation happens. This is valid only when `times`\n       is more than 1.\n\n     Valid options properties for the `doubletap` gesture:"
    },
    "path": {
     "!type": "+Object",
     "!doc": "(Optional) Indicates the path of the finger\n       movement. Its an object with three optional properties: `point`,\n       `xdist` and  `ydist`."
    },
    "duration": {
     "!type": "number",
     "!doc": "(Optional) The duration of the\n       gesture in milliseconds."
    },
    "axis": {
     "!type": "string",
     "!doc": "(Optional) Valid values are either\n       \"x\" or \"y\". Indicates axis to move along. The flick can move to one of\n       4 directions(left, right, up and down)."
    },
    "distance": {
     "!type": "number",
     "!doc": "(Optional) Distance to move in pixels"
    },
    "center": {
     "!type": "[?]",
     "!doc": "(Optional) The center of the circle where\n       two fingers are placed. Default is the center of the node element."
    },
    "r1": {
     "!type": "number",
     "!doc": "(Optional) Pixel radius of the start circle\n       where 2 fingers will be on when the gesture starts. The circles are\n       centered at the center of the element. Default is a fourth of the node\n       element width or height, whichever is smaller."
    },
    "r2": {
     "!type": "number",
     "!doc": "(Optional) Pixel radius of the end circle\n       when this gesture ends. Default is a fourth of the node element width or\n       height, whichever is smaller."
    },
    "start": {
     "!type": "number",
     "!doc": "(Optional) Starting degree of the first\n       finger. The value is relative to the path of the north. Default is 0\n       (i.e., 12:00 on a clock)."
    },
    "rotation": {
     "!type": "number",
     "!doc": "(Required) Degrees to rotate from\n       the starting degree. A negative value means rotation to the\n       counter-clockwise direction."
    }
   },
   "NumberParseConfig": {
    "prefix": {
     "!type": "string",
     "!doc": "String to be removed from the start, like a currency designator \"\""
    },
    "decimalPlaces": {
     "!type": "number",
     "!doc": "Ignored, it is accepted only for compatibility with [Y.Date.format](#method_format)."
    },
    "decimalSeparator": {
     "!type": "string",
     "!doc": "Decimal separator."
    },
    "thousandsSeparator": {
     "!type": "string",
     "!doc": "Thousands separator."
    },
    "suffix": {
     "!type": "string",
     "!doc": "String to be removed from the end of the number, like \" items\"."
    }
   },
   "PaginatorConfig": {
    "page": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Paginator.html#attribute_page",
     "!doc": "Current page count. First page is 1.",
     "!data": {
      "submodule": "paginator-core"
     }
    },
    "totalPages": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Paginator.html#attribute_totalPages",
     "!doc": "Total number of pages to display",
     "!data": {
      "submodule": "paginator-core"
     }
    },
    "itemsPerPage": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Paginator.html#attribute_itemsPerPage",
     "!doc": "Maximum number of items per page. A value of negative one (-1) indicates\n    all items on one page.",
     "!data": {
      "submodule": "paginator-core"
     }
    },
    "totalItems": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Paginator.html#attribute_totalItems",
     "!doc": "Total number of items in all pages.",
     "!data": {
      "submodule": "paginator-core"
     }
    }
   },
   "Paginator": {
    "UrlConfig": {
     "pageUrl": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Paginator.Url.html#attribute_pageUrl",
      "!doc": "URL to return formatted with the page number. URL uses `Y.Lang.sub` for page number stubstitutions.\n\nFor example, if the page number is `3`, setting the `pageUrl` to `\"?pg={page}\"`, will result in `?pg=3`",
      "!data": {
       "submodule": "paginator-url"
      }
     }
    }
   },
   "ParallelConfig": {
    "context": {
     "!type": "+Object",
     "!doc": "The execution context of the callback to done"
    }
   },
   "PjaxBaseNavigateConfig": {
    "replace": {
     "!type": "bool",
     "!doc": "Whether or not the current history\n      entry will be replaced, or a new entry will be created. Will default\n      to `true` if the specified `url` is the same as the current URL."
    },
    "force": {
     "!type": "bool",
     "!doc": "Whether the enhanced navigation\n      should occur even in browsers without HTML5 history."
    }
   },
   "PjaxBaseConfig": {
    "linkSelector": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PjaxBase.html#attribute_linkSelector",
     "!doc": "CSS selector string used to filter link click events so that only the links\nwhich match it will have the enhanced navigation behavior of Pjax applied.\n\nWhen a link is clicked and that link matches this selector, Pjax will\nattempt to dispatch to any route handlers matching the links `href` URL. If\nHTML5 history is not supported or if no route handlers match, the link click\nwill be handled by the browser just like any old link.",
     "!data": {
      "submodule": "pjax-base"
     }
    },
    "navigateOnHash": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PjaxBase.html#attribute_navigateOnHash",
     "!doc": "Whether navigating to a hash-fragment identifier on the current page should\nbe enhanced and cause the `navigate` event to fire.\n\nBy default Pjax allows the browser to perform its default action when a user\nis navigating within a page by clicking in-page links\n(e.g. `<a href=\"#top\">Top of page</a>`) and does not attempt to interfere or\nenhance in-page navigation.",
     "!data": {
      "submodule": "pjax-base"
     }
    },
    "scrollToTop": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PjaxBase.html#attribute_scrollToTop",
     "!doc": "Whether the page should be scrolled to the top after navigating to a URL.\n\nWhen the user clicks the browsers back button, the previous scroll position\nwill be maintained.",
     "!data": {
      "submodule": "pjax-base"
     }
    },
    "allowFallThrough": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PjaxBase.html#attribute_allowFallThrough",
     "!doc": "Whether to set `window.location` when calling `navigate()`\nif no routes match the specified URL.",
     "!data": {
      "submodule": "pjax-base"
     }
    }
   },
   "PjaxContentConfig": {
    "addPjaxParam": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PjaxContent.html#attribute_addPjaxParam",
     "!doc": "If `true`, a \"pjax=1\" query parameter will be appended to all URLs\nrequested via Pjax.\n\nBrowsers ignore HTTP request headers when caching content, so if the\nsame URL is used to request a partial Pjax page and a full page, the\nbrowser will cache them under the same key and may later load the\ncached partial page when the user actually requests a full page (or vice\nversa).\n\nTo prevent this, we can add a bogus query parameter to the URL so that\nPjax URLs will always be cached separately from non-Pjax URLs.",
     "!data": {
      "submodule": "pjax-content"
     }
    },
    "contentSelector": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PjaxContent.html#attribute_contentSelector",
     "!doc": "CSS selector used to extract a specific portion of the content of a page\nloaded via Pjax.\n\nFor example, if you wanted to load the page `example.html` but only use\nthe content within an element with the id \"pjax-content\", youd set\n`contentSelector` to \"#pjax-content\".\n\nIf not set, the entire page will be used.",
     "!data": {
      "submodule": "pjax-content"
     }
    },
    "titleSelector": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PjaxContent.html#attribute_titleSelector",
     "!doc": "CSS selector used to extract a page title from the content of a page\nloaded via Pjax.\n\nBy default this is set to extract the title from the `<title>` element,\nbut you could customize it to extract the title from an `<h1>`, or from\nany other element, if thats more appropriate for the content youre\nloading.",
     "!data": {
      "submodule": "pjax-content"
     }
    },
    "timeout": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/PjaxContent.html#attribute_timeout",
     "!doc": "Time in milliseconds after which an Ajax request should time out.",
     "!data": {
      "submodule": "pjax-content"
     }
    }
   },
   "PjaxConfig": {
    "!proto": "config.RouterConfig",
    "container": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Pjax.html#attribute_container",
     "!doc": "Node into which content should be inserted when a page is loaded via\nPjax. This nodes existing contents will be removed to make way for the\nnew content.\n\nIf not set, loaded content will not be automatically inserted into the\npage."
    }
   },
   "RecordConfig": {
    "id": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Record.html#attribute_id",
     "!doc": "Unique ID of the record instance"
    },
    "data": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Record.html#attribute_data",
     "!doc": "The object stored within the record instance"
    }
   },
   "RecordsetConfig": {
    "!proto": "config.BaseConfig",
    "records": {
     "!type": "[+queue_promote.Record]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Recordset.html#attribute_records",
     "!doc": "An array of Records that the Recordset is storing.  Passing an array\nof raw record data is also accepted.  The data for each item will be\nwrapped in a Record instance.",
     "!data": {
      "submodule": "recordset-base"
     }
    },
    "table": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Recordset.html#attribute_table",
     "!doc": "A hash table where the ID of the record is the key, and the record\ninstance is the value.",
     "!data": {
      "submodule": "recordset-base"
     }
    },
    "key": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Recordset.html#attribute_key",
     "!doc": "The ID to use as the key in the hash table.",
     "!data": {
      "submodule": "recordset-base"
     }
    }
   },
   "RecordsetIndexerConfig": {
    "hashTables": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/RecordsetIndexer.html#attribute_hashTables",
     "!doc": "Collection of all the hashTables created by the plugin.\nThe individual tables can be accessed by the key they are hashing against.",
     "!data": {
      "submodule": "recordset-indexer"
     }
    }
   },
   "RecordsetSortConfig": {
    "lastSortProperties": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/RecordsetSort.html#attribute_lastSortProperties",
     "!doc": "The last properties used to sort. Consists of an object literal with the keys \"field\", \"desc\", and \"sorter\"",
     "!data": {
      "submodule": "recordset-sort"
     }
    },
    "defaultSorter": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/RecordsetSort.html#attribute_defaultSorter",
     "!doc": "A boolean telling if the recordset is in a sorted state.",
     "!data": {
      "submodule": "recordset-sort"
     }
    }
   },
   "ResizeConfig": {
    "!proto": "config.BaseConfig",
    "autoHide": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_autoHide",
     "!doc": "False to ensure that the resize handles are always visible, true to\ndisplay them only when the user mouses over the resizable borders."
    },
    "defMinHeight": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_defMinHeight",
     "!doc": "The default minimum height of the element. Only used when\nResizeConstrained is not plugged."
    },
    "defMinWidth": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_defMinWidth",
     "!doc": "The default minimum width of the element. Only used when\nResizeConstrained is not plugged."
    },
    "handles": {
     "!type": "[?]|string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_handles",
     "!doc": "The handles to use (any combination of): t, b, r, l, bl,\nbr, tl, tr. Can use a shortcut of All."
    },
    "handlesWrapper": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_handlesWrapper",
     "!doc": "Node to wrap the resize handles."
    },
    "node": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_node",
     "!doc": "The selector or element to resize. Required."
    },
    "resizing": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_resizing",
     "!doc": "True when the element is being Resized."
    },
    "wrap": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_wrap",
     "!doc": "True to wrap an element with a div if needed (required for textareas\nand images, defaults to false) in favor of the handles config option.\nThe wrapper element type (default div) could be over-riden passing the\n<code>wrapper</code> attribute."
    },
    "wrapTypes": {
     "!type": "+Regex",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_wrapTypes",
     "!doc": "Elements that requires a wrapper by default. Normally are elements\nwhich cannot have children elements."
    },
    "wrapper": {
     "!type": "string|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Resize.html#attribute_wrapper",
     "!doc": "Element to wrap the <code>wrapTypes</code>. This element will house\nthe handles elements."
    }
   },
   "ScrollViewConfig": {
    "!proto": "config.WidgetConfig",
    "axis": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_axis",
     "!doc": "Specifies ability to scroll on x, y, or x and y axis/axes.",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "scrollX": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_scrollX",
     "!doc": "The current scroll position in the x-axis",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "scrollY": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_scrollY",
     "!doc": "The current scroll position in the y-axis",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "deceleration": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_deceleration",
     "!doc": "Drag coefficent for inertial scrolling. The closer to 1 this\nvalue is, the less friction during scrolling.",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "bounce": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_bounce",
     "!doc": "Drag coefficient for intertial scrolling at the upper\nand lower boundaries of the scrollview. Set to 0 to\ndisable \"rubber-banding\".",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "flick": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_flick",
     "!doc": "The minimum distance and/or velocity which define a flick. Can be set to false,\nto disable flick support (note: drag support is enabled/disabled separately)",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "drag": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_drag",
     "!doc": "Enable/Disable dragging the ScrollView content (note: flick support is enabled/disabled separately)",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "snapDuration": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_snapDuration",
     "!doc": "The default duration to use when animating the bounce snap back.",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "snapEasing": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_snapEasing",
     "!doc": "The default easing to use when animating the bounce snap back.",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "easing": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_easing",
     "!doc": "The default easing used when animating the flick",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "frameDuration": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_frameDuration",
     "!doc": "The interval (ms) used when animating the flick for JS-timer animations",
     "!data": {
      "submodule": "scrollview-base"
     }
    },
    "bounceRange": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ScrollView.html#attribute_bounceRange",
     "!doc": "The default bounce distance in pixels",
     "!data": {
      "submodule": "scrollview-base"
     }
    }
   },
   "ClickableRailConfig": {
    "clickableRail": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ClickableRail.html#attribute_clickableRail",
     "!doc": "Enable or disable clickable rail support.",
     "!data": {
      "submodule": "clickable-rail"
     }
    }
   },
   "SliderBaseConfig": {
    "!proto": "config.WidgetConfig",
    "axis": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SliderBase.html#attribute_axis",
     "!doc": "Axis upon which the Sliders thumb moves.  &quot;x&quot; for\nhorizontal, &quot;y&quot; for vertical.",
     "!data": {
      "submodule": "slider-base"
     }
    },
    "length": {
     "!type": "string|number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SliderBase.html#attribute_length",
     "!doc": "The length of the rail (exclusive of the end caps if positioned by\nCSS).  This corresponds to the movable range of the thumb.",
     "!data": {
      "submodule": "slider-base"
     }
    },
    "thumbUrl": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SliderBase.html#attribute_thumbUrl",
     "!doc": "Path to the thumb image.  This will be used as both the thumb and\nshadow as a sprite.  Defaults at render() to thumb-x.png or\nthumb-y.png in the skin directory of the current skin.",
     "!data": {
      "submodule": "slider-base"
     }
    }
   },
   "SliderValueRangeConfig": {
    "min": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SliderValueRange.html#attribute_min",
     "!doc": "The value associated with the farthest top, left position of the\nrail.  Can be greater than the configured <code>max</code> if you\nwant values to increase from right-to-left or bottom-to-top.",
     "!data": {
      "submodule": "slider-value-range"
     }
    },
    "max": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SliderValueRange.html#attribute_max",
     "!doc": "The value associated with the farthest bottom, right position of\nthe rail.  Can be less than the configured <code>min</code> if\nyou want values to increase from right-to-left or bottom-to-top.",
     "!data": {
      "submodule": "slider-value-range"
     }
    },
    "minorStep": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SliderValueRange.html#attribute_minorStep",
     "!doc": "amount to increment/decrement the Slider value\nwhen the arrow up/down/left/right keys are pressed",
     "!data": {
      "submodule": "slider-value-range"
     }
    },
    "majorStep": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SliderValueRange.html#attribute_majorStep",
     "!doc": "amount to increment/decrement the Slider value\nwhen the page up/down keys are pressed",
     "!data": {
      "submodule": "slider-value-range"
     }
    },
    "value": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/SliderValueRange.html#attribute_value",
     "!doc": "The value associated with the thumbs current position on the\nrail. Defaults to the value inferred from the thumbs current\nposition. Specifying value in the constructor will move the\nthumb to the position that corresponds to the supplied value.",
     "!data": {
      "submodule": "slider-value-range"
     }
    }
   },
   "SortableConfig": {
    "!proto": "config.BaseConfig",
    "handles": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Sortable.html#attribute_handles",
     "!doc": "Drag handles to pass on to the internal DD.Delegate instance."
    },
    "container": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Sortable.html#attribute_container",
     "!doc": "A selector query to get the container to listen for mousedown events on. All \"nodes\" should be a child of this container."
    },
    "nodes": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Sortable.html#attribute_nodes",
     "!doc": "A selector query to get the children of the \"container\" to make draggable elements from."
    },
    "opacity": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Sortable.html#attribute_opacity",
     "!doc": "The opacity to change the proxy item to when dragging."
    },
    "opacityNode": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Sortable.html#attribute_opacityNode",
     "!doc": "The node to set opacity on when dragging (dragNode or currentNode). Default: currentNode."
    },
    "id": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Sortable.html#attribute_id",
     "!doc": "The id of this Sortable, used to get a reference to this Sortable list from another list."
    },
    "moveType": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Sortable.html#attribute_moveType",
     "!doc": "How should an item move to another list: insert, swap, move, copy. Default: insert"
    },
    "invalid": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Sortable.html#attribute_invalid",
     "!doc": "A selector string to test if a list item is invalid and not sortable"
    }
   },
   "TabConfig": {
    "!proto": "config.WidgetConfig",
    "triggerEvent": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Tab.html#attribute_triggerEvent"
    },
    "label": {
     "!type": "+HTML",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Tab.html#attribute_label"
    },
    "content": {
     "!type": "+HTML",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Tab.html#attribute_content"
    },
    "panelNode": {
     "!type": "+Y.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Tab.html#attribute_panelNode"
    }
   },
   "TreeConfig": {
    "!proto": "config.BaseConfig",
    "nodes": {
     "!type": "[+Object]|[+tree.Tree.Node]",
     "!doc": "Array of tree node config\n        objects or `Tree.Node` instances to add to this tree at initialization\n        time."
    },
    "rootNode": {
     "!type": "+Object|+tree.Tree.Node",
     "!doc": "Node to use as the root node of\n        this tree."
    }
   },
   "Tree": {
    "Node": {
     "LabelableConfig": {
      "label": {
       "!type": "string",
       "!doc": "Label for this node."
      }
     },
     "OpenableCloseConfig": {
      "silent": {
       "!type": "bool",
       "!doc": "If `true`, the `close` event\n        will be suppressed."
      },
      "src": {
       "!type": "string",
       "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
      }
     },
     "OpenableOpenConfig": {
      "silent": {
       "!type": "bool",
       "!doc": "If `true`, the `open` event\n        will be suppressed."
      },
      "src": {
       "!type": "string",
       "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
      }
     },
     "OpenableToggleOpenConfig": {
      "silent": {
       "!type": "bool",
       "!doc": "If `true`, events will be\n        suppressed."
      },
      "src": {
       "!type": "string",
       "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
      }
     },
     "SelectableSelectConfig": {
      "silent": {
       "!type": "bool",
       "!doc": "If `true`, the `select` event\n        will be suppressed."
      },
      "src": {
       "!type": "string",
       "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
      }
     },
     "SelectableUnselectConfig": {
      "silent": {
       "!type": "bool",
       "!doc": "If `true`, the `unselect` event\n        will be suppressed."
      },
      "src": {
       "!type": "string",
       "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
      }
     },
     "SortableSortConfig": {
      "silent": {
       "!type": "bool",
       "!doc": "If `true`, no `sort` event will be\n        fired."
      },
      "sortComparator": {
       "!type": "fn()",
       "!doc": "Custom comparator function to\n        use. If specified, this will become the nodes new comparator\n        function, overwriting any previous comparator function that was set\n        for the node."
      },
      "sortReverse": {
       "!type": "bool",
       "!doc": "If `true`, children will be\n        sorted in reverse (descending) order. Otherwise theyll be sorted in\n        ascending order. This will become the nodes new sort order,\n        overwriting any previous sort order that was set for the node."
      },
      "src": {
       "!type": "string",
       "!doc": "Source of the sort operation. Will be\n        passed along to the `sort` event facade."
      }
     }
    },
    "OpenableCloseNodeConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `close` event\n        will be suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "OpenableOpenNodeConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `open` event\n        will be suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "OpenableToggleOpenNodeConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, events will be\n        suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "SelectableSelectNodeConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `select` event\n        will be suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "SelectableUnselectConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `unselect` event\n        will be suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "SelectableUnselectNodeConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `unselect` event\n        will be suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "SelectableConfig": {
     "multiSelect": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Tree.Selectable.html#attribute_multiSelect",
      "!doc": "Whether or not to allow multiple nodes to be selected at once.",
      "!data": {
       "submodule": "tree-selectable"
      }
     }
    },
    "SortableConfig": {
     "sortComparator": {
      "!type": "fn()",
      "!doc": "Default comparator function to use\n    when sorting a nodes children if the node itself doesnt have a custom\n    comparator function. If not specified, insertion order will be used by\n    default."
     },
     "sortReverse": {
      "!type": "bool",
      "!doc": "If `true`, node children will be\n    sorted in reverse (descending) order by default. Otherwise theyll be sorted\n    in ascending order."
     }
    },
    "SortableSortConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, no `sort` events will be\n        fired."
     },
     "sortComparator": {
      "!type": "fn()",
      "!doc": "Custom comparator function to\n        use. If specified, this will become the new comparator function for\n        each node, overwriting any previous comparator function that was set\n        for the node."
     },
     "sortReverse": {
      "!type": "bool",
      "!doc": "If `true`, children will be\n        sorted in reverse (descending) order. Otherwise theyll be sorted in\n        ascending order. This will become each nodes new sort order,\n        overwriting any previous sort order that was set for the node."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the sort operation. Will be\n        passed along to the `sort` event facade."
     }
    },
    "SortableSortNodeConfig": {
     "deep": {
      "!type": "bool",
      "!doc": "If `true`, all of this nodes\n        children (and their children, and so on) will be traversed and\n        re-sorted as well."
     },
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, no `sort` event will be\n        fired."
     },
     "sortComparator": {
      "!type": "fn()",
      "!doc": "Custom comparator function to\n        use. If specified, this will become the nodes new comparator\n        function, overwriting any previous comparator function that was set\n        for the node."
     },
     "sortReverse": {
      "!type": "bool",
      "!doc": "If `true`, children will be\n        sorted in reverse (descending) order. Otherwise theyll be sorted in\n        ascending order. This will become the nodes new sort order,\n        overwriting any previous sort order that was set for the node."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the sort operation. Will be\n        passed along to the `sort` event facade."
     }
    },
    "NodeAppendConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `add` event will\n        be suppressed."
     }
    },
    "NodeEmptyConfig": {
     "destroy": {
      "!type": "bool",
      "!doc": "If `true`, the children will\n        also be destroyed, which makes them available for garbage collection\n        and means they cant be reused."
     },
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, `remove` events will\n        be suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "NodeFindConfig": {
     "depth": {
      "!type": "number",
      "!doc": "Depth limit. If specified, descendants\n        will only be traversed to this depth before backtracking and moving\n        on."
     }
    },
    "NodeInsertConfig": {
     "index": {
      "!type": "number",
      "!doc": "Index at which to insert the child node.\n        If not specified, the node will be appended as the last child of the\n        parent."
     },
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `add` event will\n        be suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be\n        passed along to the event facade of the resulting event. This can be\n        used to distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "NodePrependConfig": {
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `add` event will\n        be suppressed."
     }
    },
    "NodeRemoveConfig": {
     "destroy": {
      "!type": "bool",
      "!doc": "If `true`, this node and all\n        its children will also be destroyed, which makes them available for\n        garbage collection and means they cant be reused."
     },
     "silent": {
      "!type": "bool",
      "!doc": "If `true`, the `remove` event\n        will be suppressed."
     },
     "src": {
      "!type": "string",
      "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
     }
    },
    "NodeTraverseConfig": {
     "depth": {
      "!type": "number",
      "!doc": "Depth limit. If specified, descendants\n        will only be traversed to this depth before backtracking and moving\n        on."
     }
    }
   },
   "TreeAppendNodeConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, the `add` event will\n        be suppressed."
    }
   },
   "TreeClearConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, the `clear` event\n        will be suppressed."
    },
    "src": {
     "!type": "string",
     "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
    }
   },
   "TreeDestroyNodeConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, `remove` events will\n        be suppressed."
    },
    "src": {
     "!type": "string",
     "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting events. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
    }
   },
   "TreeEmptyNodeConfig": {
    "destroy": {
     "!type": "bool",
     "!doc": "If `true`, the children will\n        also be destroyed, which makes them available for garbage collection\n        and means they cant be reused."
    },
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, `remove` events will\n        be suppressed."
    },
    "src": {
     "!type": "string",
     "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting events. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
    }
   },
   "TreeFindNodeConfig": {
    "depth": {
     "!type": "number",
     "!doc": "Depth limit. If specified, descendants\n        will only be traversed to this depth before backtracking and moving\n        on."
    }
   },
   "TreeInsertNodeConfig": {
    "index": {
     "!type": "number",
     "!doc": "Index at which to insert the child node.\n        If not specified, the node will be appended as the last child of the\n        parent."
    },
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, the `add` event will\n        be suppressed."
    },
    "src": {
     "!type": "string",
     "!doc": "Source of the change, to be\n        passed along to the event facade of the resulting event. This can be\n        used to distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
    }
   },
   "TreePrependNodeConfig": {
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, the `add` event will\n        be suppressed."
    }
   },
   "TreeRemoveNodeConfig": {
    "destroy": {
     "!type": "bool",
     "!doc": "If `true`, the node and all its\n        children will also be destroyed, which makes them available for\n        garbage collection and means they cant be reused."
    },
    "silent": {
     "!type": "bool",
     "!doc": "If `true`, the `remove` event\n        will be suppressed."
    },
    "src": {
     "!type": "string",
     "!doc": "Source of the change, to be passed along\n        to the event facade of the resulting event. This can be used to\n        distinguish between changes triggered by a user and changes\n        triggered programmatically, for example."
    }
   },
   "TreeTraverseNodeConfig": {
    "depth": {
     "!type": "number",
     "!doc": "Depth limit. If specified, descendants\n        will only be traversed to this depth before backtracking and moving\n        on."
    }
   },
   "UploaderFlashConfig": {
    "!proto": "config.WidgetConfig",
    "appendNewFiles": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_appendNewFiles",
     "!doc": "A Boolean indicating whether newly selected files should be appended\nto the existing file list, or whether they should replace it."
    },
    "buttonClassNames": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_buttonClassNames",
     "!doc": "The names of CSS classes that correspond to different button states\nof the \"Select Files\" control. These classes are assigned to the\n\"Select Files\" control based on the mouse states reported by the\nFlash player. The keys for the class names are:\n<ul>\n  <li> <strong>`hover`</strong>: the class corresponding to mouse hovering over\n     the \"Select Files\" button.</li>\n  <li> <strong>`active`</strong>: the class corresponding to mouse down state of\n     the \"Select Files\" button.</li>\n  <li> <strong>`disabled`</strong>: the class corresponding to the disabled state\n     of the \"Select Files\" button.</li>\n  <li> <strong>`focus`</strong>: the class corresponding to the focused state of\n     the \"Select Files\" button.</li>\n</ul>"
    },
    "enabled": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_enabled",
     "!doc": "A Boolean indicating whether the uploader is enabled or disabled for user input."
    },
    "errorAction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_errorAction",
     "!doc": "The action  performed when an upload error occurs for a specific file being uploaded.\nThe possible values are:\n<ul>\n  <li> <strong>`UploaderQueue.CONTINUE`</strong>: the error is ignored and the upload process is continued.</li>\n  <li> <strong>`UploaderQueue.STOP`</strong>: the upload process is stopped as soon as any other parallel file\n    uploads are finished.</li>\n  <li> <strong>`UploaderQueue.RESTART_ASAP`</strong>: the file is added back to the front of the queue.</li>\n  <li> <strong>`UploaderQueue.RESTART_AFTER`</strong>: the file is added to the back of the queue.</li>\n</ul>"
    },
    "fileFilters": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_fileFilters",
     "!doc": "An array indicating what fileFilters should be applied to the file\nselection dialog. Each element in the array should be an object with\nthe following key-value pairs:\n{\n  description : String\n        extensions: String of the form &lquot;*.ext1;*.ext2;*.ext3;...&rquot;\n}"
    },
    "fileFilterFunction": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_fileFilterFunction",
     "!doc": "A filtering function that is applied to every file selected by the user.\nThe function receives the `Y.File` object and must return a Boolean value.\nIf a `false` value is returned, the file in question is not added to the\nlist of files to be uploaded.\nUse this function to put limits on file sizes or check the file names for\ncorrect extension, but make sure that a server-side check is also performed,\nsince any client-side restrictions are only advisory and can be circumvented."
    },
    "fileFieldName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_fileFieldName",
     "!doc": "A String specifying what should be the POST field name for the file\ncontent in the upload request."
    },
    "fileList": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_fileList",
     "!doc": "The array of files to be uploaded. All elements in the array\nmust be instances of `Y.FileFlash` and be instantiated with a `fileId`\nretrieved from an instance of the uploader."
    },
    "multipleFiles": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_multipleFiles",
     "!doc": "A Boolean indicating whether multiple file selection is enabled."
    },
    "postVarsPerFile": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_postVarsPerFile",
     "!doc": "An object, keyed by `fileId`, containing sets of key-value pairs\nthat should be passed as POST variables along with each corresponding\nfile. This attribute is only used if no POST variables are specifed\nin the upload method call."
    },
    "selectButtonLabel": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_selectButtonLabel",
     "!doc": "The label for the \"Select Files\" widget. This is the value that replaces the\n`{selectButtonLabel}` token in the `SELECT_FILES_BUTTON` template."
    },
    "selectFilesButton": {
     "!type": "+node.Node|+widget.Widget",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_selectFilesButton",
     "!doc": "The widget that serves as the \"Select Files\" control for the file uploader"
    },
    "simLimit": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_simLimit",
     "!doc": "The number of files that can be uploaded\nsimultaneously if the automatic queue management\nis used. This value can be in the range between 2\nand 5."
    },
    "swfURL": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_swfURL",
     "!doc": "The URL to the SWF file of the flash uploader. A copy local to\nthe server that hosts the page on which the uploader appears is\nrecommended."
    },
    "tabElements": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_tabElements",
     "!doc": "The ids or `Node` references of the DOM elements that precede\nand follow the `Select Files` button in the tab order. Specifying\nthese allows keyboard navigation to and from the Flash player\nlayer of the uploader.\nThe two keys corresponding to the DOM elements are:\n       <ul>\n  <li> `from`: the id or the `Node` reference corresponding to the\n    DOM element that precedes the `Select Files` button in the tab order.</li>\n  <li> `to`: the id or the `Node` reference corresponding to the\n    DOM element that follows the `Select Files` button in the tab order.</li>\n</ul>"
    },
    "uploadURL": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_uploadURL",
     "!doc": "The URL to which file upload requested are POSTed. Only used if a different url is not passed to the upload method call."
    },
    "retryCount": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderFlash.html#attribute_retryCount",
     "!doc": "The number of times to try re-uploading a file that failed to upload before\ncancelling its upload."
    }
   },
   "UploaderHTML5Config": {
    "!proto": "config.WidgetConfig",
    "appendNewFiles": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_appendNewFiles",
     "!doc": "A Boolean indicating whether newly selected files should be appended\nto the existing file list, or whether they should replace it."
    },
    "buttonClassNames": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_buttonClassNames",
     "!doc": "The names of CSS classes that correspond to different button states\nof the \"Select Files\" control. These classes are assigned to the\n\"Select Files\" control based on the configuration of the uploader.\nCurrently, the only class name used is that corresponding to the\n`disabled` state of the uploader. Other button states should be managed\ndirectly via CSS selectors.\n<ul>\n  <li> <strong>`disabled`</strong>: the class corresponding to the disabled state\n     of the \"Select Files\" button.</li>\n</ul>"
    },
    "dragAndDropArea": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_dragAndDropArea",
     "!doc": "The node that serves as the drop target for files."
    },
    "enabled": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_enabled",
     "!doc": "A Boolean indicating whether the uploader is enabled or disabled for user input."
    },
    "errorAction": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_errorAction",
     "!doc": "The action  performed when an upload error occurs for a specific file being uploaded.\nThe possible values are:\n<ul>\n  <li> <strong>`UploaderQueue.CONTINUE`</strong>: the error is ignored and the upload process is continued.</li>\n  <li> <strong>`UploaderQueue.STOP`</strong>: the upload process is stopped as soon as any other parallel file\n    uploads are finished.</li>\n  <li> <strong>`UploaderQueue.RESTART_ASAP`</strong>: the file is added back to the front of the queue.</li>\n  <li> <strong>`UploaderQueue.RESTART_AFTER`</strong>: the file is added to the back of the queue.</li>\n</ul>"
    },
    "fileFilters": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_fileFilters",
     "!doc": "An array indicating what fileFilters should be applied to the file\nselection dialog. Each element in the array should be a string\nindicating the Media (MIME) type for the files that should be supported\nfor selection. The Media type strings should be properly formatted\nor this parameter will be ignored. Examples of valid strings include:\n\"audio/*\", \"video/*\", \"application/pdf\", etc. More information\non valid Media type strings is available here:\nhttp://www.iana.org/assignments/media-types/index.html"
    },
    "fileFilterFunction": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_fileFilterFunction",
     "!doc": "A filtering function that is applied to every file selected by the user.\nThe function receives the `Y.File` object and must return a Boolean value.\nIf a `false` value is returned, the file in question is not added to the\nlist of files to be uploaded.\nUse this function to put limits on file sizes or check the file names for\ncorrect extension, but make sure that a server-side check is also performed,\nsince any client-side restrictions are only advisory and can be circumvented."
    },
    "fileFieldName": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_fileFieldName",
     "!doc": "A String specifying what should be the POST field name for the file\ncontent in the upload request."
    },
    "fileList": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_fileList",
     "!doc": "The array of files to be uploaded. All elements in the array\nmust be instances of `Y.File` and be instantiated with an instance\nof native JavaScript File() class."
    },
    "multipleFiles": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_multipleFiles",
     "!doc": "A Boolean indicating whether multiple file selection is enabled."
    },
    "postVarsPerFile": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_postVarsPerFile",
     "!doc": "An object, keyed by `fileId`, containing sets of key-value pairs\nthat should be passed as POST variables along with each corresponding\nfile. This attribute is only used if no POST variables are specifed\nin the upload method call."
    },
    "selectButtonLabel": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_selectButtonLabel",
     "!doc": "The label for the \"Select Files\" widget. This is the value that replaces the\n`{selectButtonLabel}` token in the `SELECT_FILES_BUTTON` template."
    },
    "selectFilesButton": {
     "!type": "+node.Node|+widget.Widget",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_selectFilesButton",
     "!doc": "The widget that serves as the \"Select Files control for the file uploader"
    },
    "simLimit": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_simLimit",
     "!doc": "The number of files that can be uploaded\nsimultaneously if the automatic queue management\nis used. This value can be in the range between 2\nand 5."
    },
    "uploadURL": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_uploadURL",
     "!doc": "The URL to which file upload requested are POSTed. Only used if a different url is not passed to the upload method call."
    },
    "uploadHeaders": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_uploadHeaders",
     "!doc": "Additional HTTP headers that should be included\nin the upload request."
    },
    "withCredentials": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_withCredentials",
     "!doc": "A Boolean that specifies whether the file should be\nuploaded with the appropriate user credentials for the\ndomain."
    },
    "retryCount": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/UploaderHTML5.html#attribute_retryCount",
     "!doc": "The number of times to try re-uploading a file that failed to upload before\ncancelling its upload."
    }
   },
   "Uploader": {
    "QueueConfig": {
     "!proto": "config.BaseConfig",
     "simUploads": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_simUploads",
      "!doc": "Maximum number of simultaneous uploads; must be in the\nrange between 1 and 5. The value of `2` is default. It\nis recommended that this value does not exceed 3."
     },
     "errorAction": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_errorAction",
      "!doc": "The action to take in case of error. The valid values for this attribute are:\n`Y.Uploader.Queue.CONTINUE` (the upload process should continue on other files,\nignoring the error), `Y.Uploader.Queue.STOP` (the upload process\nshould stop completely), `Y.Uploader.Queue.RESTART_ASAP` (the upload\nshould restart immediately on the errored out file and continue as planned), or\nY.Uploader.Queue.RESTART_AFTER (the upload of the errored out file should restart\nafter all other files have uploaded)"
     },
     "bytesUploaded": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_bytesUploaded",
      "!doc": "The total number of bytes that has been uploaded."
     },
     "bytesTotal": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_bytesTotal",
      "!doc": "The total number of bytes in the queue."
     },
     "fileList": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_fileList",
      "!doc": "The queue file list. This file list should only be modified\nbefore the upload has been started; modifying it after starting\nthe upload has no effect, and `addToQueueTop` or `addToQueueBottom` methods\nshould be used instead."
     },
     "fileFieldName": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_fileFieldName",
      "!doc": "A String specifying what should be the POST field name for the file\ncontent in the upload request."
     },
     "uploadURL": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_uploadURL",
      "!doc": "The URL to POST the file upload requests to."
     },
     "uploadHeaders": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_uploadHeaders",
      "!doc": "Additional HTTP headers that should be included\nin the upload request. Due to Flash Player security\nrestrictions, this attribute is only honored in the\nHTML5 Uploader."
     },
     "withCredentials": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_withCredentials",
      "!doc": "A Boolean that specifies whether the file should be\nuploaded with the appropriate user credentials for the\ndomain. Due to Flash Player security restrictions, this\nattribute is only honored in the HTML5 Uploader."
     },
     "perFileParameters": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_perFileParameters",
      "!doc": "An object, keyed by `fileId`, containing sets of key-value pairs\nthat should be passed as POST variables along with each corresponding\nfile."
     },
     "retryCount": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Uploader.Queue.html#attribute_retryCount",
      "!doc": "The number of times to try re-uploading a file that failed to upload before\ncancelling its upload."
     }
    }
   },
   "WidgetAutohideConfig": {
    "hideOn": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetAutohide.html#attribute_hideOn",
     "!doc": "An array of objects corresponding to the nodes, events, and keycodes to hide the widget on.\nThe implementer can supply an array of objects, with each object having the following properties:\n<p>eventName: (string, required): The eventName to listen to.</p>\n<p>node: (Y.Node, optional): The Y.Node that will fire the event (defaults to the boundingBox of the widget)</p>\n<p>keyCode: (string, optional): If listening for key events, specify the keyCode</p>\n<p>By default, this attribute consists of one object which will cause the widget to hide if the\nescape key is pressed.</p>"
    }
   },
   "WidgetButtonsConfig": {
    "buttons": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetButtons.html#attribute_buttons",
     "!doc": "Collection containing a widgets buttons.\n\nThe collection is an Object which contains an Array of `Y.Node`s for every\n`WidgetStdMod` section (header, body, footer) which has one or more buttons.\nAll button nodes have the `Y.Plugin.Button` plugin applied.\n\nThis attribute is very flexible in the values it will accept. `buttons` can\nbe specified as a single Array, or an Object of Arrays keyed to a particular\nsection.\n\nAll specified values will be normalized to this type of structure:\n\n    {\n        header: [...],\n        footer: [...]\n    }\n\nA button can be specified as a `Y.Node`, config Object, or String name for a\npredefined button on the `BUTTONS` prototype property. When a config Object\nis provided, it will be merged with any defaults provided by a button with\nthe same `name` defined on the `BUTTONS` property.\n\nSee `addButton()` for the detailed list of configuration properties.\n\nFor convenience, a widgets buttons will always persist and remain rendered\nafter header/body/footer content updates. Buttons should be removed by\nupdating this attribute or using the `removeButton()` method."
    },
    "defaultButton": {
     "!type": "+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetButtons.html#attribute_defaultButton",
     "!doc": "The current default button as configured through this widgets `buttons`.\n\nA button can be configured as the default button in the following ways:\n\n  * As a config Object with an `isDefault` property:\n    `{label: Okay, isDefault: true}`.\n\n  * As a Node with a `data-default` attribute:\n    `<button data-default=\"true\">Okay</button>`.\n\nThis attribute is **read-only**; anytime there are changes to this widgets\n`buttons`, the `defaultButton` will be updated if needed.\n\n**Note:** If two or more buttons are configured to be the default button,\nthe last one wins."
    }
   },
   "WidgetChildConfig": {
    "selected": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetChild.html#attribute_selected",
     "!doc": "Number indicating if the Widget is selected.  Possible\nvalues are:\n<dl>\n<dt>0</dt> <dd>(Default) Not selected</dd>\n<dt>1</dt> <dd>Fully selected</dd>\n<dt>2</dt> <dd>Partially selected</dd>\n</dl>"
    },
    "index": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetChild.html#attribute_index",
     "!doc": "Number representing the Widgets ordinal position in its\nparent Widget."
    },
    "parent": {
     "!type": "+widget.Widget",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetChild.html#attribute_parent",
     "!doc": "Retrieves the parent of the Widget in the object hierarchy."
    },
    "depth": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetChild.html#attribute_depth",
     "!doc": "Number representing the depth of this Widget relative to\nthe root Widget in the object heirarchy."
    },
    "root": {
     "!type": "+widget.Widget",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetChild.html#attribute_root",
     "!doc": "Returns the root Widget in the object hierarchy.  If the\nROOT_TYPE property is set, the search for the root Widget will be\nconstrained to parent Widgets of the specified type."
    }
   },
   "WidgetModalityConfig": {
    "maskNode": {
     "!type": "+Y.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetModality.html#attribute_maskNode",
     "!doc": "Returns a Y.Node instance of the node being used as the mask."
    },
    "modal": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetModality.html#attribute_modal",
     "!doc": "Whether the widget should be modal or not."
    },
    "focusOn": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetModality.html#attribute_focusOn",
     "!doc": "An array of objects corresponding to the nodes and events that will trigger a re-focus back on the widget.\nThe implementer can supply an array of objects, with each object having the following properties:\n<p>eventName: (string, required): The eventName to listen to.</p>\n<p>node: (Y.Node, optional): The Y.Node that will fire the event (defaults to the boundingBox of the widget)</p>\n<p>By default, this attribute consists of two objects which will cause the widget to re-focus if anything\noutside the widget is clicked on or focussed upon.</p>"
    }
   },
   "WidgetParentConfig": {
    "defaultChildType": {
     "!type": "string|+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetParent.html#attribute_defaultChildType",
     "!doc": "String representing the default type of the children\nmanaged by this Widget.  Can also supply default type as a constructor\nreference."
    },
    "activeDescendant": {
     "!type": "+widget.Widget",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetParent.html#attribute_activeDescendant",
     "!doc": "Returns the Widgets currently focused descendant Widget."
    },
    "multiple": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetParent.html#attribute_multiple",
     "!doc": "Boolean indicating if multiple children can be selected at\nonce.  Whether or not multiple selection is enabled is always delegated\nto the value of the <code>multiple</code> attribute of the root widget\nin the object hierarchy."
    },
    "selection": {
     "!type": "+collection.ArrayList|+widget.Widget",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetParent.html#attribute_selection",
     "!doc": "Returns the currently selected child Widget.  If the\n<code>mulitple</code> attribte is set to <code>true</code> will\nreturn an Y.ArrayList instance containing the currently selected\nchildren.  If no children are selected, will return null."
    }
   },
   "WidgetPositionAlignConfig": {
    "align": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetPositionAlign.html#attribute_align",
     "!doc": "The alignment configuration for this widget.\n\nThe `align` attribute is used to align a reference point on the widget, with\nthe reference point on another `Node`, or the viewport. The object which\n`align` expects has the following properties:\n\n  * __`node`__: The `Node` to which the widget is to be aligned. If set to\n    `null`, or not provided, the widget is aligned to the viewport.\n\n  * __`points`__: A two element Array, defining the two points on the widget\n    and `Node`/viewport which are to be aligned. The first element is the\n    point on the widget, and the second element is the point on the\n    `Node`/viewport. Supported alignment points are defined as static\n    properties on `WidgetPositionAlign`."
    },
    "centered": {
     "!type": "bool|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetPositionAlign.html#attribute_centered",
     "!doc": "A convenience Attribute, which can be used as a shortcut for the `align`\nAttribute.\n\nIf set to `true`, the widget is centered in the viewport. If set to a `Node`\nreference or valid selector String, the widget will be centered within the\n`Node`. If set to `false`, no center positioning is applied."
    },
    "alignOn": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetPositionAlign.html#attribute_alignOn",
     "!doc": "An Array of Objects corresponding to the `Node`s and events that will cause\nthe alignment of this widget to be synced to the DOM.\n\nThe `alignOn` Attribute is expected to be an Array of Objects with the\nfollowing properties:\n\n  * __`eventName`__: The String event name to listen for.\n\n  * __`node`__: The optional `Node` that will fire the event, it can be a\n    `Node` reference or a selector String. This will default to the widgets\n    `boundingBox`."
    }
   },
   "WidgetPositionConstrainConfig": {
    "constrain": {
     "!type": "bool|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetPositionConstrain.html#attribute_constrain",
     "!doc": "The node to constrain the widgets bounding box to, when setting xy. Can also be\nset to true, to constrain to the viewport."
    },
    "preventOverlap": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetPositionConstrain.html#attribute_preventOverlap",
     "!doc": "If set to true, and WidgetPositionAlign is also added to the Widget,\nconstrained positioning will attempt to prevent the widgets bounding box from overlapping\nthe element to which it has been aligned, by flipping the orientation of the alignment\nfor corner based alignments"
    }
   },
   "WidgetPositionConfig": {
    "x": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetPosition.html#attribute_x",
     "!doc": "Page X co-ordinate for the widget. This attribute acts as a facade for the\nxy attribute. Changes in position can be monitored by listening for xyChange events."
    },
    "y": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetPosition.html#attribute_y",
     "!doc": "Page Y co-ordinate for the widget. This attribute acts as a facade for the\nxy attribute. Changes in position can be monitored by listening for xyChange events."
    },
    "xy": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetPosition.html#attribute_xy",
     "!doc": "Page XY co-ordinate pair for the widget."
    }
   },
   "WidgetStackConfig": {
    "shim": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetStack.html#attribute_shim",
     "!doc": "Boolean flag to indicate whether or not a shim should be added to the Widgets\nboundingBox, to protect it from select box bleedthrough."
    },
    "zIndex": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetStack.html#attribute_zIndex",
     "!doc": "The z-index to apply to the Widgets boundingBox. Non-numerical values for\nzIndex will be converted to 0"
    }
   },
   "WidgetStdModConfig": {
    "headerContent": {
     "!type": "+HTML",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetStdMod.html#attribute_headerContent",
     "!doc": "The content to be added to the header section. This will replace any existing content\nin the header. If you want to append, or insert new content, use the <a href=\"#method_setStdModContent\">setStdModContent</a> method."
    },
    "footerContent": {
     "!type": "+HTML",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetStdMod.html#attribute_footerContent",
     "!doc": "The content to be added to the footer section. This will replace any existing content\nin the footer. If you want to append, or insert new content, use the <a href=\"#method_setStdModContent\">setStdModContent</a> method."
    },
    "bodyContent": {
     "!type": "+HTML",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetStdMod.html#attribute_bodyContent",
     "!doc": "The content to be added to the body section. This will replace any existing content\nin the body. If you want to append, or insert new content, use the <a href=\"#method_setStdModContent\">setStdModContent</a> method."
    },
    "fillHeight": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/WidgetStdMod.html#attribute_fillHeight",
     "!doc": "The section (WidgetStdMod.HEADER, WidgetStdMod.BODY or WidgetStdMod.FOOTER) which should be resized to fill the height of the standard module, when a\nheight is set on the Widget. If a height is not set on the widget, then all sections are sized based on\ntheir content."
    }
   },
   "WidgetConfig": {
    "!proto": "config.BaseConfig",
    "id": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_id",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "rendered": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_rendered",
     "!doc": "Flag indicating whether or not this Widget\nhas been through the render lifecycle phase.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "boundingBox": {
     "!type": "string|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_boundingBox",
     "!doc": "The outermost DOM node for the Widget, used for sizing and positioning\nof a Widget as well as a containing element for any decorator elements used\nfor skinning.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "contentBox": {
     "!type": "string|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_contentBox",
     "!doc": "A DOM node that is a direct descendant of a Widgets bounding box that\nhouses its content.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "tabIndex": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_tabIndex",
     "!doc": "Number (between -32767 to 32767) indicating the widgets\nposition in the default tab flow.  The value is used to set the\n\"tabIndex\" attribute on the widgets bounding box.  Negative values allow\nthe widget to receive DOM focus programmatically (by calling the focus\nmethod), while being removed from the default tab flow.  A value of\nnull removes the \"tabIndex\" attribute from the widgets bounding box.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "focused": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_focused",
     "!doc": "Boolean indicating if the Widget, or one of its descendants,\nhas focus.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "disabled": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_disabled",
     "!doc": "Boolean indicating if the Widget should be disabled. The disabled implementation\nis left to the specific classes extending widget.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "visible": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_visible",
     "!doc": "Boolean indicating whether or not the Widget is visible.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "height": {
     "!type": "string|number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_height",
     "!doc": "String with units, or number, representing the height of the Widget. If a number is provided,\nthe default unit, defined by the Widgets DEF_UNIT, property is used.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "width": {
     "!type": "string|number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_width",
     "!doc": "String with units, or number, representing the width of the Widget. If a number is provided,\nthe default unit, defined by the Widgets DEF_UNIT, property is used.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "strings": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_strings",
     "!doc": "Collection of strings used to label elements of the Widgets UI.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "render": {
     "!type": "bool|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_render",
     "!doc": "Whether or not to render the widget automatically after init, and optionally, to which parent node.",
     "!data": {
      "submodule": "widget-base"
     }
    },
    "srcNode": {
     "!type": "string|+node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Widget.html#attribute_srcNode",
     "!doc": "The DOM node to parse for configuration values, passed to the Widgets HTML_PARSER definition",
     "!data": {
      "submodule": "widget-htmlparser"
     }
    }
   },
   "FeaturesAddConfig": {
    "name": {
     "!type": "string",
     "!doc": "The name of the test"
    },
    "test": {
     "!type": "fn()",
     "!doc": "The test function to execute, the only argument to the function is the `Y` instance"
    },
    "trigger": {
     "!type": "string",
     "!doc": "The module that triggers this test."
    }
   },
   "YUIAddConfig": {
    "requires": {
     "!type": "[?]",
     "!doc": "Array of other module names that must be\n        attached before this module can be attached."
    },
    "optional": {
     "!type": "[?]",
     "!doc": "Array of optional module names that should\n        be attached before this module is attached if theyve already been\n        loaded. If the `loadOptional` YUI option is `true`, optional modules\n        that have not yet been loaded will be loaded just as if they were hard\n        requirements."
    },
    "use": {
     "!type": "[?]",
     "!doc": "Array of module names that are included within\n        or otherwise provided by this module, and which should be attached\n        automatically when this module is attached. This makes it possible to\n        create \"virtual rollup\" modules that simply attach a collection of other\n        modules or submodules."
    }
   }
  },
  "anim": {
   "!data": {
    "submodules": {
     "anim-color": {},
     "anim-curve": {},
     "anim-easing": {},
     "anim-node-plugin": {},
     "anim-scroll": {},
     "anim-shape-transform": {},
     "anim-shape": {},
     "anim-xy": {},
     "anim-base": {}
    }
   },
   "Anim": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "run": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#method_run",
      "!doc": "Starts or resumes an animation.",
      "!data": {
       "submodule": "anim-base"
      }
     },
     "pause": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#method_pause",
      "!doc": "Pauses the animation and\nfreezes it in its current state and time.\nCalling run() will continue where it left off.",
      "!data": {
       "submodule": "anim-base"
      }
     },
     "stop": {
      "!type": "fn(finish: bool) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#method_stop",
      "!doc": "Stops the animation and resets its time.",
      "!data": {
       "submodule": "anim-base"
      }
     }
    },
    "getBezier": {
     "!type": "fn(points: [number], t: number) -> [number]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#method_getBezier",
     "!doc": "Get the current position of the animated element based on t.\nEach point is an array of \"x\" and \"y\" values (0 = x, 1 = y)\nAt least 2 points are required (start and end).\nFirst point is start. Last point is end.\nAdditional control points are optional.",
     "!data": {
      "submodule": "anim-curve"
     }
    },
    "RE_DEFAULT_UNIT": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#property_RE_DEFAULT_UNIT",
     "!doc": "Regex of properties that should use the default unit.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "DEFAULT_UNIT": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#property_DEFAULT_UNIT",
     "!doc": "The default unit to use with properties that pass the RE_DEFAULT_UNIT test.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "intervalTime": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#property_intervalTime",
     "!doc": "Time in milliseconds passed to setInterval for frame processing",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "behaviors": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#property_behaviors",
     "!doc": "Bucket for custom getters and setters",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "DEFAULT_SETTER": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#property_DEFAULT_SETTER",
     "!doc": "The default setter to use when setting object properties.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "DEFAULT_GETTER": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#property_DEFAULT_GETTER",
     "!doc": "The default getter to use when getting object properties.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "run": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#method_run",
     "!doc": "Runs all animation instances.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "pause": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#method_pause",
     "!doc": "Pauses all animation instances.",
     "!data": {
      "submodule": "anim-base"
     }
    },
    "stop": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Anim.html#method_stop",
     "!doc": "Stops all animation instances.",
     "!data": {
      "submodule": "anim-base"
     }
    }
   },
   "Easing": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html",
    "prototype": {
     "easeNone": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_easeNone",
      "!doc": "Uniform speed between points.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "easeIn": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_easeIn",
      "!doc": "Begins slowly and accelerates towards end. (quadratic)",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "easeOut": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_easeOut",
      "!doc": "Begins quickly and decelerates towards end.  (quadratic)",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "easeBoth": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_easeBoth",
      "!doc": "Begins slowly and decelerates towards end. (quadratic)",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "easeInStrong": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_easeInStrong",
      "!doc": "Begins slowly and accelerates towards end. (quartic)",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "easeOutStrong": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_easeOutStrong",
      "!doc": "Begins quickly and decelerates towards end.  (quartic)",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "easeBothStrong": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_easeBothStrong",
      "!doc": "Begins slowly and decelerates towards end. (quartic)",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "elasticIn": {
      "!type": "fn(t: number, b: number, c: number, d: number, a: number, p: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_elasticIn",
      "!doc": "Snap in elastic effect.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "elasticOut": {
      "!type": "fn(t: number, b: number, c: number, d: number, a: number, p: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_elasticOut",
      "!doc": "Snap out elastic effect.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "elasticBoth": {
      "!type": "fn(t: number, b: number, c: number, d: number, a: number, p: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_elasticBoth",
      "!doc": "Snap both elastic effect.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "backIn": {
      "!type": "fn(t: number, b: number, c: number, d: number, s: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_backIn",
      "!doc": "Backtracks slightly, then reverses direction and moves to end.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "backOut": {
      "!type": "fn(t: number, b: number, c: number, d: number, s: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_backOut",
      "!doc": "Overshoots end, then reverses and comes back to end.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "backBoth": {
      "!type": "fn(t: number, b: number, c: number, d: number, s: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_backBoth",
      "!doc": "Backtracks slightly, then reverses direction, overshoots end,\nthen reverses and comes back to end.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "bounceIn": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_bounceIn",
      "!doc": "Bounce off of start.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "bounceOut": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_bounceOut",
      "!doc": "Bounces off end.",
      "!data": {
       "submodule": "anim-easing"
      }
     },
     "bounceBoth": {
      "!type": "fn(t: number, b: number, c: number, d: number) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Easing.html#method_bounceBoth",
      "!doc": "Bounces off start and end.",
      "!data": {
       "submodule": "anim-easing"
      }
     }
    }
   }
  },
  "app": {
   "!data": {
    "submodules": {
     "app-content": {},
     "app-transitions-native": {},
     "app-transitions": {},
     "model-sync-rest": {},
     "view-node-map": {},
     "app-base": {},
     "lazy-model-list": {},
     "model-list": {},
     "model": {},
     "router": {},
     "view": {}
    }
   },
   "App": {
    "!type": "fn(config?: +config.AppConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/App.html",
    "!data": {
     "extends": [
      "app.App.Content",
      "app.App.Transitions",
      "pjax.PjaxContent"
     ]
    },
    "prototype": {
     "!proto": "app.App.Base.prototype"
    },
    "Content": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Content.html",
     "!data": {
      "extends": [
       "pjax.PjaxContent"
      ]
     },
     "route": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Content.html#property_route",
      "!doc": "A stack of middleware which forms a pjax-style content route.\n\nThis route will load the rendered HTML content from the server, then create and\nshow a new view using those contents.",
      "!data": {
       "submodule": "app-content"
      }
     },
     "prototype": {
      "showContent": {
       "!type": "fn(content: +HTMLElement|+node.Node|string, options?: +config.App.ContentShowContentConfig, callback?: fn(view: +app.View)) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Content.html#method_showContent",
       "!doc": "Sets this apps `activeView` attribute using the specified `content`.\n\nThis provides an easy way to view-ify HTML content which should be shown as\nthis apps active/visible view. This method will determine the appropriate\nview `container` node based on the specified `content`. By default, a new\n`Y.View` instance will be created unless `options.view` is specified.\n\nUnder the hood, this method calls the `showView()` method, so refer to its\ndocs for more information.",
       "!data": {
        "submodule": "app-content"
       }
      }
     }
    },
    "Transitions": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Transitions.html",
     "!data": {
      "extends": [
       "app.App.TransitionsNative"
      ]
     },
     "FX": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Transitions.html#property_FX",
      "!doc": "Collect of transitions -> fx.\n\nA transition (e.g. \"fade\") is a simple name given to a configuration of fx to\napply, consisting of `viewIn` and `viewOut` properties whos values are names of\nfx registered on `Y.Transition.fx`.\n\nBy default transitions: `fade`, `slideLeft`, and `slideRight` have fx defined.",
      "!data": {
       "submodule": "app-transitions"
      }
     },
     "prototype": {
      "transitions": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Transitions.html#property_transitions",
       "!doc": "Default transitions to use when the `activeView` changes.\n\nThe following are types of changes for which transitions can be defined that\ncorrespond to the relationship between the new and previous `activeView`:\n\n  * `navigate`: The default transition to use when changing the `activeView`\n    of the application.\n\n  * `toChild`: The transition to use when the new `activeView` is configured\n    as a child of the previously active view via its `parent` property as\n    defined in this apps `views`.\n\n  * `toParent`: The transition to use when the new `activeView` is\n    configured as the `parent` of the previously active view as defined in\n    this apps `views`.\n\n**Note:** Transitions are an opt-in feature and will only be used in\nbrowsers which support native CSS3 transitions.",
       "!data": {
        "submodule": "app-transitions"
       }
      },
      "showView": {
       "!type": "fn(view: string|+app.View, config?: +config.App.TransitionsShowViewConfig, options?: +config.App.TransitionsShowViewConfig, callback?: fn(view: +app.View)) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Transitions.html#method_showView",
       "!doc": "Sets which view is active/visible for the application. This will set the\napps `activeView` attribute to the specified `view`.\n\nThe `view` will be \"attached\" to this app, meaning it will be both rendered\ninto this apps `viewContainer` node and all of its events will bubble to\nthe app. The previous `activeView` will be \"detached\" from this app.\n\nWhen a string-name is provided for a view which has been registered on this\napps `views` object, the referenced metadata will be used and the\n`activeView` will be set to either a preserved view instance, or a new\ninstance of the registered view will be created using the specified `config`\nobject passed-into this method.\n\nA callback function can be specified as either the third or fourth argument,\nand this function will be called after the new `view` becomes the\n`activeView`, is rendered to the `viewContainer`, and is ready to use.",
       "!data": {
        "submodule": "app-transitions"
       }
      }
     }
    },
    "Base": {
     "!type": "fn(config?: +config.App.BaseConfig)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html",
     "!data": {
      "extends": [
       "app.View",
       "app.Router",
       "pjax.PjaxBase"
      ]
     },
     "prototype": {
      "!proto": "base.Base.prototype",
      "views": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#property_views",
       "!doc": "Hash of view-name to metadata used to declaratively describe an\napplications views and their relationship with the app and its other views.\n\nThe view metadata is composed of Objects keyed to a view-name that can have\nany or all of the following properties:\n\n  * `type`: Function or a string representing the view constructor to use to\n    create view instances. If a string is used, the constructor function is\n    assumed to be on the `Y` object; e.g. `\"SomeView\"` -> `Y.SomeView`.\n\n  * `preserve`: Boolean for whether the view instance should be retained. By\n    default, the view instance will be destroyed when it is no longer the\n    `activeView`. If `true` the view instance will simply be `removed()`\n    from the DOM when it is no longer active. This is useful when the view\n    is frequently used and may be expensive to re-create.\n\n  * `parent`: String to another named view in this hash that represents the\n    parent view within the applications view hierarchy; e.g. a `\"photo\"`\n    view could have `\"album\"` has its `parent` view. This parent/child\n    relationship is a useful cue for things like transitions.\n\n  * `instance`: Used internally to manage the current instance of this named\n    view. This can be used if your view instance is created up-front, or if\n    you would rather manage the View lifecycle, but you probably should just\n    let this be handled for you.\n\nIf `views` are specified at instantiation time, the metadata in the `views`\nObject here will be used as defaults when creating the instances `views`.\n\nEvery `Y.App` instance gets its own copy of a `views` object so this Object\non the prototype will not be polluted.",
       "!data": {
        "submodule": "app-base"
       }
      },
      "createView": {
       "!type": "fn(name: string, config?: +config.App.BaseCreateViewConfig) -> +app.View",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#method_createView",
       "!doc": "Creates and returns a new view instance using the provided `name` to look up\nthe view info metadata defined in the `views` object. The passed-in `config`\nobject is passed to the view constructor function.\n\nThis function also maps a view instance back to its view info metadata.",
       "!data": {
        "submodule": "app-base"
       }
      },
      "getViewInfo": {
       "!type": "fn(view: +app.View|string) -> +Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#method_getViewInfo",
       "!doc": "Returns the metadata associated with a view instance or view name defined on\nthe `views` object.",
       "!data": {
        "submodule": "app-base"
       }
      },
      "navigate": {
       "!type": "fn(url: string, options?: +config.App.BaseNavigateConfig)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#method_navigate",
       "!doc": "Navigates to the specified URL if there is a route handler that matches. In\nbrowsers capable of using HTML5 history or when `serverRouting` is falsy,\nthe navigation will be enhanced by firing the `navigate` event and having\nthe app handle the \"request\". When `serverRouting` is `true`, non-HTML5\nbrowsers will navigate to the new URL via a full page reload.\n\nWhen there is a route handler for the specified URL and it is being\nnavigated to, this method will return `true`, otherwise it will return\n`false`.\n\n**Note:** The specified URL _must_ be of the same origin as the current URL,\notherwise an error will be logged and navigation will not occur. This is\nintended as both a security constraint and a purposely imposed limitation as\nit does not make sense to tell the app to navigate to a URL on a\ndifferent scheme, host, or port.",
       "!data": {
        "submodule": "app-base"
       }
      },
      "render": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#method_render",
       "!doc": "Renders this application by appending the `viewContainer` node to the\n`container` node if it isnt already a child of the container, and the\n`activeView` will be appended the view container, if it isnt already.\n\nYou should call this method at least once, usually after the initialization\nof your app instance so the proper DOM structure is setup and optionally\nappend the container to the DOM if its not there already.\n\nYou may override this method to customize the apps rendering, but you\nshould expect that the `viewContainer`s contents will be modified by the\napp for the purpose of rendering the `activeView` when it changes.",
       "!data": {
        "submodule": "app-base"
       }
      },
      "showView": {
       "!type": "fn(view: string|+app.View, config?: +config.App.BaseShowViewConfig, options?: +config.App.BaseShowViewConfig, callback?: fn(view: +app.View)) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/App.Base.html#method_showView",
       "!doc": "Sets which view is active/visible for the application. This will set the\napps `activeView` attribute to the specified `view`.\n\nThe `view` will be \"attached\" to this app, meaning it will be both rendered\ninto this apps `viewContainer` node and all of its events will bubble to\nthe app. The previous `activeView` will be \"detached\" from this app.\n\nWhen a string-name is provided for a view which has been registered on this\napps `views` object, the referenced metadata will be used and the\n`activeView` will be set to either a preserved view instance, or a new\ninstance of the registered view will be created using the specified `config`\nobject passed-into this method.\n\nA callback function can be specified as either the third or fourth argument,\nand this function will be called after the new `view` becomes the\n`activeView`, is rendered to the `viewContainer`, and is ready to use.",
       "!data": {
        "submodule": "app-base"
       }
      }
     }
    },
    "CLASS_NAMES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/App.html#property_CLASS_NAMES",
     "!doc": "CSS classes used by `Y.App`.",
     "!data": {
      "submodule": "app-base"
     }
    },
    "serverRouting": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/App.html#property_serverRouting",
     "!doc": "Default `serverRouting` attribute value for all apps.",
     "!data": {
      "submodule": "app-base"
     }
    }
   },
   "ModelSync": {
    "Local": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.Local.html",
     "prototype": {
      "root": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.Local.html#property_root",
       "!doc": "Root used as the key inside of localStorage and/or the in-memory store."
      },
      "storage": {
       "!type": "+Storage",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.Local.html#property_storage",
       "!doc": "Shortcut for access to localStorage."
      },
      "sync": {
       "!type": "fn(action: string, options?: +Object, callback?: fn(err: +Error, response?: ?))",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.Local.html#method_sync",
       "!doc": "Creates a synchronization layer with the localStorage API, if available.\nOtherwise, falls back to a in-memory data store.\n\nThis method is called internally by load(), save(), and destroy()."
      }
     }
    },
    "REST": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html",
     "CSRF_TOKEN": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#property_CSRF_TOKEN",
      "!doc": "A request authenticity token to validate HTTP requests made by this extension\nwith the server when the request results in changing persistent state. This\nallows you to protect your server from Cross-Site Request Forgery attacks.\n\nA CSRF token provided by the server can be embedded in the HTML document and\nassigned to `YUI.Env.CSRF_TOKEN` like this:\n\n    <script>\n        YUI.Env.CSRF_TOKEN = {{session.authenticityToken}};\n    </script>\n\nThe above should come after YUI seed file so that `YUI.Env` will be defined.\n\n**Note:** This can be overridden on a per-request basis. See `sync()` method.\n\nWhen a value for the CSRF token is provided, either statically or via `options`\npassed to the `save()` and `destroy()` methods, the applicable HTTP requests\nwill have a `X-CSRF-Token` header added with the token value.",
      "!data": {
       "submodule": "model-sync-rest"
      }
     },
     "EMULATE_HTTP": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#property_EMULATE_HTTP",
      "!doc": "Static flag to use the HTTP POST method instead of PUT or DELETE.\n\nIf the server-side HTTP framework isnt RESTful, setting this flag to `true`\nwill cause all PUT and DELETE requests to instead use the POST HTTP method, and\nadd a `X-HTTP-Method-Override` HTTP header with the value of the method type\nwhich was overridden.",
      "!data": {
       "submodule": "model-sync-rest"
      }
     },
     "HTTP_HEADERS": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#property_HTTP_HEADERS",
      "!doc": "Default headers used with all XHRs.\n\nBy default the `Accept` and `Content-Type` headers are set to\n\"application/json\", this signals to the HTTP server to process the request\nbodies as JSON and send JSON responses. If youre sending and receiving content\nother than JSON, you can override these headers and the `parse()` and\n`serialize()` methods.\n\n**Note:** These headers will be merged with any request-specific headers, and\nthe request-specific headers will take precedence.",
      "!data": {
       "submodule": "model-sync-rest"
      }
     },
     "HTTP_METHODS": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#property_HTTP_METHODS",
      "!doc": "Static mapping of RESTful HTTP methods corresponding to CRUD actions.",
      "!data": {
       "submodule": "model-sync-rest"
      }
     },
     "HTTP_TIMEOUT": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#property_HTTP_TIMEOUT",
      "!doc": "The number of milliseconds before the XHRs will timeout/abort. This defaults to\n30 seconds.\n\n**Note:** This can be overridden on a per-request basis. See `sync()` method.",
      "!data": {
       "submodule": "model-sync-rest"
      }
     },
     "prototype": {
      "root": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#property_root",
       "!doc": "A string which represents the root or collection part of the URL which\nrelates to a Model or ModelList. Usually this value should be same for all\ninstances of a specific Model/ModelList subclass.\n\nWhen sub-classing `Y.Model`, usually youll only need to override this\nproperty, which lets the URLs for the XHRs be generated by convention. If\nthe `root` string ends with a trailing-slash, XHR URLs will also end with a\n\"/\", and if the `root` does not end with a slash, neither will the XHR URLs.",
       "!data": {
        "submodule": "model-sync-rest"
       }
      },
      "url": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#property_url",
       "!doc": "A string which specifies the URL to use when making XHRs, if not value is\nprovided, the URLs used to make XHRs will be generated by convention.\n\nWhile a `url` can be provided for each Model/ModelList instance, usually\nyoull want to either rely on the default convention or provide a tokenized\nstring on the prototype which can be used for all instances.\n\nWhen sub-classing `Y.Model`, you will probably be able to rely on the\ndefault convention of generating URLs in conjunction with the `root`\nproperty and whether the model is new or not (i.e. has an `id`). If the\n`root` property ends with a trailing-slash, the generated URL for the\nspecific model will also end with a trailing-slash.",
       "!data": {
        "submodule": "model-sync-rest"
       }
      },
      "getURL": {
       "!type": "fn(action?: string, options?: +Object) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#method_getURL",
       "!doc": "Returns the URL for this model or model list for the given `action` and\n`options`, if specified.\n\nThis method correctly handles the variations of `root` and `url` values and\nis called by the `sync()` method to get the URLs used to make the XHRs.\n\nYou can override this method if you need to provide a specific\nimplementation for how the URLs of your Model and ModelList subclasses need\nto be generated.",
       "!data": {
        "submodule": "model-sync-rest"
       }
      },
      "parseIOResponse": {
       "!type": "fn(response: +Object) -> ?",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#method_parseIOResponse",
       "!doc": "Called to parse the response object returned from `Y.io()`. This method\nreceives the full response object and is expected to \"prep\" a response which\nis suitable to pass to the `parse()` method.\n\nBy default the response body is returned (`responseText`), because it\nusually represents the entire entity of this model on the server.\n\nIf you need to parse data out of the responses headers you should do so by\noverriding this method. If youd like the entire response object from the\nXHR to be passed to your `parse()` method, you can simply assign this\nproperty to `false`.",
       "!data": {
        "submodule": "model-sync-rest"
       }
      },
      "serialize": {
       "!type": "fn(action?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#method_serialize",
       "!doc": "Serializes `this` model to be used as the HTTP request entity body.\n\nBy default this model will be serialized to a JSON string via its `toJSON()`\nmethod.\n\nYou can override this method when the HTTP server expects a different\nrepresentation of this models data that is different from the default JSON\nserialization. If youre sending and receive content other than JSON, be\nsure change the `Accept` and `Content-Type` `HTTP_HEADERS` as well.\n\n**Note:** A models `toJSON()` method can also be overridden. If you only\nneed to modify which attributes are serialized to JSON, thats a better\nplace to start.",
       "!data": {
        "submodule": "model-sync-rest"
       }
      },
      "sync": {
       "!type": "fn(action: string, options?: +config.ModelSync.RESTSyncConfig, callback?: fn(err: +Error, response?: ?))",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelSync.REST.html#method_sync",
       "!doc": "Communicates with a RESTful HTTP server by sending and receiving data via\nXHRs. This method is called internally by load(), save(), and destroy().\n\nThe URL used for each XHR will be retrieved by calling the `getURL()` method\nand passing it the specified `action` and `options`.\n\nThis method relies heavily on standard RESTful HTTP conventions",
       "!data": {
        "submodule": "model-sync-rest"
       }
      }
     }
    }
   },
   "View": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "containerTemplate": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#property_containerTemplate",
      "!doc": "Template for this views container.",
      "!data": {
       "submodule": "view"
      }
     },
     "events": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#property_events",
      "!doc": "Hash of CSS selectors mapped to events to delegate to elements matching\nthose selectors.\n\nCSS selectors are relative to the `container` element. Events are attached\nto the container, and delegation is used so that subscribers are only\nnotified of events that occur on elements inside the container that match\nthe specified selectors. This allows the containers contents to be re-\nrendered as needed without losing event subscriptions.\n\nEvent handlers can be specified either as functions or as strings that map\nto function names on this view instance or its prototype.\n\nThe `this` object in event handlers will refer to this view instance. If\nyoud prefer `this` to be something else, use `Y.bind()` to bind a custom\n`this` object.",
      "!data": {
       "submodule": "view"
      }
     },
     "template": {
      "!type": "+mixed",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#property_template",
      "!doc": "Template for this views contents.\n\nThis is a convenience property that has no default behavior of its own.\nIts only provided as a convention to allow you to store whatever you\nconsider to be a template, whether thats an HTML string, a `Y.Node`\ninstance, a Mustache template, or anything else your little heart\ndesires.\n\nHow this template gets used is entirely up to you and your custom\n`render()` method.",
      "!data": {
       "submodule": "view"
      }
     },
     "destroy": {
      "!type": "fn(options?: +config.ViewDestroyConfig) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#method_destroy",
      "!doc": "Destroys this View, detaching any DOM events and optionally also destroying\nits container node.\n\nBy default, the container node will not be destroyed. Pass an _options_\nobject with a truthy `remove` property to destroy the container as well.",
      "!data": {
       "submodule": "view"
      }
     },
     "attachEvents": {
      "!type": "fn(events?: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#method_attachEvents",
      "!doc": "Attaches delegated event handlers to this views container element. This\nmethod is called internally to subscribe to events configured in the\n`events` attribute when the view is initialized.\n\nYou may override this method to customize the event attaching logic.",
      "!data": {
       "submodule": "view"
      }
     },
     "create": {
      "!type": "fn(container?: +HTMLElement|+node.Node|string) -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#method_create",
      "!doc": "Creates and returns a container node for this view.\n\nBy default, the container is created from the HTML template specified in the\n`containerTemplate` property, and is _not_ added to the DOM automatically.\n\nYou may override this method to customize how the container node is created\n(such as by rendering it from a custom template format). Your method must\nreturn a `Y.Node` instance.",
      "!data": {
       "submodule": "view"
      }
     },
     "detachEvents": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#method_detachEvents",
      "!doc": "Detaches DOM events that have previously been attached to the container by\n`attachEvents()`.",
      "!data": {
       "submodule": "view"
      }
     },
     "remove": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#method_remove",
      "!doc": "Removes this views container element from the DOM (if its in the DOM),\nbut doesnt destroy it or any event listeners attached to it.",
      "!data": {
       "submodule": "view"
      }
     },
     "render": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.html#method_render",
      "!doc": "Renders this view.\n\nThis method is a noop by default. Override it to provide a custom\nimplementation that renders this views content and appends it to the\ncontainer element. Ideally your `render` method should also return `this` as\nthe end to allow chaining, but thats up to you.\n\nSince theres no default renderer, youre free to render your view however\nyou see fit, whether that means manipulating the DOM directly, dumping\nstrings into `innerHTML`, or using a template language of some kind.\n\nFor basic templating needs, `Y.Node.create()` and `Y.Lang.sub()` may\nsuffice, but there are no restrictions on what tools or techniques you can\nuse to render your view. All you need to do is append something to the\ncontainer element at some point, and optionally append the container\nto the DOM if its not there already.",
      "!data": {
       "submodule": "view"
      }
     }
    },
    "NodeMap": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/View.NodeMap.html",
     "getByNode": {
      "!type": "fn(node: +node.Node|+HTMLElement|string) -> +app.View",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/View.NodeMap.html#method_getByNode",
      "!doc": "Returns the nearest View instance associated with the given Node. The Node may\nbe a View container or any child of a View container.\n\nNote that only instances of Views that have the Y.View.NodeMap extension mixed\nin will be returned. The base View class doesnt provide this functionality by\ndefault due to the additional memory management overhead involved in maintaining\na mapping of Nodes to View instances.",
      "!data": {
       "submodule": "view-node-map"
      }
     }
    }
   },
   "LazyModelList": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html",
    "prototype": {
     "!proto": "app.ModelList.prototype",
     "free": {
      "!type": "fn(model?: +app.Model|number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html#method_free",
      "!doc": "Deletes the specified model from the model cache to release memory. The\nmodel wont be destroyed or removed from the list, just freed from the\ncache; it can still be instantiated again using `revive()`.\n\nIf no model or model index is specified, all cached models in this list will\nbe freed.\n\nNote: Specifying an index is faster than specifying a model instance, since\nthe latter requires an `indexOf()` call.",
      "!data": {
       "submodule": "lazy-model-list"
      }
     },
     "get": {
      "!type": "fn(name: string) -> [string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html#method_get",
      "!doc": "Overrides ModelList#get() to return a map of property values rather than\nperforming attribute lookups.",
      "!data": {
       "submodule": "lazy-model-list"
      }
     },
     "getAsHTML": {
      "!type": "fn(name: string) -> [string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html#method_getAsHTML",
      "!doc": "Overrides ModelList#getAsHTML() to return a map of HTML-escaped property\nvalues rather than performing attribute lookups.",
      "!data": {
       "submodule": "lazy-model-list"
      }
     },
     "getAsURL": {
      "!type": "fn(name: string) -> [string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html#method_getAsURL",
      "!doc": "Overrides ModelList#getAsURL() to return a map of URL-encoded property\nvalues rather than performing attribute lookups.",
      "!data": {
       "submodule": "lazy-model-list"
      }
     },
     "indexOf": {
      "!type": "fn(needle: +app.Model|+Object) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html#method_indexOf",
      "!doc": "Returns the index of the given object or Model instance in this\nLazyModelList.",
      "!data": {
       "submodule": "lazy-model-list"
      }
     },
     "reset": {
      "!type": "fn(models?: [+Object]|[+app.Model]|+app.ModelList, options?: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html#method_reset",
      "!doc": "Overrides ModelList#reset() to work with plain objects.",
      "!data": {
       "submodule": "lazy-model-list"
      }
     },
     "revive": {
      "!type": "fn(item?: number|+Object) -> +app.Model|[+app.Model]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html#method_revive",
      "!doc": "Revives an item (or all items) into a full Model instance. The _item_\nargument may be the index of an object in this list, an actual object (which\nmust exist in the list), or may be omitted to revive all items in the list.\n\nOnce revived, Model instances are attached to this list and cached so that\nreviving them in the future doesnt require another Model instantiation. Use\nthe `free()` method to explicitly uncache and detach a previously revived\nModel instance.\n\nNote: Specifying an index rather than an object will be faster, since\nobjects require an `indexOf()` lookup in order to retrieve the index.",
      "!data": {
       "submodule": "lazy-model-list"
      }
     },
     "toJSON": {
      "!type": "fn() -> [+Object]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/LazyModelList.html#method_toJSON",
      "!doc": "Overrides ModelList#toJSON() to use toArray() instead, since its more\nefficient for LazyModelList.",
      "!data": {
       "submodule": "lazy-model-list"
      }
     }
    }
   },
   "ModelList": {
    "!type": "fn(config: +config.ModelListConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html",
    "!data": {
     "extends": [
      "collection.ArrayList"
     ]
    },
    "prototype": {
     "!proto": "base.Base.prototype",
     "model": {
      "!type": "+app.Model|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#property_model",
      "!doc": "The `Model` class or subclass of the models in this list.\n\nThe class specified here will be used to create model instances\nautomatically based on attribute hashes passed to the `add()`, `create()`,\nand `reset()` methods.\n\nYou may specify the class as an actual class reference or as a string that\nresolves to a class reference at runtime (the latter can be useful if the\nspecified class will be loaded lazily).",
      "!data": {
       "submodule": "model-list"
      }
     },
     "add": {
      "!type": "fn(models: +app.Model|[+app.Model]|+app.ModelList|+Object|[+Object], options?: +config.ModelListAddConfig) -> +app.Model|[+app.Model]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_add",
      "!doc": "Adds the specified model or array of models to this list. You may also pass\nanother ModelList instance, in which case all the models in that list will\nbe added to this one as well.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "comparator": {
      "!type": "fn(model: +app.Model) -> number|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_comparator",
      "!doc": "Define this method to provide a function that takes a model as a parameter\nand returns a value by which that model should be sorted relative to other\nmodels in this list.\n\nBy default, no comparator is defined, meaning that models will not be sorted\n(theyll be stored in the order theyre added).",
      "!data": {
       "submodule": "model-list"
      }
     },
     "create": {
      "!type": "fn(model: +app.Model|+Object, options?: +config.ModelListCreateConfig, callback?: fn(err: +Error, response: ?)) -> +app.Model",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_create",
      "!doc": "Creates or updates the specified model on the server, then adds it to this\nlist if the server indicates success.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "each": {
      "!type": "fn(callback: fn(model: +app.Model, index: number, list: +app.ModelList), thisObj?: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_each",
      "!doc": "Executes the supplied function on each model in this list.\n\nBy default, the callback functions `this` object will refer to the model\ncurrently being iterated. Specify a `thisObj` to override the `this` object\nif desired.\n\nNote: Iteration is performed on a copy of the internal array of models, so\nits safe to delete a model from the list during iteration.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "filter": {
      "!type": "fn(options?: +config.ModelListFilterConfig, callback: fn(model: +app.Model, index: number, list: +app.ModelList)) -> [+app.Model]|+app.ModelList",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_filter",
      "!doc": "Executes the supplied function on each model in this list. Returns an array\ncontaining the models for which the supplied function returned a truthy\nvalue.\n\nThe callback functions `this` object will refer to this ModelList. Use\n`Y.bind()` to bind the `this` object to another object if desired.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "get": {
      "!type": "fn(name: string) -> ?|?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_get",
      "!doc": "If _name_ refers to an attribute on this ModelList instance, returns the\nvalue of that attribute. Otherwise, returns an array containing the values\nof the specified attribute from each model in this list.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "getAsHTML": {
      "!type": "fn(name: string) -> string|[string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_getAsHTML",
      "!doc": "If _name_ refers to an attribute on this ModelList instance, returns the\nHTML-escaped value of that attribute. Otherwise, returns an array containing\nthe HTML-escaped values of the specified attribute from each model in this\nlist.\n\nThe values are escaped using `Escape.html()`.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "getAsURL": {
      "!type": "fn(name: string) -> string|[string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_getAsURL",
      "!doc": "If _name_ refers to an attribute on this ModelList instance, returns the\nURL-encoded value of that attribute. Otherwise, returns an array containing\nthe URL-encoded values of the specified attribute from each model in this\nlist.\n\nThe values are encoded using the native `encodeURIComponent()` function.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "getByClientId": {
      "!type": "fn(clientId: string) -> +app.Model",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_getByClientId",
      "!doc": "Returns the model with the specified _clientId_, or `null` if not found.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "getById": {
      "!type": "fn(id: string|number) -> +app.Model",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_getById",
      "!doc": "Returns the model with the specified _id_, or `null` if not found.\n\nNote that models arent expected to have an id until theyre saved, so if\nyoure working with unsaved models, it may be safer to call\n`getByClientId()`.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "invoke": {
      "!type": "fn(name: string, args?: ?) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_invoke",
      "!doc": "Calls the named method on every model in the list. Any arguments provided\nafter _name_ will be passed on to the invoked method.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "item": {
      "!type": "fn(index: number) -> +app.Model",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_item",
      "!doc": "Returns the model at the specified _index_.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "load": {
      "!type": "fn(options?: +Object, callback?: fn(err: +Error, response: ?)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_load",
      "!doc": "Loads this list of models from the server.\n\nThis method delegates to the `sync()` method to perform the actual load\noperation, which is an asynchronous action. Specify a _callback_ function to\nbe notified of success or failure.\n\nIf the load operation succeeds, a `reset` event will be fired.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "map": {
      "!type": "fn(fn: fn(model: +app.Model, index: number, models: [+app.Model]), thisObj?: +Object) -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_map",
      "!doc": "Executes the specified function on each model in this list and returns an\narray of the functions collected return values.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "parse": {
      "!type": "fn(response: ?) -> [+Object]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_parse",
      "!doc": "Called to parse the _response_ when the list is loaded from the server.\nThis method receives a server _response_ and is expected to return an array\nof model attribute hashes.\n\nThe default implementation assumes that _response_ is either an array of\nattribute hashes or a JSON string that can be parsed into an array of\nattribute hashes. If _response_ is a JSON string and either `Y.JSON` or the\nnative `JSON` object are available, it will be parsed automatically. If a\nparse error occurs, an `error` event will be fired and the model will not be\nupdated.\n\nYou may override this method to implement custom parsing logic if necessary.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "remove": {
      "!type": "fn(models: +app.Model|[+app.Model]|+app.ModelList|number|[number], options?: +config.ModelListRemoveConfig) -> +app.Model|[+app.Model]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_remove",
      "!doc": "Removes the specified model or array of models from this list. You may also\npass another ModelList instance to remove all the models that are in both\nthat instance and this instance, or pass numerical indices to remove the\nmodels at those indices.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "reset": {
      "!type": "fn(models?: [+app.Model]|+app.ModelList|[+Object], options?: +config.ModelListResetConfig) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_reset",
      "!doc": "Completely replaces all models in the list with those specified, and fires a\nsingle `reset` event.\n\nUse `reset` when you want to add or remove a large number of items at once\nwith less overhead, and without firing `add` or `remove` events for each\none.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "some": {
      "!type": "fn(callback: fn(model: +app.Model, index: number, list: +app.ModelList), thisObj?: +Object) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_some",
      "!doc": "Executes the supplied function on each model in this list, and stops\niterating if the callback returns `true`.\n\nBy default, the callback functions `this` object will refer to the model\ncurrently being iterated. Specify a `thisObj` to override the `this` object\nif desired.\n\nNote: Iteration is performed on a copy of the internal array of models, so\nits safe to delete a model from the list during iteration.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "sort": {
      "!type": "fn(options?: +config.ModelListSortConfig) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_sort",
      "!doc": "Forcibly re-sorts the list.\n\nUsually it shouldnt be necessary to call this method since the list\nmaintains its sort order when items are added and removed, but if you change\nthe `comparator` function after items are already in the list, youll need\nto re-sort.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "sync": {
      "!type": "fn(action: string, options?: +Object, callback?: fn(err: +Error, response?: ?))",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_sync",
      "!doc": "Override this method to provide a custom persistence implementation for this\nlist. The default method just calls the callback without actually doing\nanything.\n\nThis method is called internally by `load()` and its implementations relies\non the callback being called. This effectively means that when a callback is\nprovided, it must be called at some point for the class to operate correctly.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "toArray": {
      "!type": "fn() -> [+app.Model]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_toArray",
      "!doc": "Returns an array containing the models in this list.",
      "!data": {
       "submodule": "model-list"
      }
     },
     "toJSON": {
      "!type": "fn() -> [+Object]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ModelList.html#method_toJSON",
      "!doc": "Returns an array containing attribute hashes for each model in this list,\nsuitable for being passed to `Y.JSON.stringify()`.\n\nUnder the hood, this method calls `toJSON()` on each model in the list and\npushes the results into an array.",
      "!data": {
       "submodule": "model-list"
      }
     }
    }
   },
   "Model": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "changed": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#property_changed",
      "!doc": "Hash of attributes that have changed since the last time this model was\nsaved.",
      "!data": {
       "submodule": "model"
      }
     },
     "idAttribute": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#property_idAttribute",
      "!doc": "Name of the attribute to use as the unique id (or primary key) for this\nmodel.\n\nThe default is `id`, but if your persistence layer uses a different name for\nthe primary key (such as `_id` or `uid`), you can specify that here.\n\nThe built-in `id` attribute will always be an alias for whatever attribute\nname you specify here, so getting and setting `id` will always behave the\nsame as getting and setting your custom id attribute.",
      "!data": {
       "submodule": "model"
      }
     },
     "lastChange": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#property_lastChange",
      "!doc": "Hash of attributes that were changed in the last `change` event. Each item\nin this hash is an object with the following properties:\n\n  * `newVal`: The new value of the attribute after it changed.\n  * `prevVal`: The old value of the attribute before it changed.\n  * `src`: The source of the change, or `null` if no source was specified.",
      "!data": {
       "submodule": "model"
      }
     },
     "lists": {
      "!type": "[+app.ModelList]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#property_lists",
      "!doc": "Array of `ModelList` instances that contain this model.\n\nWhen a model is in one or more lists, the models events will bubble up to\nthose lists. You can subscribe to a model event on a list to be notified\nwhen any model in the list fires that event.\n\nThis property is updated automatically when this model is added to or\nremoved from a `ModelList` instance. You shouldnt alter it manually. When\nworking with models in a list, you should always add and remove models using\nthe lists `add()` and `remove()` methods.",
      "!data": {
       "submodule": "model"
      }
     },
     "destroy": {
      "!type": "fn(options?: +config.ModelDestroyConfig, callback?: fn(err: +Error)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_destroy",
      "!doc": "Destroys this model instance and removes it from its containing lists, if\nany.\n\nThe _callback_, if one is provided, will be called after the model is\ndestroyed.\n\nIf `options.remove` is `true`, then this method delegates to the `sync()`\nmethod to delete the model from the persistence layer, which is an\nasynchronous action. In this case, the _callback_ (if provided) will be\ncalled after the sync layer indicates success or failure of the delete\noperation.",
      "!data": {
       "submodule": "model"
      }
     },
     "generateClientId": {
      "!type": "fn() -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_generateClientId",
      "!doc": "Returns a clientId string thats unique among all models on the current page\n(even models in other YUI instances). Uniqueness across pageviews is\nunlikely.",
      "!data": {
       "submodule": "model"
      }
     },
     "get": {
      "!type": "fn(name: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_get",
      "!doc": "Returns the value of the specified attribute.\n\nIf the attributes value is an object, _name_ may use dot notation to\nspecify the path to a specific property within the object, and the value of\nthat property will be returned.",
      "!data": {
       "submodule": "model"
      }
     },
     "getAsHTML": {
      "!type": "fn(name: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_getAsHTML",
      "!doc": "Returns an HTML-escaped version of the value of the specified string\nattribute. The value is escaped using `Y.Escape.html()`.",
      "!data": {
       "submodule": "model"
      }
     },
     "getAsURL": {
      "!type": "fn(name: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_getAsURL",
      "!doc": "Returns a URL-encoded version of the value of the specified string\nattribute. The value is encoded using the native `encodeURIComponent()`\nfunction.",
      "!data": {
       "submodule": "model"
      }
     },
     "isModified": {
      "!type": "fn() -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_isModified",
      "!doc": "Returns `true` if any attribute of this model has been changed since the\nmodel was last saved.\n\nNew models (models for which `isNew()` returns `true`) are implicitly\nconsidered to be \"modified\" until the first time theyre saved.",
      "!data": {
       "submodule": "model"
      }
     },
     "isNew": {
      "!type": "fn() -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_isNew",
      "!doc": "Returns `true` if this model is \"new\", meaning it hasnt been saved since it\nwas created.\n\nNewness is determined by checking whether the models `id` attribute has\nbeen set. An empty id is assumed to indicate a new model, whereas a\nnon-empty id indicates a model that was either loaded or has been saved\nsince it was created.",
      "!data": {
       "submodule": "model"
      }
     },
     "load": {
      "!type": "fn(options?: +Object, callback?: fn(err: +Error, response: ?)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_load",
      "!doc": "Loads this model from the server.\n\nThis method delegates to the `sync()` method to perform the actual load\noperation, which is an asynchronous action. Specify a _callback_ function to\nbe notified of success or failure.\n\nA successful load operation will fire a `load` event, while an unsuccessful\nload operation will fire an `error` event with the `src` value \"load\".\n\nIf the load operation succeeds and one or more of the loaded attributes\ndiffer from this models current attributes, a `change` event will be fired.",
      "!data": {
       "submodule": "model"
      }
     },
     "parse": {
      "!type": "fn(response: ?) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_parse",
      "!doc": "Called to parse the _response_ when the model is loaded from the server.\nThis method receives a server _response_ and is expected to return an\nattribute hash.\n\nThe default implementation assumes that _response_ is either an attribute\nhash or a JSON string that can be parsed into an attribute hash. If\n_response_ is a JSON string and either `Y.JSON` or the native `JSON` object\nare available, it will be parsed automatically. If a parse error occurs, an\n`error` event will be fired and the model will not be updated.\n\nYou may override this method to implement custom parsing logic if necessary.",
      "!data": {
       "submodule": "model"
      }
     },
     "save": {
      "!type": "fn(options?: +Object, callback?: fn(err: +Error, response: ?)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_save",
      "!doc": "Saves this model to the server.\n\nThis method delegates to the `sync()` method to perform the actual save\noperation, which is an asynchronous action. Specify a _callback_ function to\nbe notified of success or failure.\n\nA successful save operation will fire a `save` event, while an unsuccessful\nsave operation will fire an `error` event with the `src` value \"save\".\n\nIf the save operation succeeds and one or more of the attributes returned in\nthe servers response differ from this models current attributes, a\n`change` event will be fired.",
      "!data": {
       "submodule": "model"
      }
     },
     "set": {
      "!type": "fn(name: string, value: ?, options?: +config.ModelSetConfig) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_set",
      "!doc": "Sets the value of a single attribute. If model validation fails, the\nattribute will not be set and an `error` event will be fired.\n\nUse `setAttrs()` to set multiple attributes at once.",
      "!data": {
       "submodule": "model"
      }
     },
     "setAttrs": {
      "!type": "fn(attributes: +Object, options?: +config.ModelSetAttrsConfig) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_setAttrs",
      "!doc": "Sets the values of multiple attributes at once. If model validation fails,\nthe attributes will not be set and an `error` event will be fired.",
      "!data": {
       "submodule": "model"
      }
     },
     "sync": {
      "!type": "fn(action: string, options?: +Object, callback?: fn(err: +Error, response?: ?))",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_sync",
      "!doc": "Override this method to provide a custom persistence implementation for this\nmodel. The default just calls the callback without actually doing anything.\n\nThis method is called internally by `load()`, `save()`, and `destroy()`, and\ntheir implementations rely on the callback being called. This effectively\nmeans that when a callback is provided, it must be called at some point for\nthe class to operate correctly.",
      "!data": {
       "submodule": "model"
      }
     },
     "toJSON": {
      "!type": "fn() -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_toJSON",
      "!doc": "Returns a copy of this models attributes that can be passed to\n`Y.JSON.stringify()` or used for other nefarious purposes.\n\nThe `clientId` attribute is not included in the returned object.\n\nIf youve specified a custom attribute name in the `idAttribute` property,\nthe default `id` attribute will not be included in the returned object.\n\nNote: The ECMAScript 5 specification states that objects may implement a\n`toJSON` method to provide an alternate object representation to serialize\nwhen passed to `JSON.stringify(obj)`.  This allows class instances to be\nserialized as if they were plain objects.  This is why Models `toJSON`\nreturns an object, not a JSON string.\n\nSee <http://es5.github.com/#x15.12.3> for details.",
      "!data": {
       "submodule": "model"
      }
     },
     "undo": {
      "!type": "fn(attrNames?: [string], options?: +config.ModelUndoConfig) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_undo",
      "!doc": "Reverts the last change to the model.\n\nIf an _attrNames_ array is provided, then only the named attributes will be\nreverted (and only if they were modified in the previous change). If no\n_attrNames_ array is provided, then all changed attributes will be reverted\nto their previous values.\n\nNote that only one level of undo is available: from the current state to the\nprevious state. If `undo()` is called when no previous state is available,\nit will simply do nothing.",
      "!data": {
       "submodule": "model"
      }
     },
     "validate": {
      "!type": "fn(attrs: +Object, callback: fn(err?: ?))",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Model.html#method_validate",
      "!doc": "Override this method to provide custom validation logic for this model.\n\nWhile attribute-specific validators can be used to validate individual\nattributes, this method gives you a hook to validate a hash of all\nattributes before the model is saved. This method is called automatically\nbefore `save()` takes any action. If validation fails, the `save()` call\nwill be aborted.\n\nIn your validation method, call the provided `callback` function with no\narguments to indicate success. To indicate failure, pass a single argument,\nwhich may contain an error message, an array of error messages, or any other\nvalue. This value will be passed along to the `error` event.",
      "!data": {
       "submodule": "model"
      }
     }
    }
   },
   "Router": {
    "!type": "fn(config?: +config.RouterConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "dispatch": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_dispatch",
      "!doc": "Dispatches to the first route handler that matches the current URL, if any.\n\nIf `dispatch()` is called before the `ready` event has fired, it will\nautomatically wait for the `ready` event before dispatching. Otherwise it\nwill dispatch immediately.",
      "!data": {
       "submodule": "router"
      }
     },
     "getPath": {
      "!type": "fn() -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_getPath",
      "!doc": "Gets the current route path.",
      "!data": {
       "submodule": "router"
      }
     },
     "hasRoute": {
      "!type": "fn(url: string) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_hasRoute",
      "!doc": "Returns `true` if this router has at least one route that matches the\nspecified URL, `false` otherwise. This also checks that any named `param`\nhandlers also accept app param values in the `url`.\n\nThis method enforces the same-origin security constraint on the specified\n`url`; any URL which is not from the same origin as the current URL will\nalways return `false`.",
      "!data": {
       "submodule": "router"
      }
     },
     "match": {
      "!type": "fn(path: string) -> [+Object]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_match",
      "!doc": "Returns an array of route objects that match the specified URL path.\n\nIf this router has a `root`, then the specified `path` _must_ be\nsemantically within the `root` path to match any routes.\n\nThis method is called internally to determine which routes match the current\npath whenever the URL changes. You may override it if you want to customize\nthe route matching logic, although this usually shouldnt be necessary.\n\nEach returned route object has the following properties:\n\n  * `callback`: A function or a string representing the name of a function\n    this router that should be executed when the route is triggered.\n\n  * `keys`: An array of strings representing the named parameters defined in\n    the routes path specification, if any.\n\n  * `path`: The routes path specification, which may be either a string or\n    a regex.\n\n  * `regex`: A regular expression version of the routes path specification.\n    This regex is used to determine whether the route matches a given path.",
      "!data": {
       "submodule": "router"
      }
     },
     "param": {
      "!type": "fn(name: string, handler: fn(value: string, name: string)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_param",
      "!doc": "Adds a handler for a route param specified by _name_.\n\nParam handlers can be registered via this method and are used to\nvalidate/format values of named params in routes before dispatching to the\nroutes handler functions. Using param handlers allows routes to defined\nusing string paths which allows for `req.params` to use named params, but\nstill applying extra validation or formatting to the param values parsed\nfrom the URL.\n\nIf a param handler regex or function returns a value of `false`, `null`,\n`undefined`, or `NaN`, the current route will not match and be skipped. All\nother return values will be used in place of the original param value parsed\nfrom the URL.",
      "!data": {
       "submodule": "router"
      }
     },
     "removeRoot": {
      "!type": "fn(url: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_removeRoot",
      "!doc": "Removes the `root` URL from the front of _url_ (if its there) and returns\nthe result. The returned path will always have a leading `/`.",
      "!data": {
       "submodule": "router"
      }
     },
     "removeQuery": {
      "!type": "fn(url: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_removeQuery",
      "!doc": "Removes a query string from the end of the _url_ (if one exists) and returns\nthe result.",
      "!data": {
       "submodule": "router"
      }
     },
     "replace": {
      "!type": "fn(url?: string) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_replace",
      "!doc": "Replaces the current browser history entry with a new one, and dispatches to\nthe first matching route handler, if any.\n\nBehind the scenes, this method uses HTML5 `pushState()` in browsers that\nsupport it (or the location hash in older browsers and IE) to change the\nURL.\n\nThe specified URL must share the same origin (i.e., protocol, host, and\nport) as the current page, or an error will occur.",
      "!data": {
       "submodule": "router"
      }
     },
     "route": {
      "!type": "fn(route: string|+RegExp|+Object, callbacks: [?]|string) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_route",
      "!doc": "Adds a route handler for the specified `route`.\n\nThe `route` parameter may be a string or regular expression to represent a\nURL path, or a route object. If its a string (which is most common), it may\ncontain named parameters: `:param` will match any single part of a URL path\n(not including `/` characters), and `*param` will match any number of parts\nof a URL path (including `/` characters). These named parameters will be\nmade available as keys on the `req.params` object thats passed to route\nhandlers.\n\nIf the `route` parameter is a regex, all pattern matches will be made\navailable as numbered keys on `req.params`, starting with `0` for the full\nmatch, then `1` for the first subpattern match, and so on.\n\nAlternatively, an object can be provided to represent the route and it may\ncontain a `path` property which is a string or regular expression which\ncauses the route to be process as described above. If the route object\nalready contains a `regex` or `regexp` property, the route will be\nconsidered fully-processed and will be associated with any `callacks`\nspecified on the object and those specified as parameters to this method.\n**Note:** Any additional data contained on the route object will be\npreserved.\n\nHeres a set of sample routes along with URL paths that they match:\n\n  * Route: `/photos/:tag/:page`\n    * URL: `/photos/kittens/1`, params: `{tag: kittens, page: 1}`\n    * URL: `/photos/puppies/2`, params: `{tag: puppies, page: 2}`\n\n  * Route: `/file/*path`\n    * URL: `/file/foo/bar/baz.txt`, params: `{path: foo/bar/baz.txt}`\n    * URL: `/file/foo`, params: `{path: foo}`\n\n**Middleware**: Routes also support an arbitrary number of callback\nfunctions. This allows you to easily reuse parts of your route-handling code\nwith different route. This method is liberal in how it processes the\nspecified `callbacks`, you can specify them as separate arguments, or as\narrays, or both.\n\nIf multiple route match a given URL, they will be executed in the order they\nwere added. The first route that was added will be the first to be executed.\n\n**Passing Control**: Invoking the `next()` function within a route callback\nwill pass control to the next callback function (if any) or route handler\n(if any). If a value is passed to `next()`, its assumed to be an error,\ntherefore stopping the dispatch chain, unless that value is: `\"route\"`,\nwhich is special case and dispatching will skip to the next route handler.\nThis allows middleware to skip any remaining middleware for a particular\nroute.",
      "!data": {
       "submodule": "router"
      }
     },
     "save": {
      "!type": "fn(url?: string) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_save",
      "!doc": "Saves a new browser history entry and dispatches to the first matching route\nhandler, if any.\n\nBehind the scenes, this method uses HTML5 `pushState()` in browsers that\nsupport it (or the location hash in older browsers and IE) to change the\nURL and create a history entry.\n\nThe specified URL must share the same origin (i.e., protocol, host, and\nport) as the current page, or an error will occur.",
      "!data": {
       "submodule": "router"
      }
     },
     "upgrade": {
      "!type": "fn() -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_upgrade",
      "!doc": "Upgrades a hash-based URL to an HTML5 URL if necessary. In non-HTML5\nbrowsers, this method is a noop.",
      "!data": {
       "submodule": "router"
      }
     }
    },
    "dispatch": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Router.html#method_dispatch",
     "!doc": "Dispatches to the first route handler that matches the specified `path` for\nall active router instances.\n\nThis provides a mechanism to cause all active router instances to dispatch\nto their route handlers without needing to change the URL or fire the\n`history:change` or `hashchange` event.",
     "!data": {
      "submodule": "router"
     }
    }
   }
  },
  "arraysort": {
   "ArraySort": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ArraySort.html",
    "compare": {
     "!type": "fn(a: +Object, b: +Object, desc: bool) -> bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ArraySort.html#method_compare",
     "!doc": "Comparator function for simple case-insensitive sorting of an array of\nstrings."
    },
    "naturalCompare": {
     "!type": "fn(a: number|string, b: number|string, options?: +config.ArraySortNaturalCompareConfig) -> number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ArraySort.html#method_naturalCompare",
     "!doc": "Performs a natural-order comparison of two strings or numbers (or a string\nand a number). This ensures that a value like foo2 will be sorted before\nfoo10, whereas a standard ASCII sort would sort foo10 first."
    }
   }
  },
  "async_queue": {
   "!data": {
    "module": "async-queue"
   },
   "AsyncQueue": {
    "!type": "fn(callback: fn())",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html",
    "prototype": {
     "!proto": "event_custom.EventTarget.prototype",
     "defaults": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#property_defaults",
      "!doc": "Callback defaults for this instance.  Static defaults that are not\noverridden are also included."
     },
     "next": {
      "!type": "fn() -> fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_next",
      "!doc": "Returns the next callback needing execution.  If a callback is\nconfigured to repeat via iterations or until, it will be returned until\nthe completion criteria is met.\n\nWhen the queue is empty, null is returned."
     },
     "run": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_run",
      "!doc": "Sets the queue in motion.  All queued callbacks will be executed in\norder unless pause() or stop() is called or if one of the callbacks is\nconfigured with autoContinue: false."
     },
     "isRunning": {
      "!type": "fn() -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_isRunning",
      "!doc": "Determines if the queue is waiting for a callback to complete execution."
     },
     "add": {
      "!type": "fn(callback: fn()) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_add",
      "!doc": "Add any number of callbacks to the end of the queue. Callbacks may be\nprovided as functions or objects."
     },
     "pause": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_pause",
      "!doc": "Pause the execution of the queue after the execution of the current\ncallback completes.  If called from code outside of a queued callback,\nclears the timeout for the pending callback. Paused queue can be\nrestarted with q.run()"
     },
     "stop": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_stop",
      "!doc": "Stop and clear the queue after the current execution of the\ncurrent callback completes."
     },
     "indexOf": {
      "!type": "fn(callback: string) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_indexOf",
      "!doc": "Returns the current index of a callback.  Pass in either the id or\ncallback function from getCallback."
     },
     "getCallback": {
      "!type": "fn(id: string) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_getCallback",
      "!doc": "Retrieve a callback by its id.  Useful to modify the configuration\nwhile the queue is running."
     },
     "promote": {
      "!type": "fn(callback: string|+Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_promote",
      "!doc": "Promotes the named callback to the top of the queue. If a callback is\ncurrently executing or looping (via until or iterations), the promotion\nis scheduled to occur after the current callback has completed."
     },
     "remove": {
      "!type": "fn(callback: string|+Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_remove",
      "!doc": "Removes the callback from the queue.  If the queue is active, the\nremoval is scheduled to occur after the current callback has completed."
     },
     "size": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#method_size",
      "!doc": "Returns the number of callbacks in the queue."
     }
    },
    "defaults": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AsyncQueue.html#property_defaults",
     "!doc": "<p>Static default values used to populate callback configuration properties.\nPreconfigured defaults include:</p>\n\n<ul>\n <li><code>autoContinue</code>: <code>true</code></li>\n <li><code>iterations</code>: 1</li>\n <li><code>timeout</code>: 10 (10ms between callbacks)</li>\n <li><code>until</code>: (function to run until iterations &lt;= 0)</li>\n</ul>"
    }
   }
  },
  "attribute": {
   "!data": {
    "submodules": {
     "attribute-base": {},
     "attribute-core": {},
     "attribute-extras": {},
     "attribute-observable": {},
     "attribute-complex": {}
    }
   },
   "Attribute": {
    "!type": "fn(attrs: +Object, values: +Object, lazy: bool)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Attribute.html",
    "!data": {
     "extends": [
      "attribute.AttributeCore",
      "attribute.AttributeObservable",
      "event_custom.EventTarget",
      "attribute.AttributeExtras"
     ]
    },
    "INVALID_VALUE": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Attribute.html#property_INVALID_VALUE",
     "!doc": "<p>The value to return from an attribute setter in order to prevent the set from going through.</p>\n\n<p>You can return this value from your setter if you wish to combine validator and setter\nfunctionality into a single setter function, which either returns the massaged value to be stored or\nAttributeCore.INVALID_VALUE to prevent invalid values from being stored.</p>",
     "!data": {
      "submodule": "attribute-base"
     }
    },
    "protectAttrs": {
     "!type": "fn(attrs: +Object) -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Attribute.html#method_protectAttrs",
     "!doc": "Utility method to protect an attribute configuration hash, by merging the\nentire object and the individual attr config objects.",
     "!data": {
      "submodule": "attribute-base"
     }
    }
   },
   "AttributeCore": {
    "!type": "fn(attrs: +Object, values: +Object, lazy: bool)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html",
    "INVALID_VALUE": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#property_INVALID_VALUE",
     "!doc": "<p>The value to return from an attribute setter in order to prevent the set from going through.</p>\n\n<p>You can return this value from your setter if you wish to combine validator and setter\nfunctionality into a single setter function, which either returns the massaged value to be stored or\nAttributeCore.INVALID_VALUE to prevent invalid values from being stored.</p>",
     "!data": {
      "submodule": "attribute-core"
     }
    },
    "protectAttrs": {
     "!type": "fn(attrs: +Object) -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#method_protectAttrs",
     "!doc": "Utility method to protect an attribute configuration hash, by merging the\nentire object and the individual attr config objects.",
     "!data": {
      "submodule": "attribute-core"
     }
    },
    "prototype": {
     "addAttr": {
      "!type": "fn(name: string, config: +config.AttributeCoreAddAttrConfig, lazy: bool) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#method_addAttr",
      "!doc": "<p>\nAdds an attribute with the provided configuration to the host object.\n</p>\n<p>\nThe config argument object supports the following properties:\n</p>\n\n<dl>\n   <dt>value &#60;Any&#62;</dt>\n   <dd>The initial value to set on the attribute</dd>\n\n   <dt>valueFn &#60;Function | String&#62;</dt>\n   <dd>\n   <p>A function, which will return the initial value to set on the attribute. This is useful\n   for cases where the attribute configuration is defined statically, but needs to\n   reference the host instance (\"this\") to obtain an initial value. If both the value and valueFn properties are defined,\n   the value returned by the valueFn has precedence over the value property, unless it returns undefined, in which\n   case the value property is used.</p>\n\n   <p>valueFn can also be set to a string, representing the name of the instance method to be used to retrieve the value.</p>\n   </dd>\n\n   <dt>readOnly &#60;boolean&#62;</dt>\n   <dd>Whether or not the attribute is read only. Attributes having readOnly set to true\n       cannot be modified by invoking the set method.</dd>\n\n   <dt>writeOnce &#60;boolean&#62; or &#60;string&#62;</dt>\n   <dd>\n       Whether or not the attribute is \"write once\". Attributes having writeOnce set to true,\n       can only have their values set once, be it through the default configuration,\n       constructor configuration arguments, or by invoking set.\n       <p>The writeOnce attribute can also be set to the string \"initOnly\",\n        in which case the attribute can only be set during initialization\n       (when used with Base, this means it can only be set during construction)</p>\n   </dd>\n\n   <dt>setter &#60;Function | String&#62;</dt>\n   <dd>\n   <p>The setter function used to massage or normalize the value passed to the set method for the attribute.\n   The value returned by the setter will be the final stored value. Returning\n   <a href=\"#property_Attribute.INVALID_VALUE\">Attribute.INVALID_VALUE</a>, from the setter will prevent\n   the value from being stored.\n   </p>\n\n   <p>setter can also be set to a string, representing the name of the instance method to be used as the setter function.</p>\n   </dd>\n\n   <dt>getter &#60;Function | String&#62;</dt>\n   <dd>\n   <p>\n   The getter function used to massage or normalize the value returned by the get method for the attribute.\n   The value returned by the getter function is the value which will be returned to the user when they\n   invoke get.\n   </p>\n\n   <p>getter can also be set to a string, representing the name of the instance method to be used as the getter function.</p>\n   </dd>\n\n   <dt>validator &#60;Function | String&#62;</dt>\n   <dd>\n   <p>\n   The validator function invoked prior to setting the stored value. Returning\n   false from the validator function will prevent the value from being stored.\n   </p>\n\n   <p>validator can also be set to a string, representing the name of the instance method to be used as the validator function.</p>\n   </dd>\n\n   <dt>lazyAdd &#60;boolean&#62;</dt>\n   <dd>Whether or not to delay initialization of the attribute until the first call to get/set it.\n   This flag can be used to over-ride lazy initialization on a per attribute basis, when adding multiple attributes through\n   the <a href=\"#method_addAttrs\">addAttrs</a> method.</dd>\n\n</dl>\n\n<p>The setter, getter and validator are invoked with the value and name passed in as the first and second arguments, and with\nthe context (\"this\") set to the host object.</p>\n\n<p>Configuration properties outside of the list mentioned above are considered private properties used internally by attribute,\nand are not intended for public use.</p>",
      "!data": {
       "submodule": "attribute-core"
      }
     },
     "attrAdded": {
      "!type": "fn(name: string) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#method_attrAdded",
      "!doc": "Checks if the given attribute has been added to the host",
      "!data": {
       "submodule": "attribute-core"
      }
     },
     "get": {
      "!type": "fn(name: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#method_get",
      "!doc": "Returns the current value of the attribute. If the attribute\nhas been configured with a getter function, this method will delegate\nto the getter to obtain the value of the attribute.",
      "!data": {
       "submodule": "attribute-core"
      }
     },
     "set": {
      "!type": "fn(name: string, value: ?, opts?: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#method_set",
      "!doc": "Sets the value of an attribute.",
      "!data": {
       "submodule": "attribute-core"
      }
     },
     "setAttrs": {
      "!type": "fn(attrs: +Object, opts?: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#method_setAttrs",
      "!doc": "Sets multiple attribute values.",
      "!data": {
       "submodule": "attribute-core"
      }
     },
     "getAttrs": {
      "!type": "fn(attrs: [string]|bool) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#method_getAttrs",
      "!doc": "Gets multiple attribute values.",
      "!data": {
       "submodule": "attribute-core"
      }
     },
     "addAttrs": {
      "!type": "fn(cfgs: +Object, values: +Object, lazy: bool) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeCore.html#method_addAttrs",
      "!doc": "Configures a group of attributes, and sets initial values.\n\n<p>\n<strong>NOTE:</strong> This method does not isolate the configuration object by merging/cloning.\nThe caller is responsible for merging/cloning the configuration object if required.\n</p>",
      "!data": {
       "submodule": "attribute-core"
      }
     }
    }
   },
   "AttributeExtras": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeExtras.html",
    "prototype": {
     "modifyAttr": {
      "!type": "fn(name: string, config: +config.AttributeExtrasModifyAttrConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeExtras.html#method_modifyAttr",
      "!doc": "Updates the configuration of an attribute which has already been added.\n<p>\nThe properties which can be modified through this interface are limited\nto the following subset of attributes, which can be safely modified\nafter a value has already been set on the attribute:\n</p>\n<dl>\n <dt>readOnly;</dt>\n <dt>writeOnce;</dt>\n <dt>broadcast; and</dt>\n <dt>getter.</dt>\n</dl>\n<p>\nNote: New attributes cannot be added using this interface. New attributes must be\nadded using {{#crossLink \"AttributeCore/addAttr:method\"}}addAttr{{/crossLink}}, or an\nappropriate manner for a class which utilises Attributes (e.g. the\n{{#crossLink \"Base/ATTRS:property\"}}ATTRS{{/crossLink}} property in\n{{#crossLink \"Base\"}}Base{{/crossLink}}).\n</p>",
      "!data": {
       "submodule": "attribute-extras"
      }
     },
     "removeAttr": {
      "!type": "fn(name: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeExtras.html#method_removeAttr",
      "!doc": "Removes an attribute from the host object",
      "!data": {
       "submodule": "attribute-extras"
      }
     },
     "reset": {
      "!type": "fn(name: string) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeExtras.html#method_reset",
      "!doc": "Resets the attribute (or all attributes) to its initial value, as long as\nthe attribute is not readOnly, or writeOnce.",
      "!data": {
       "submodule": "attribute-extras"
      }
     }
    }
   },
   "AttributeObservable": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeObservable.html",
    "!data": {
     "extends": [
      "event_custom.EventTarget"
     ]
    },
    "prototype": {
     "set": {
      "!type": "fn(name: string, value: ?, opts: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeObservable.html#method_set",
      "!doc": "Sets the value of an attribute.",
      "!data": {
       "submodule": "attribute-observable"
      }
     },
     "setAttrs": {
      "!type": "fn(attrs: +Object, opts: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeObservable.html#method_setAttrs",
      "!doc": "Sets multiple attribute values.",
      "!data": {
       "submodule": "attribute-observable"
      }
     }
    }
   },
   "State": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/State.html",
    "prototype": {
     "data": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/State.html#property_data",
      "!doc": "Hash of attributes"
     },
     "add": {
      "!type": "fn(name: string, key: string, val: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/State.html#method_add",
      "!doc": "Adds a property to an item."
     },
     "addAll": {
      "!type": "fn(name: string, obj: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/State.html#method_addAll",
      "!doc": "Adds multiple properties to an item."
     },
     "remove": {
      "!type": "fn(name: string, key: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/State.html#method_remove",
      "!doc": "Removes a property from an item."
     },
     "removeAll": {
      "!type": "fn(name: string, obj: +Object|[?])",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/State.html#method_removeAll",
      "!doc": "Removes multiple properties from an item, or removes the item completely."
     },
     "get": {
      "!type": "fn(name: string, key: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/State.html#method_get",
      "!doc": "For a given item, returns the value of the property requested, or undefined if not found."
     },
     "getAll": {
      "!type": "fn(name: string, reference: bool) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/State.html#method_getAll",
      "!doc": "For the given item, returns an object with all of the\nitems property/value pairs. By default the object returned\nis a shallow copy of the stored data, but passing in true\nas the second parameter will return a reference to the stored\ndata."
     }
    }
   }
  },
  "autocomplete": {
   "!data": {
    "submodules": {
     "autocomplete-base": {},
     "autocomplete-filters-accentfold": {},
     "autocomplete-filters": {},
     "autocomplete-highlighters-accentfold": {},
     "autocomplete-highlighters": {},
     "autocomplete-list-keys": {},
     "autocomplete-list": {},
     "autocomplete-plugin": {},
     "autocomplete-sources": {}
    }
   },
   "AutoCompleteBase": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html",
    "prototype": {
     "clearCache": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#method_clearCache",
      "!doc": "Clears the result cache.",
      "!data": {
       "submodule": "autocomplete-base"
      }
     },
     "sendRequest": {
      "!type": "fn(query?: string, requestTemplate?: fn()) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#method_sendRequest",
      "!doc": "Sends a request to the configured source. If no source is configured, this\nmethod wont do anything.\n\nUsually theres no reason to call this method manually; it will be called\nautomatically when user input causes a `query` event to be fired. The only\ntime youll need to call this method manually is if you want to force a\nrequest to be sent when no user input has occurred.",
      "!data": {
       "submodule": "autocomplete-base"
      }
     }
    },
    "SOURCE_TYPES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteBase.html#property_SOURCE_TYPES",
     "!doc": "Mapping of built-in source types to their setter functions. DataSource instances\nand DataSource-like objects are handled natively, so are not mapped here.",
     "!data": {
      "submodule": "autocomplete-base"
     }
    }
   },
   "AutoCompleteFilters": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html",
    "charMatchFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_charMatchFold",
     "!doc": "Accent folding version of `charMatch()`.",
     "!data": {
      "submodule": "autocomplete-filters-accentfold"
     }
    },
    "phraseMatchFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_phraseMatchFold",
     "!doc": "Accent folding version of `phraseMatch()`.",
     "!data": {
      "submodule": "autocomplete-filters-accentfold"
     }
    },
    "startsWithFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_startsWithFold",
     "!doc": "Accent folding version of `startsWith()`.",
     "!data": {
      "submodule": "autocomplete-filters-accentfold"
     }
    },
    "subWordMatchFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_subWordMatchFold",
     "!doc": "Accent folding version of `subWordMatch()`.",
     "!data": {
      "submodule": "autocomplete-filters-accentfold"
     }
    },
    "wordMatchFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_wordMatchFold",
     "!doc": "Accent folding version of `wordMatch()`.",
     "!data": {
      "submodule": "autocomplete-filters-accentfold"
     }
    },
    "charMatch": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_charMatch",
     "!doc": "Returns an array of results that contain all of the characters in the query,\nin any order (not necessarily consecutive). Case-insensitive.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "charMatchCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_charMatchCase",
     "!doc": "Case-sensitive version of `charMatch()`.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "phraseMatch": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_phraseMatch",
     "!doc": "Returns an array of results that contain the complete query as a phrase.\nCase-insensitive.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "phraseMatchCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_phraseMatchCase",
     "!doc": "Case-sensitive version of `phraseMatch()`.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "startsWith": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_startsWith",
     "!doc": "Returns an array of results that start with the complete query as a phrase.\nCase-insensitive.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "startsWithCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_startsWithCase",
     "!doc": "Case-sensitive version of `startsWith()`.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "subWordMatch": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_subWordMatch",
     "!doc": "Returns an array of results in which all the words of the query match either\nwhole words or parts of words in the result. Non-word characters like\nwhitespace and certain punctuation are ignored. Case-insensitive.\n\nThis is basically a combination of `wordMatch()` (by ignoring whitespace and\nword order) and `phraseMatch()` (by allowing partial matching instead of\nrequiring the entire word to match).\n\nExample use case: Trying to find personal names independently of name order\n(Western or Eastern order) and supporting immediate feedback by allowing\npartial occurences. So queries like \"J. Doe\", \"Doe, John\", and \"J. D.\" would\nall match \"John Doe\".",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "subWordMatchCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_subWordMatchCase",
     "!doc": "Case-sensitive version of `subWordMatch()`.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "wordMatch": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_wordMatch",
     "!doc": "Returns an array of results that contain all of the words in the query, in\nany order. Non-word characters like whitespace and certain punctuation are\nignored. Case-insensitive.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    },
    "wordMatchCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteFilters.html#method_wordMatchCase",
     "!doc": "Case-sensitive version of `wordMatch()`.",
     "!data": {
      "submodule": "autocomplete-filters"
     }
    }
   },
   "AutoCompleteHighlighters": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html",
    "charMatchFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_charMatchFold",
     "!doc": "Accent-folding version of `charMatch()`.",
     "!data": {
      "submodule": "autocomplete-highlighters-accentfold"
     }
    },
    "phraseMatchFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_phraseMatchFold",
     "!doc": "Accent-folding version of `phraseMatch()`.",
     "!data": {
      "submodule": "autocomplete-highlighters-accentfold"
     }
    },
    "startsWithFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_startsWithFold",
     "!doc": "Accent-folding version of `startsWith()`.",
     "!data": {
      "submodule": "autocomplete-highlighters-accentfold"
     }
    },
    "subWordMatchFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_subWordMatchFold",
     "!doc": "Accent-folding version of `subWordMatch()`.",
     "!data": {
      "submodule": "autocomplete-highlighters-accentfold"
     }
    },
    "wordMatchFold": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_wordMatchFold",
     "!doc": "Accent-folding version of `wordMatch()`.",
     "!data": {
      "submodule": "autocomplete-highlighters-accentfold"
     }
    },
    "charMatch": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_charMatch",
     "!doc": "Highlights any individual query character that occurs anywhere in a result.\nCase-insensitive.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "charMatchCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_charMatchCase",
     "!doc": "Case-sensitive version of `charMatch()`.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "phraseMatch": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_phraseMatch",
     "!doc": "Highlights the complete query as a phrase anywhere within a result. Case-\ninsensitive.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "phraseMatchCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_phraseMatchCase",
     "!doc": "Case-sensitive version of `phraseMatch()`.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "startsWith": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_startsWith",
     "!doc": "Highlights the complete query as a phrase at the beginning of a result.\nCase-insensitive.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "startsWithCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_startsWithCase",
     "!doc": "Case-sensitive version of `startsWith()`.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "subWordMatch": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_subWordMatch",
     "!doc": "Highlights portions of results in which words from the query match either\nwhole words or parts of words in the result. Non-word characters like\nwhitespace and certain punctuation are ignored. Case-insensitive.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "subWordMatchCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_subWordMatchCase",
     "!doc": "Case-sensitive version of `subWordMatch()`.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "wordMatch": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_wordMatch",
     "!doc": "Highlights individual words in results that are also in the query. Non-word\ncharacters like punctuation are ignored. Case-insensitive.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    },
    "wordMatchCase": {
     "!type": "fn(query: string, results: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteHighlighters.html#method_wordMatchCase",
     "!doc": "Case-sensitive version of `wordMatch()`.",
     "!data": {
      "submodule": "autocomplete-highlighters"
     }
    }
   },
   "AutoCompleteList": {
    "!type": "fn(config: +config.AutoCompleteListConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html",
    "!data": {
     "extends": [
      "autocomplete.AutoCompleteBase",
      "widget_position.WidgetPosition",
      "widget_position_align.WidgetPositionAlign"
     ]
    },
    "prototype": {
     "!proto": "widget.Widget.prototype",
     "hide": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#method_hide",
      "!doc": "Hides the list, unless the `alwaysShowList` attribute is `true`.",
      "!data": {
       "submodule": "autocomplete-list"
      }
     },
     "selectItem": {
      "!type": "fn(itemNode?: +node.Node, originEvent?: +event_custom.EventFacade) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AutoCompleteList.html#method_selectItem",
      "!doc": "Selects the specified _itemNode_, or the current `activeItem` if _itemNode_\nis not specified.",
      "!data": {
       "submodule": "autocomplete-list"
      }
     }
    }
   }
  },
  "base": {
   "!data": {
    "submodules": {
     "base-base": {},
     "base-build": {},
     "base-core": {},
     "base-observable": {},
     "base-pluginhost": {}
    }
   },
   "Base": {
    "!type": "fn(config: +config.BaseConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html",
    "!data": {
     "extends": [
      "base.BaseCore",
      "base.BaseObservable",
      "attribute.AttributeCore",
      "attribute.AttributeObservable",
      "attribute.AttributeExtras",
      "event_custom.EventTarget"
     ]
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html#property_NAME",
     "!doc": "<p>\nThe string to be used to identify instances of\nthis class, for example in prefixing events.\n</p>\n<p>\nClasses extending Base, should define their own\nstatic NAME property, which should be camelCase by\nconvention (e.g. MyClass.NAME = \"myClass\";).\n</p>",
     "!data": {
      "submodule": "base-base"
     }
    },
    "ATTRS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html#property_ATTRS",
     "!doc": "The default set of attributes which will be available for instances of this class, and\ntheir configuration. In addition to the configuration properties listed by\nAttributes <a href=\"Attribute.html#method_addAttr\">addAttr</a> method, the attribute\ncan also be configured with a \"cloneDefaultValue\" property, which defines how the statically\ndefined value field should be protected (\"shallow\", \"deep\" and false are supported values).\n\nBy default if the value is an object literal or an array it will be \"shallow\" cloned, to\nprotect the default value.",
     "!data": {
      "submodule": "base-base"
     }
    },
    "modifyAttrs": {
     "!type": "fn(ctor?: fn(), configs: +Object)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html#method_modifyAttrs",
     "!doc": "Provides a way to safely modify a `Y.Base` subclass static `ATTRS` after\nthe class has been defined or created.\n\nBase-based classes cache information about the class hierarchy in order to\nefficiently create instances. This cache includes includes the aggregated\n`ATTRS` configs. If the static `ATTRS` configs need to be modified after the\nclass has been defined or create, then use this method which will make sure\nto clear any cached data before making any modifications.",
     "!data": {
      "submodule": "base-base"
     }
    },
    "build": {
     "!type": "fn(name: fn(), main: fn(), extensions: fn(), cfg: +Object) -> fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html#method_build",
     "!doc": "<p>\nBuilds a custom constructor function (class) from the\nmain function, and array of extension functions (classes)\nprovided. The NAME field for the constructor function is\ndefined by the first argument passed in.\n</p>\n<p>\nThe cfg object supports the following properties\n</p>\n<dl>\n   <dt>dynamic &#60;boolean&#62;</dt>\n   <dd>\n   <p>If true (default), a completely new class\n   is created which extends the main class, and acts as the\n   host on which the extension classes are augmented.</p>\n   <p>If false, the extensions classes are augmented directly to\n   the main class, modifying the main class prototype.</p>\n   </dd>\n   <dt>aggregates &#60;String[]&#62;</dt>\n   <dd>An array of static property names, which will get aggregated\n   on to the built class, in addition to the default properties build\n   will always aggregate as defined by the main class static _buildCfg\n   property.\n   </dd>\n</dl>",
     "!data": {
      "submodule": "base-build"
     }
    },
    "create": {
     "!type": "fn(name: string, main: fn(), extensions: fn(), px: +Object, sx: +Object) -> fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html#method_create",
     "!doc": "Creates a new class (constructor function) which extends the base class passed in as the second argument,\nand mixes in the array of extensions provided.\n\nPrototype properties or methods can be added to the new class, using the px argument (similar to Y.extend).\n\nStatic properties or methods can be added to the new class, using the sx argument (similar to Y.extend).\n\n**NOTE FOR COMPONENT DEVELOPERS**: Both the `base` class, and `extensions` can define static a `_buildCfg`\nproperty, which acts as class creation meta-data, and drives how special static properties from the base\nclass, or extensions should be copied, aggregated or (custom) mixed into the newly created class.\n\nThe `_buildCfg` property is a hash with 3 supported properties: `statics`, `aggregates` and `custom`, e.g:\n\n    // If the Base/Main class is the thing introducing the property:\n\n    MyBaseClass._buildCfg = {\n\n       // Static properties/methods to copy (Alias) to the built class.\n       statics: [\"CopyThisMethod\", \"CopyThisProperty\"],\n\n       // Static props to aggregate onto the built class.\n       aggregates: [\"AggregateThisProperty\"],\n\n       // Static properties which need custom handling (e.g. deep merge etc.)\n       custom: {\n          \"CustomProperty\" : function(property, Receiver, Supplier) {\n             ...\n             var triggers = Receiver.CustomProperty.triggers;\n             Receiver.CustomProperty.triggers = triggers.concat(Supplier.CustomProperty.triggers);\n             ...\n          }\n       }\n    };\n\n    MyBaseClass.CopyThisMethod = function() {...};\n    MyBaseClass.CopyThisProperty = \"foo\";\n    MyBaseClass.AggregateThisProperty = {...};\n    MyBaseClass.CustomProperty = {\n       triggers: [...]\n    }\n\n    // Or, if the Extension is the thing introducing the property:\n\n    MyExtension._buildCfg = {\n        statics : ...\n        aggregates : ...\n        custom : ...\n    }\n\nThis way, when users pass your base or extension class to `Y.Base.create` or `Y.Base.mix`, they dont need to\nknow which properties need special handling. `Y.Base` has a buildCfg which defines `ATTRS` for custom mix handling\n(to protect the static config objects), and `Y.Widget` has a buildCfg which specifies `HTML_PARSER` for\nstraight up aggregation.",
     "!data": {
      "submodule": "base-build"
     }
    },
    "mix": {
     "!type": "fn(main: fn(), extensions: fn()) -> fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html#method_mix",
     "!doc": "<p>Mixes in a list of extensions to an existing class.</p>",
     "!data": {
      "submodule": "base-build"
     }
    },
    "plug": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html#method_plug",
     "!doc": "Alias for <a href=\"Plugin.Host.html#method_Plugin.Host.plug\">Plugin.Host.plug</a>. See aliased\nmethod for argument and return value details.",
     "!data": {
      "submodule": "base-pluginhost"
     }
    },
    "unplug": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Base.html#method_unplug",
     "!doc": "Alias for <a href=\"Plugin.Host.html#method_Plugin.Host.unplug\">Plugin.Host.unplug</a>. See the\naliased method for argument and return value details.",
     "!data": {
      "submodule": "base-pluginhost"
     }
    }
   },
   "BaseCore": {
    "!type": "fn(cfg: +Object)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html",
    "!data": {
     "extends": [
      "attribute.AttributeCore"
     ]
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#property_NAME",
     "!doc": "The string to be used to identify instances of this class.\n\nClasses extending BaseCore, should define their own\nstatic NAME property, which should be camelCase by\nconvention (e.g. MyClass.NAME = \"myClass\";).",
     "!data": {
      "submodule": "base-core"
     }
    },
    "ATTRS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#property_ATTRS",
     "!doc": "The default set of attributes which will be available for instances of this class, and\ntheir configuration. In addition to the configuration properties listed by\nAttributeCores <a href=\"AttributeCore.html#method_addAttr\">addAttr</a> method,\nthe attribute can also be configured with a \"cloneDefaultValue\" property, which\ndefines how the statically defined value field should be protected\n(\"shallow\", \"deep\" and false are supported values).\n\nBy default if the value is an object literal or an array it will be \"shallow\"\ncloned, to protect the default value.",
     "!data": {
      "submodule": "base-core"
     }
    },
    "modifyAttrs": {
     "!type": "fn(ctor?: fn(), configs: +Object)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#method_modifyAttrs",
     "!doc": "Provides a way to safely modify a `Y.BaseCore` subclass static `ATTRS`\nafter the class has been defined or created.\n\nBaseCore-based classes cache information about the class hierarchy in order\nto efficiently create instances. This cache includes includes the aggregated\n`ATTRS` configs. If the static `ATTRS` configs need to be modified after the\nclass has been defined or create, then use this method which will make sure\nto clear any cached data before making any modifications.",
     "!data": {
      "submodule": "base-core"
     }
    },
    "prototype": {
     "name": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#property_name",
      "!doc": "The string used to identify the class of this object.",
      "!data": {
       "submodule": "base-core"
      }
     },
     "init": {
      "!type": "fn(cfg: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#method_init",
      "!doc": "Init lifecycle method, invoked during construction. Sets up attributes\nand invokes initializers for the class hierarchy.",
      "!data": {
       "submodule": "base-core"
      }
     },
     "destroy": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#method_destroy",
      "!doc": "Destroy lifecycle method. Invokes destructors for the class hierarchy.",
      "!data": {
       "submodule": "base-core"
      }
     },
     "toString": {
      "!type": "fn() -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseCore.html#method_toString",
      "!doc": "Default toString implementation. Provides the constructor NAME\nand the instance guid, if set.",
      "!data": {
       "submodule": "base-core"
      }
     }
    }
   },
   "BaseObservable": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseObservable.html",
    "!data": {
     "extends": [
      "attribute.AttributeObservable",
      "event_custom.EventTarget"
     ]
    },
    "prototype": {
     "init": {
      "!type": "fn(config: +config.BaseObservableInitConfig) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseObservable.html#method_init",
      "!doc": "Init lifecycle method, invoked during construction.\nFires the init event prior to setting up attributes and\ninvoking initializers for the class hierarchy.",
      "!data": {
       "submodule": "base-observable"
      }
     },
     "destroy": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/BaseObservable.html#method_destroy",
      "!doc": "<p>\nDestroy lifecycle method. Fires the destroy\nevent, prior to invoking destructors for the\nclass hierarchy.\n</p>\n<p>\nSubscribers to the destroy\nevent can invoke preventDefault on the event object, to prevent destruction\nfrom proceeding.\n</p>",
      "!data": {
       "submodule": "base-observable"
      }
     }
    }
   }
  },
  "button": {
   "Button": {
    "!type": "fn(config: +config.ButtonConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Button.html",
    "!data": {
     "extends": [
      "button_core.ButtonCore"
     ]
    },
    "prototype": {
     "!proto": "widget.Widget.prototype",
     "BOUNDING_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Button.html#property_BOUNDING_TEMPLATE",
      "!doc": "Bounding box template that will contain the Buttons DOM subtree."
     },
     "CONTENT_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Button.html#property_CONTENT_TEMPLATE",
      "!doc": "Content box template"
     }
    },
    "CLASS_NAMES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Button.html#property_CLASS_NAMES",
     "!doc": "List of class names used in the Buttons DOM"
    }
   },
   "ToggleButton": {
    "!type": "fn(config: +config.ToggleButtonConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html",
    "prototype": {
     "!proto": "button.Button.prototype",
     "trigger": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#property_trigger"
     },
     "selectedAttrName": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#property_selectedAttrName"
     },
     "initializer": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#method_initializer"
     },
     "destructor": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#method_destructor"
     },
     "bindUI": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#method_bindUI",
      "!doc": "Hooks up events for the widget"
     },
     "syncUI": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#method_syncUI",
      "!doc": "Syncs the UI for the widget"
     },
     "toggle": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#method_toggle",
      "!doc": "Toggles the selected/pressed/checked state of a ToggleButton"
     }
    },
    "CLASS_NAMES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ToggleButton.html#property_CLASS_NAMES",
     "!doc": "Array of static constants used to identify the classnames applied to DOM nodes"
    }
   }
  },
  "button_core": {
   "!data": {
    "module": "button-core"
   },
   "ButtonCore": {
    "!type": "fn(config: +config.ButtonCoreConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html",
    "!data": {
     "extends": [
      "attribute.AttributeCore"
     ]
    },
    "prototype": {
     "TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#property_TEMPLATE"
     },
     "enable": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#method_enable",
      "!doc": "Sets the buttons `disabled` DOM attribute to `false`"
     },
     "disable": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#method_disable",
      "!doc": "Sets the buttons `disabled` DOM attribute to `true`"
     },
     "getNode": {
      "!type": "fn() -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#method_getNode",
      "!doc": "Gets the buttons host node"
     },
     "label": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#config_label",
      "!doc": "The text of the buttons label"
     },
     "labelHTML": {
      "!type": "+HTML",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#config_labelHTML",
      "!doc": "The HTML of the buttons label\n\nThis attribute accepts HTML and inserts it into the DOM **without**\nsanitization.  This attribute should only be used with HTML that has\neither been escaped (using `Y.Escape.html`), or sanitized according to\nthe requirements of your application.\n\nIf all you need is support for text labels, please use the `label`\nattribute instead."
     },
     "disabled": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#config_disabled",
      "!doc": "The buttons enabled/disabled state"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#property_NAME",
     "!doc": "Name of this component."
    },
    "CLASS_NAMES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonCore.html#property_CLASS_NAMES",
     "!doc": "Array of static constants used to identify the classnames applied to DOM nodes"
    }
   }
  },
  "button_group": {
   "!data": {
    "module": "button-group"
   },
   "ButtonGroup": {
    "!type": "fn(config: +config.ButtonGroupConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html",
    "prototype": {
     "!proto": "widget.Widget.prototype",
     "renderUI": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html#method_renderUI",
      "!doc": "Creates a visual representation of the widget based on existing parameters."
     },
     "bindUI": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html#method_bindUI",
      "!doc": "Hooks up events for the widget"
     },
     "getButtons": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html#method_getButtons",
      "!doc": "Returns all buttons inside this this button group"
     },
     "getSelectedButtons": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html#method_getSelectedButtons",
      "!doc": "Returns all Y.Buttons instances that are selected"
     },
     "getSelectedValues": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html#method_getSelectedValues",
      "!doc": "Returns the values of all Y.Button instances that are selected"
     },
     "BUTTON_SELECTOR": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html#property_BUTTON_SELECTOR",
      "!doc": "Selector used to find buttons inside a ButtonGroup"
     }
    },
    "CLASS_NAMES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ButtonGroup.html#property_CLASS_NAMES",
     "!doc": "List of class names to use for ButtonGroups"
    }
   }
  },
  "button_plugin": {
   "!data": {
    "module": "button-plugin"
   },
   "Plugin": {
    "Button": {
     "!type": "fn(config: +config.Plugin.ButtonConfig)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Button.html",
     "prototype": {
      "!proto": "button_core.ButtonCore.prototype",
      "createNode": {
       "!type": "fn(node: +Object, config: +config.Plugin.ButtonCreateNodeConfig) -> +Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Button.html#method_createNode",
       "!doc": "A factory that plugs a Y.Node instance with Y.Plugin.Button"
      }
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Button.html#property_NAME",
      "!doc": "Name of this component."
     },
     "NS": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Button.html#property_NS",
      "!doc": "Namespace of this component."
     }
    }
   }
  },
  "cache": {
   "!data": {
    "submodules": {
     "cache-base": {},
     "cache-offline": {},
     "cache-plugin": {}
    }
   },
   "Cache": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "add": {
      "!type": "fn(request: +Object, response: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#method_add",
      "!doc": "Adds a new entry to the cache of the format\n{request:request, response:response, cached:cached, expires:expires}.\nIf cache is full, evicts the stalest entry before adding the new one.",
      "!data": {
       "submodule": "cache-base"
      }
     },
     "flush": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#method_flush",
      "!doc": "Flushes cache.",
      "!data": {
       "submodule": "cache-base"
      }
     },
     "retrieve": {
      "!type": "fn(request: +Object) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#method_retrieve",
      "!doc": "Retrieves cached object for given request, if available, and refreshes\nentry in the cache. Returns null if there is no cache match.",
      "!data": {
       "submodule": "cache-base"
      }
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cache.html#property_NAME",
     "!doc": "Class name.",
     "!data": {
      "submodule": "cache-base"
     }
    }
   },
   "CacheOffline": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html",
    "prototype": {
     "!proto": "cache.Cache.prototype",
     "add": {
      "!type": "fn(request: +Object, response: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html#method_add",
      "!doc": "Adds a new entry to the cache of the format\n{request:request, response:response, cached:cached, expires: expires}.",
      "!data": {
       "submodule": "cache-offline"
      }
     },
     "retrieve": {
      "!type": "fn(request: +Object) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html#method_retrieve",
      "!doc": "Retrieves cached object for given request, if available.\nReturns null if there is no cache match.",
      "!data": {
       "submodule": "cache-offline"
      }
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html#property_NAME",
     "!doc": "Class name.",
     "!data": {
      "submodule": "cache-offline"
     }
    },
    "flushAll": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CacheOffline.html#method_flushAll",
     "!doc": "Removes all items from all sandboxes. Useful if localStorage has\nexceeded quota. Only supported on browsers that implement HTML 5\nlocalStorage.",
     "!data": {
      "submodule": "cache-offline"
     }
    }
   },
   "Plugin": {
    "Cache": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Cache.html",
     "!data": {
      "extends": [
       "plugin.Plugin.Base"
      ]
     },
     "prototype": {
      "!proto": "cache.Cache.prototype"
     },
     "NS": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Cache.html#property_NS",
      "!doc": "The namespace for the plugin. This will be the property on the host which\nreferences the plugin instance.",
      "!data": {
       "submodule": "cache-plugin"
      }
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Cache.html#property_NAME",
      "!doc": "Class name.",
      "!data": {
       "submodule": "cache-plugin"
      }
     }
    }
   }
  },
  "calendar": {
   "!data": {
    "submodules": {
     "calendar-base": {}
    }
   },
   "CalendarBase": {
    "!type": "fn(config: +config.CalendarBaseConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html",
    "prototype": {
     "!proto": "widget.Widget.prototype",
     "initializer": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#method_initializer",
      "!doc": "Designated initializer\nInitializes instance-level properties of\ncalendar.",
      "!data": {
       "submodule": "calendar-base"
      }
     },
     "renderUI": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#method_renderUI",
      "!doc": "renderUI implementation\n\nCreates a visual representation of the calendar based on existing parameters.",
      "!data": {
       "submodule": "calendar-base"
      }
     },
     "bindUI": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#method_bindUI",
      "!doc": "bindUI implementation\n\nAssigns listeners to relevant events that change the state\nof the calendar.",
      "!data": {
       "submodule": "calendar-base"
      }
     },
     "selectDates": {
      "!type": "fn(dates: +datatype_date.Date|[?]) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#method_selectDates",
      "!doc": "Selects a given date or array of dates.",
      "!data": {
       "submodule": "calendar-base"
      }
     },
     "deselectDates": {
      "!type": "fn(dates?: +datatype_date.Date|[?]) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CalendarBase.html#method_deselectDates",
      "!doc": "Deselects a given date or array of dates, or deselects\nall dates if no argument is specified.",
      "!data": {
       "submodule": "calendar-base"
      }
     }
    }
   },
   "Calendar": {
    "!type": "fn(config: +config.CalendarConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html",
    "prototype": {
     "!proto": "calendar.CalendarBase.prototype",
     "initializer": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#method_initializer",
      "!doc": "Designated initializer. Activates the navigation plugin for the calendar."
     },
     "subtractMonth": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#method_subtractMonth",
      "!doc": "Subtracts one month from the current calendar view."
     },
     "subtractYear": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#method_subtractYear",
      "!doc": "Subtracts one year from the current calendar view."
     },
     "addMonth": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#method_addMonth",
      "!doc": "Adds one month to the current calendar view."
     },
     "addYear": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Calendar.html#method_addYear",
      "!doc": "Adds one year to the current calendar view."
     }
    }
   }
  },
  "calendarnavigator": {
   "Plugin": {
    "CalendarNavigator": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CalendarNavigator.html",
     "prototype": {
      "!proto": "plugin.Plugin.Base.prototype",
      "initializer": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CalendarNavigator.html#method_initializer",
       "!doc": "The initializer lifecycle implementation. Modifies the host widgets\nrender to add navigation controls."
      },
      "destructor": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CalendarNavigator.html#method_destructor",
       "!doc": "The initializer destructor implementation. Responsible for destroying the initialized\ncontrol mechanisms."
      }
     },
     "NS": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CalendarNavigator.html#property_NS",
      "!doc": "The namespace for the plugin. This will be the property on the widget, which will\nreference the plugin instance, when its plugged in."
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CalendarNavigator.html#property_NAME",
      "!doc": "The NAME of the CalendarNavigator class. Used to prefix events generated\nby the plugin class."
     },
     "ATTRS": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CalendarNavigator.html#property_ATTRS",
      "!doc": "Static property used to define the default attribute\nconfiguration for the plugin."
     }
    }
   }
  },
  "charts": {
   "!data": {
    "submodules": {
     "series-area": {},
     "series-areaspline": {},
     "axis": {},
     "axis-base": {},
     "series-bar": {},
     "series-candlestick": {},
     "charts-base": {},
     "charts-legend": {},
     "series-cartesian": {},
     "series-base": {},
     "axis-category": {},
     "axis-category-base": {},
     "series-column": {},
     "series-combo": {},
     "series-combospline": {},
     "series-curve-util": {},
     "series-fill-util": {},
     "series-histogram": {},
     "series-line": {},
     "series-line-util": {},
     "series-marker": {},
     "axis-numeric": {},
     "axis-numeric-base": {},
     "series-ohlc": {},
     "series-pie": {},
     "series-plot-util": {},
     "series-range": {},
     "series-spline": {},
     "series-area-stacked": {},
     "series-areaspline-stacked": {},
     "axis-stacked": {},
     "axis-stacked-base": {},
     "series-bar-stacked": {},
     "series-column-stacked": {},
     "series-combo-stacked": {},
     "series-combospline-stacked": {},
     "series-line-stacked": {},
     "series-marker-stacked": {},
     "series-spline-stacked": {},
     "series-stacked": {},
     "axis-time": {},
     "axis-time-base": {}
    }
   },
   "Axis": {
    "!type": "fn(config: +config.AxisConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html",
    "!data": {
     "extends": [
      "charts.AxisBase",
      "charts.TopAxisLayout",
      "charts.RightAxisLayout",
      "charts.BottomAxisLayout",
      "charts.LeftAxisLayout"
     ]
    },
    "prototype": {
     "!proto": "widget.Widget.prototype",
     "getLabelByIndex": {
      "!type": "fn(i: number, l: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#method_getLabelByIndex",
      "!doc": "Calculates and returns a value based on the number of labels and the index of\nthe current label.",
      "!data": {
       "submodule": "axis"
      }
     },
     "getMaxLabelBounds": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#method_getMaxLabelBounds",
      "!doc": "Returns the coordinates (top, right, bottom, left) for the bounding box of the last label.",
      "!data": {
       "submodule": "axis"
      }
     },
     "getMinLabelBounds": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#method_getMinLabelBounds",
      "!doc": "Returns the coordinates (top, right, bottom, left) for the bounding box of the first label.",
      "!data": {
       "submodule": "axis"
      }
     },
     "getTotalMajorUnits": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#method_getTotalMajorUnits",
      "!doc": "Returns the total number of majorUnits that will appear on an axis.",
      "!data": {
       "submodule": "axis"
      }
     },
     "getMajorUnitDistance": {
      "!type": "fn(len: number, uiLen: number, majorUnit: +Object) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#method_getMajorUnitDistance",
      "!doc": "Returns the distance between major units on an axis.",
      "!data": {
       "submodule": "axis"
      }
     },
     "getMinimumValue": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#method_getMinimumValue",
      "!doc": "Returns a string corresponding to the first label on an\naxis.",
      "!data": {
       "submodule": "axis"
      }
     },
     "getMaximumValue": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Axis.html#method_getMaximumValue",
      "!doc": "Returns a string corresponding to the last label on an\naxis.",
      "!data": {
       "submodule": "axis"
      }
     }
    }
   },
   "AxisBase": {
    "!type": "fn(config: +config.AxisBaseConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html",
    "!data": {
     "extends": [
      "charts.Renderer"
     ]
    },
    "prototype": {
     "!proto": "base.Base.prototype",
     "getOrigin": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#method_getOrigin",
      "!doc": "Returns the value corresponding to the origin on the axis.",
      "!data": {
       "submodule": "axis-base"
      }
     },
     "addKey": {
      "!type": "fn(value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#method_addKey",
      "!doc": "Adds an array to the key hash.",
      "!data": {
       "submodule": "axis-base"
      }
     },
     "removeKey": {
      "!type": "fn(value: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#method_removeKey",
      "!doc": "Removes an array from the key hash.",
      "!data": {
       "submodule": "axis-base"
      }
     },
     "getKeyValueAt": {
      "!type": "fn(key: string, index: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#method_getKeyValueAt",
      "!doc": "Returns a value based of a key value and an index.",
      "!data": {
       "submodule": "axis-base"
      }
     },
     "getDataByKey": {
      "!type": "fn(value: string|[?]) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#method_getDataByKey",
      "!doc": "Returns values based on key identifiers. When a string is passed as an argument, an array of values is returned.\nWhen an array of keys is passed as an argument, an object literal with an array of values mapped to each key is\nreturned.",
      "!data": {
       "submodule": "axis-base"
      }
     },
     "getTotalMajorUnits": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#method_getTotalMajorUnits",
      "!doc": "Returns the total number of majorUnits that will appear on an axis.",
      "!data": {
       "submodule": "axis-base"
      }
     },
     "getEdgeOffset": {
      "!type": "fn(ct: number, l: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AxisBase.html#method_getEdgeOffset",
      "!doc": "Gets the distance that the first and last ticks are offset from there respective\nedges.",
      "!data": {
       "submodule": "axis-base"
      }
     }
    }
   },
   "CartesianChart": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html",
    "prototype": {
     "!proto": "charts.ChartBase.prototype",
     "_addToAxesCollection": {
      "!type": "fn(position: string, axis: +charts.Axis)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#method__addToAxesCollection",
      "!doc": "Adds axis instance to the appropriate array based on position",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "getSeriesItems": {
      "!type": "fn(series: +charts.CartesianSeries, index: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#method_getSeriesItems",
      "!doc": "Returns an object literal containing a categoryItem and a valueItem for a given series index. Below is the structure of each:",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "_getAriaMessage": {
      "!type": "fn(key: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CartesianChart.html#method__getAriaMessage",
      "!doc": "Returns the appropriate message based on the key press.",
      "!data": {
       "submodule": "charts-base"
      }
     }
    }
   },
   "CategoryAxis": {
    "!type": "fn(config: +config.CategoryAxisConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryAxis.html",
    "!data": {
     "extends": [
      "charts.CategoryImpl"
     ]
    },
    "prototype": {
     "!proto": "charts.Axis.prototype",
     "getMinimumValue": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryAxis.html#method_getMinimumValue",
      "!doc": "Returns a string corresponding to the first label on an\naxis.",
      "!data": {
       "submodule": "axis-category"
      }
     },
     "getMaximumValue": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryAxis.html#method_getMaximumValue",
      "!doc": "Returns a string corresponding to the last label on an\naxis.",
      "!data": {
       "submodule": "axis-category"
      }
     }
    }
   },
   "CategoryImpl": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryImpl.html",
    "prototype": {
     "formatLabel": {
      "!type": "fn(value: +Object) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryImpl.html#method_formatLabel",
      "!doc": "Formats a label based on the axis type and optionally specified format.",
      "!data": {
       "submodule": "axis-category-base"
      }
     },
     "getDataByKey": {
      "!type": "fn(value: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryImpl.html#method_getDataByKey",
      "!doc": "Returns an array of values based on an identifier key.",
      "!data": {
       "submodule": "axis-category-base"
      }
     },
     "getTotalMajorUnits": {
      "!type": "fn(majorUnit: +Object, len: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryImpl.html#method_getTotalMajorUnits",
      "!doc": "Returns the total number of majorUnits that will appear on an axis.",
      "!data": {
       "submodule": "axis-category-base"
      }
     },
     "getKeyValueAt": {
      "!type": "fn(key: string, index: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CategoryImpl.html#method_getKeyValueAt",
      "!doc": "Returns a value based of a key value and an index.",
      "!data": {
       "submodule": "axis-category-base"
      }
     }
    }
   },
   "ChartBase": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html",
    "prototype": {
     "getSeries": {
      "!type": "fn(val: ?) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#method_getSeries",
      "!doc": "Returns a series instance by index or key value.",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "getAxisByKey": {
      "!type": "fn(val: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#method_getAxisByKey",
      "!doc": "Returns an `Axis` instance by key reference. If the axis was explicitly set through the `axes` attribute,\nthe key will be the same as the key used in the `axes` object. For default axes, the key for\nthe category axis is the value of the `categoryKey` (`category`). For the value axis, the default\nkey is `values`.",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "getCategoryAxis": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#method_getCategoryAxis",
      "!doc": "Returns the category axis for the chart.",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "toggleTooltip": {
      "!type": "fn(e: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#method_toggleTooltip",
      "!doc": "Event listener for toggling the tooltip. If a tooltip is visible, hide it. If not, it\nwill create and show a tooltip based on the event object.",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "hideTooltip": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#method_hideTooltip",
      "!doc": "Hides the default tooltip",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "_getAllKeys": {
      "!type": "fn(dp: [?]) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ChartBase.html#method__getAllKeys",
      "!doc": "Returns all the keys contained in a  `dataProvider`.",
      "!data": {
       "submodule": "charts-base"
      }
     }
    }
   },
   "Graph": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html",
    "!data": {
     "extends": [
      "charts.Renderer"
     ]
    },
    "prototype": {
     "!proto": "widget.Widget.prototype",
     "getSeriesByIndex": {
      "!type": "fn(val: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#method_getSeriesByIndex",
      "!doc": "Returns a series instance based on an index.",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "getSeriesByKey": {
      "!type": "fn(val: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graph.html#method_getSeriesByKey",
      "!doc": "Returns a series instance based on a key value.",
      "!data": {
       "submodule": "charts-base"
      }
     }
    }
   },
   "NumericImpl": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html",
    "prototype": {
     "": {
      "!type": "fn(value: +Object, format: +Object) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html",
      "!doc": "Formats a label based on the axis type and optionally specified format.",
      "!data": {
       "submodule": "axis-numeric-base"
      }
     },
     "getTotalByKey": {
      "!type": "fn(key: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html#method_getTotalByKey",
      "!doc": "Returns the sum of all values per key.",
      "!data": {
       "submodule": "axis-numeric-base"
      }
     },
     "getOrigin": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/NumericImpl.html#method_getOrigin",
      "!doc": "Returns the value corresponding to the origin on the axis.",
      "!data": {
       "submodule": "axis-numeric-base"
      }
     }
    }
   },
   "PieChart": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/PieChart.html",
    "prototype": {
     "!proto": "charts.ChartBase.prototype",
     "getSeriesItem": {
      "!type": "fn(series: ?, index: ?) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/PieChart.html#method_getSeriesItem",
      "!doc": "Returns an object literal containing a categoryItem and a valueItem for a given series index.",
      "!data": {
       "submodule": "charts-base"
      }
     },
     "_getAriaMessage": {
      "!type": "fn(key: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/PieChart.html#method__getAriaMessage",
      "!doc": "Returns the appropriate message based on the key press.",
      "!data": {
       "submodule": "charts-base"
      }
     }
    }
   },
   "PieSeries": {
    "!type": "fn(config: +config.PieSeriesConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html",
    "!data": {
     "extends": [
      "charts.Plots"
     ]
    },
    "prototype": {
     "!proto": "charts.SeriesBase.prototype",
     "getTotalValues": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/PieSeries.html#method_getTotalValues",
      "!doc": "Returns the sum of all values for the series.",
      "!data": {
       "submodule": "series-pie"
      }
     }
    }
   },
   "SeriesBase": {
    "!type": "fn(config: +config.SeriesBaseConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html",
    "!data": {
     "extends": [
      "charts.Renderer"
     ]
    },
    "prototype": {
     "!proto": "base.Base.prototype",
     "getTotalValues": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SeriesBase.html#method_getTotalValues",
      "!doc": "Returns the sum of all values for the series.",
      "!data": {
       "submodule": "series-base"
      }
     }
    }
   },
   "TimeImpl": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/TimeImpl.html",
    "prototype": {
     "formatLabel": {
      "!type": "fn(value: +Object, format: +Object) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/TimeImpl.html#method_formatLabel",
      "!doc": "Formats a label based on the axis type and optionally specified format.",
      "!data": {
       "submodule": "axis-time-base"
      }
     }
    }
   }
  },
  "classnamemanager": {
   "ClassNameManager": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ClassNameManager.html",
    "classNamePrefix": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ClassNameManager.html#property_classNamePrefix",
     "!doc": "Configuration property indicating the prefix for all CSS class names in this YUI instance."
    },
    "classNameDelimiter": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ClassNameManager.html#property_classNameDelimiter",
     "!doc": "Configuration property indicating the delimiter used to compose all CSS class names in\nthis YUI instance."
    },
    "prototype": {
     "getClassName": {
      "!type": "fn(classnameSection?: string, skipPrefix: bool)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ClassNameManager.html#method_getClassName",
      "!doc": "Returns a class name prefixed with the value of the\n<code>Y.config.classNamePrefix</code> attribute + the provided strings.\nUses the <code>Y.config.classNameDelimiter</code> attribute to delimit the\nprovided strings. E.g. Y.ClassNameManager.getClassName(foo,bar); // yui-foo-bar"
     }
    }
   }
  },
  "collection": {
   "!data": {
    "submodules": {
     "array-extras": {},
     "arraylist-add": {},
     "arraylist-filter": {},
     "arraylist": {},
     "array-invoke": {}
    }
   },
   "Array": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html",
    "!data": {
     "for": "yui.Array"
    },
    "lastIndexOf": {
     "!type": "fn(a: [?], val: ?, fromIndex?: number) -> number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_lastIndexOf",
     "!doc": "Returns the index of the last item in the array that contains the specified\nvalue, or `-1` if the value isnt found.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "unique": {
     "!type": "fn(array: [?], testFn?: fn(a: ?, b: ?, index: number, array: [?])) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_unique",
     "!doc": "Returns a copy of the input array with duplicate items removed.\n\nNote: If the input array only contains strings, the `Y.Array.dedupe()` method is\na much faster alternative.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "filter": {
     "!type": "fn(a: [?], f: fn(), o?: +Object) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_filter",
     "!doc": "Executes the supplied function on each item in the array. Returns a new array\ncontaining the items for which the supplied function returned a truthy value.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "reject": {
     "!type": "fn(a: [?], f: fn(), o?: +Object) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_reject",
     "!doc": "The inverse of `Array.filter()`. Executes the supplied function on each item.\nReturns a new array containing the items for which the supplied function\nreturned `false`.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "every": {
     "!type": "fn(a: [?], f: fn(), o?: +Object) -> bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_every",
     "!doc": "Executes the supplied function on each item in the array. Iteration stops if the\nsupplied function does not return a truthy value.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "map": {
     "!type": "fn(a: [?], f: fn(), o?: +Object) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_map",
     "!doc": "Executes the supplied function on each item in the array and returns a new array\ncontaining all the values returned by the supplied function.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "reduce": {
     "!type": "fn(a: [?], init: ?, f: fn(previousValue: ?, currentValue: ?, index: number, array: [?]), o?: +Object) -> ?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_reduce",
     "!doc": "Executes the supplied function on each item in the array, \"folding\" the array\ninto a single value.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "find": {
     "!type": "fn(a: [?], f: fn(), o?: +Object) -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_find",
     "!doc": "Executes the supplied function on each item in the array, searching for the\nfirst item that matches the supplied function.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "grep": {
     "!type": "fn(a: [?], pattern: +RegExp) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_grep",
     "!doc": "Iterates over an array, returning a new array of all the elements that match the\nsupplied regular expression.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "partition": {
     "!type": "fn(a: [?], f: fn(item: ?, index: number, array: [?]), o?: +Object) -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_partition",
     "!doc": "Partitions an array into two new arrays, one with the items for which the\nsupplied function returns `true`, and one with the items for which the function\nreturns `false`.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "zip": {
     "!type": "fn(a: [?], a2: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_zip",
     "!doc": "Creates an array of arrays by pairing the corresponding elements of two arrays\ntogether into a new array.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "flatten": {
     "!type": "fn(a: [?]) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_flatten",
     "!doc": "Flattens an array of nested arrays at any abitrary depth into a single, flat\narray.",
     "!data": {
      "submodule": "array-extras"
     }
    },
    "invoke": {
     "!type": "fn(items: [?], name: string, args?: ?) -> [?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Array.html#method_invoke",
     "!doc": "Executes a named method on each item in an array of objects. Items in the array\nthat do not have a function by that name will be skipped.",
     "!data": {
      "submodule": "array-invoke"
     }
    }
   },
   "ArrayList": {
    "!type": "fn(items: [?])",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html",
    "prototype": {
     "add": {
      "!type": "fn(item: +Mixed, index: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_add",
      "!doc": "Add a single item to the ArrayList.  Does not prevent duplicates.",
      "!data": {
       "submodule": "arraylist-add"
      }
     },
     "remove": {
      "!type": "fn(needle: +Mixed, all: bool, comparator: fn()) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_remove",
      "!doc": "Removes first or all occurrences of an item to the ArrayList.  If a\ncomparator is not provided, uses itemsAreEqual method to determine\nmatches.",
      "!data": {
       "submodule": "arraylist-add"
      }
     },
     "itemsAreEqual": {
      "!type": "fn(a: +Mixed, b: +Mixed) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_itemsAreEqual",
      "!doc": "Default comparator for items stored in this list.  Used by remove().",
      "!data": {
       "submodule": "arraylist-add"
      }
     },
     "filter": {
      "!type": "fn(validator: fn()) -> +collection.ArrayList",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_filter",
      "!doc": "<p>Create a new ArrayList (or augmenting class instance) from a subset\nof items as determined by the boolean function passed as the\nargument.  The original ArrayList is unchanged.</p>\n\n<p>The validator signature is <code>validator( item )</code>.</p>",
      "!data": {
       "submodule": "arraylist-filter"
      }
     },
     "item": {
      "!type": "fn(i: +Integer) -> +Mixed",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_item",
      "!doc": "Get an item by index from the list.  Override this method if managing a\nlist of objects that have a different public representation (e.g. Node\ninstances vs DOM nodes).  The iteration methods that accept a user\nfunction will use this method for access list items for operation.",
      "!data": {
       "submodule": "arraylist"
      }
     },
     "each": {
      "!type": "fn(fn: fn(), context: +Mixed) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_each",
      "!doc": "<p>Execute a function on each item of the list, optionally providing a\ncustom execution context.  Default context is the item.</p>\n\n<p>The callback signature is <code>callback( item, index )</code>.</p>",
      "!data": {
       "submodule": "arraylist"
      }
     },
     "some": {
      "!type": "fn(fn: fn(), context: +Mixed) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_some",
      "!doc": "<p>Execute a function on each item of the list, optionally providing a\ncustom execution context.  Default context is the item.</p>\n\n<p>The callback signature is <code>callback( item, index )</code>.</p>\n\n<p>Unlike <code>each</code>, if the callback returns true, the\niteration will stop.</p>",
      "!data": {
       "submodule": "arraylist"
      }
     },
     "indexOf": {
      "!type": "fn(needle: +Mixed) -> +Integer",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_indexOf",
      "!doc": "Finds the first index of the needle in the managed array of items.",
      "!data": {
       "submodule": "arraylist"
      }
     },
     "size": {
      "!type": "fn() -> +Integer",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_size",
      "!doc": "How many items are in this list?",
      "!data": {
       "submodule": "arraylist"
      }
     },
     "isEmpty": {
      "!type": "fn() -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_isEmpty",
      "!doc": "Is this instance managing any items?",
      "!data": {
       "submodule": "arraylist"
      }
     },
     "toJSON": {
      "!type": "fn() -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_toJSON",
      "!doc": "Provides an array-like representation for JSON.stringify.",
      "!data": {
       "submodule": "arraylist"
      }
     }
    },
    "addMethod": {
     "!type": "fn(dest: +Object, name: string|[string])",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ArrayList.html#method_addMethod",
     "!doc": "<p>Adds a pass through method to dest (typically the prototype of a list\nclass) that calls the named method on each item in the list with\nwhatever parameters are passed in.  Allows for API indirection via list\ninstances.</p>\n\n<p>Accepts a single string name or an array of string names.</p>\n\n<pre><code>list.each( function ( item ) {\n    item.methodName( 1, 2, 3 );\n} );\n// becomes\nlist.methodName( 1, 2, 3 );</code></pre>\n\n<p>Additionally, the pass through methods use the item retrieved by the\n<code>_item</code> method in case there is any special behavior that is\nappropriate for API mirroring.</p>\n\n<p>If the iterated method returns a value, the return value from the\nadded method will be an array of values with each value being at the\ncorresponding index for that item.  If the iterated method does not\nreturn a value, the added method will be chainable.",
     "!data": {
      "submodule": "arraylist"
     }
    }
   }
  },
  "color": {
   "!data": {
    "submodules": {
     "color-base": {},
     "color-harmony": {},
     "color-hsl": {},
     "color-hsv": {}
    }
   },
   "Color": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html",
    "KEYWORDS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_KEYWORDS",
     "!data": {
      "submodule": "color-base"
     }
    },
    "REGEX_HEX": {
     "!type": "+RegExp",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_REGEX_HEX",
     "!doc": "NOTE: `(\\ufffe)?` is added to the Regular Expression to carve out a\nplace for the alpha channel that is returned from toArray\nwithout compromising any usage of the Regular Expression",
     "!data": {
      "submodule": "color-base"
     }
    },
    "REGEX_HEX3": {
     "!type": "+RegExp",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_REGEX_HEX3",
     "!doc": "NOTE: `(\\ufffe)?` is added to the Regular Expression to carve out a\nplace for the alpha channel that is returned from toArray\nwithout compromising any usage of the Regular Expression",
     "!data": {
      "submodule": "color-base"
     }
    },
    "REGEX_RGB": {
     "!type": "+RegExp",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_REGEX_RGB",
     "!data": {
      "submodule": "color-base"
     }
    },
    "STR_HEX": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_STR_HEX",
     "!data": {
      "submodule": "color-base"
     }
    },
    "STR_RGB": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_STR_RGB",
     "!data": {
      "submodule": "color-base"
     }
    },
    "STR_RGBA": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_STR_RGBA",
     "!data": {
      "submodule": "color-base"
     }
    },
    "TYPES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_TYPES",
     "!data": {
      "submodule": "color-base"
     }
    },
    "CONVERTS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#property_CONVERTS",
     "!data": {
      "submodule": "color-base"
     }
    },
    "prototype": {
     "convert": {
      "!type": "fn(str: string, to: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#method_convert",
      "!doc": "Converts the provided string to the provided type.\nYou can use the `Y.Color.TYPES` to get a valid `to` type.\nIf the color cannot be converted, the original color will be returned.",
      "!data": {
       "submodule": "color-base"
      }
     },
     "toHex": {
      "!type": "fn(str: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#method_toHex",
      "!doc": "Converts provided color value to a hex value string",
      "!data": {
       "submodule": "color-base"
      }
     },
     "toRGB": {
      "!type": "fn(str: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#method_toRGB",
      "!doc": "Converts provided color value to an RGB value string",
      "!data": {
       "submodule": "color-base"
      }
     },
     "toRGBA": {
      "!type": "fn(str: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#method_toRGBA",
      "!doc": "Converts provided color value to an RGB value string",
      "!data": {
       "submodule": "color-base"
      }
     },
     "toArray": {
      "!type": "fn(str: string) -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#method_toArray",
      "!doc": "Converts the provided color string to an array of values where the\n    last value is the alpha value. Will return an empty array if\n    the provided string is not able to be parsed.\n\n    NOTE: `(\\ufffe)?` is added to `HEX` and `HEX3` Regular Expressions to\n    carve out a place for the alpha channel that is returned from\n    toArray without compromising any usage of the Regular Expression\n\n    Y.Color.toArray(fff);              // [ff, ff, ff, 1]\n    Y.Color.toArray(rgb(0, 0, 0));     // [0, 0, 0, 1]\n    Y.Color.toArray(rgba(0, 0, 0, 0)); // [0, 0, 0, 1]",
      "!data": {
       "submodule": "color-base"
      }
     },
     "fromArray": {
      "!type": "fn(arr: [?], template: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#method_fromArray",
      "!doc": "Converts the array of values to a string based on the provided template.",
      "!data": {
       "submodule": "color-base"
      }
     },
     "findType": {
      "!type": "fn(str: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.html#method_findType",
      "!doc": "Finds the value type based on the str value provided.",
      "!data": {
       "submodule": "color-base"
      }
     }
    },
    "Harmony": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html",
     "prototype": {
      "getComplementary": {
       "!type": "fn(str: string, to?: string) -> [?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getComplementary",
       "!doc": "Returns an Array of two colors. The first color in the Array\n  will be the color passed in. The second will be the\n  complementary color of the color provided",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getSplit": {
       "!type": "fn(str: string, offset?: number, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getSplit",
       "!doc": "Returns an Array of three colors. The first color in the Array\n  will be the color passed in. The second two will be split\n  complementary colors.",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getAnalogous": {
       "!type": "fn(str: string, offset?: number, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getAnalogous",
       "!doc": "Returns an Array of five colors. The first color in the Array\n  will be the color passed in. The remaining four will be\n  analogous colors two in either direction from the initially\n  provided color.",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getTriad": {
       "!type": "fn(str: string, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getTriad",
       "!doc": "Returns an Array of three colors. The first color in the Array\n  will be the color passed in. The second two will be equidistant\n  from the start color and each other.",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getTetrad": {
       "!type": "fn(str: string, offset?: number, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getTetrad",
       "!doc": "Returns an Array of four colors. The first color in the Array\n  will be the color passed in. The remaining three colors are\n  equidistant offsets from the starting color and each other.",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getSquare": {
       "!type": "fn(str: string, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getSquare",
       "!doc": "Returns an Array of four colors. The first color in the Array\n  will be the color passed in. The remaining three colors are\n  equidistant offsets from the starting color and each other.",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getMonochrome": {
       "!type": "fn(str: string, count?: number, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getMonochrome",
       "!doc": "Calculates lightness offsets resulting in a monochromatic Array\n  of values.",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getSimilar": {
       "!type": "fn(str: string, offset?: number, count?: number, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getSimilar",
       "!doc": "Creates an Array of similar colors. Returned Array is prepended\n   with the color provided followed a number of colors decided\n   by count",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getOffset": {
       "!type": "fn(str: string, adjust: +config.Color.HarmonyGetOffsetConfig, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getOffset",
       "!doc": "Adjusts the provided color by the offset(s) given. You may\n  adjust hue, saturation, and/or luminance in one step.",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getBrightness": {
       "!type": "fn(str: string) -> number",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getBrightness",
       "!doc": "Returns 0 - 100 percentage of brightness from `0` (black) being the\n  darkest to `100` (white) being the brightest.",
       "!data": {
        "submodule": "color-harmony"
       }
      },
      "getSimilarBrightness": {
       "!type": "fn(str: string, match: string, to?: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.Harmony.html#method_getSimilarBrightness",
       "!doc": "Returns a new color value with adjusted luminance so that the\n  brightness of the return color matches the perceived brightness\n  of the `match` color provided.",
       "!data": {
        "submodule": "color-harmony"
       }
      }
     }
    },
    "HSL": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSL.html",
     "REGEX_HSL": {
      "!type": "+RegExp",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSL.html#property_REGEX_HSL",
      "!data": {
       "submodule": "color-hsl"
      }
     },
     "STR_HSL": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSL.html#property_STR_HSL",
      "!data": {
       "submodule": "color-hsl"
      }
     },
     "STR_HSLA": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSL.html#property_STR_HSLA",
      "!data": {
       "submodule": "color-hsl"
      }
     },
     "prototype": {
      "toHSL": {
       "!type": "fn(str: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSL.html#method_toHSL",
       "!doc": "Converts provided color value to an HSL string.",
       "!data": {
        "submodule": "color-hsl"
       }
      },
      "toHSLA": {
       "!type": "fn(str: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSL.html#method_toHSLA",
       "!doc": "Converts provided color value to an HSLA string.",
       "!data": {
        "submodule": "color-hsl"
       }
      }
     }
    },
    "HSV": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSV.html",
     "REGEX_HSV": {
      "!type": "+RegExp",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSV.html#property_REGEX_HSV",
      "!data": {
       "submodule": "color-hsv"
      }
     },
     "STR_HSV": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSV.html#property_STR_HSV",
      "!data": {
       "submodule": "color-hsv"
      }
     },
     "STR_HSVA": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSV.html#property_STR_HSVA",
      "!data": {
       "submodule": "color-hsv"
      }
     },
     "prototype": {
      "toHSV": {
       "!type": "fn(str: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSV.html#method_toHSV",
       "!doc": "Converts provided color value to an HSV string.",
       "!data": {
        "submodule": "color-hsv"
       }
      },
      "toHSVA": {
       "!type": "fn(str: string) -> string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Color.HSV.html#method_toHSVA",
       "!doc": "Converts provided color value to an HSVA string.",
       "!data": {
        "submodule": "color-hsv"
       }
      }
     }
    }
   }
  },
  "console_filters": {
   "!data": {
    "module": "console-filters"
   },
   "Plugin": {
    "ConsoleFilters": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html",
     "prototype": {
      "syncUI": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#method_syncUI",
       "!doc": "Updates the UI to be in accordance with the current state of the plugin."
      },
      "refreshConsole": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#method_refreshConsole",
       "!doc": "Repopulates the Console with entries appropriate to the current filter\nsettings."
      },
      "hideCategory": {
       "!type": "fn(cat: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#method_hideCategory",
       "!doc": "Hides any number of categories from the UI.  Convenience method for\nmyConsole.filter.set(category.foo, false); set(category.bar, false);\nand so on."
      },
      "showCategory": {
       "!type": "fn(cat: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#method_showCategory",
       "!doc": "Shows any number of categories in the UI.  Convenience method for\nmyConsole.filter.set(category.foo, true); set(category.bar, true);\nand so on."
      },
      "hideSource": {
       "!type": "fn(src: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#method_hideSource",
       "!doc": "Hides any number of sources from the UI.  Convenience method for\nmyConsole.filter.set(source.foo, false); set(source.bar, false);\nand so on."
      },
      "showSource": {
       "!type": "fn(src: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#method_showSource",
       "!doc": "Shows any number of sources in the UI.  Convenience method for\nmyConsole.filter.set(source.foo, true); set(source.bar, true);\nand so on."
      }
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#property_NAME",
      "!doc": "Plugin name."
     },
     "NS": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#property_NS",
      "!doc": "The namespace hung off the host object that this plugin will inhabit."
     },
     "CATEGORIES_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#property_CATEGORIES_TEMPLATE",
      "!doc": "Markup template used to create the container for the category filters."
     },
     "SOURCES_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#property_SOURCES_TEMPLATE",
      "!doc": "Markup template used to create the container for the source filters."
     },
     "FILTER_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ConsoleFilters.html#property_FILTER_TEMPLATE",
      "!doc": "Markup template used to create the category and source filter checkboxes."
     }
    }
   }
  },
  "console": {
   "Console": {
    "!type": "fn(config?: +config.ConsoleConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html",
    "prototype": {
     "!proto": "widget.Widget.prototype",
     "log": {
      "!type": "fn(arg: +MIXED) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#method_log",
      "!doc": "Wrapper for <code>Y.log</code>."
     },
     "clearConsole": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#method_clearConsole",
      "!doc": "Clear the console of messages and flush the buffer of pending messages."
     },
     "reset": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#method_reset",
      "!doc": "Clears the console and resets internal timers."
     },
     "collapse": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#method_collapse",
      "!doc": "Collapses the body and footer."
     },
     "expand": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#method_expand",
      "!doc": "Expands the body and footer if collapsed."
     },
     "printBuffer": {
      "!type": "fn(limit: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#method_printBuffer",
      "!doc": "Outputs buffered messages to the console UI.  This is typically called\nfrom a scheduled interval until the buffer is empty (referred to as the\nprint loop).  The number of buffered messages output to the Console is\nlimited to the number provided as an argument.  If no limit is passed,\nall buffered messages are rendered."
     },
     "syncUI": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#method_syncUI",
      "!doc": "Sync the UI state to the current attribute state."
     },
     "scrollToLatest": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#method_scrollToLatest",
      "!doc": "Scrolls to the most recent entry"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_NAME",
     "!doc": "The identity of the widget."
    },
    "LOG_LEVEL_INFO": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_LOG_LEVEL_INFO",
     "!doc": "Static identifier for logLevel configuration setting to allow all\nincoming messages to generate Console entries."
    },
    "LOG_LEVEL_WARN": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_LOG_LEVEL_WARN",
     "!doc": "Static identifier for logLevel configuration setting to allow only\nincoming messages of logLevel &quot;warn&quot; or &quot;error&quot;\nto generate Console entries."
    },
    "LOG_LEVEL_ERROR": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_LOG_LEVEL_ERROR",
     "!doc": "Static identifier for logLevel configuration setting to allow only\nincoming messages of logLevel &quot;error&quot; to generate\nConsole entries."
    },
    "ENTRY_CLASSES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_ENTRY_CLASSES",
     "!doc": "Map (object) of classNames used to populate the placeholders in the\nConsole.ENTRY_TEMPLATE markup when rendering a new Console entry.\n\n<p>By default, the keys contained in the object are:</p>\n<ul>\n   <li>entry_class</li>\n   <li>entry_meta_class</li>\n   <li>entry_cat_class</li>\n   <li>entry_src_class</li>\n   <li>entry_time_class</li>\n   <li>entry_content_class</li>\n</ul>"
    },
    "CHROME_CLASSES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_CHROME_CLASSES",
     "!doc": "Map (object) of classNames used to populate the placeholders in the\nConsole.HEADER_TEMPLATE, Console.BODY_TEMPLATE, and\nConsole.FOOTER_TEMPLATE markup when rendering the Console UI.\n\n<p>By default, the keys contained in the object are:</p>\n<ul>\n  <li>console_hd_class</li>\n  <li>console_bd_class</li>\n  <li>console_ft_class</li>\n  <li>console_controls_class</li>\n  <li>console_checkbox_class</li>\n  <li>console_pause_class</li>\n  <li>console_pause_label_class</li>\n  <li>console_button_class</li>\n  <li>console_clear_class</li>\n  <li>console_collapse_class</li>\n  <li>console_title_class</li>\n</ul>"
    },
    "HEADER_TEMPLATE": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_HEADER_TEMPLATE",
     "!doc": "Markup template used to generate the DOM structure for the header\nsection of the Console when it is rendered.  The template includes\nthese {placeholder}s:\n\n<ul>\n  <li>console_button_class - contributed by Console.CHROME_CLASSES</li>\n  <li>console_collapse_class - contributed by Console.CHROME_CLASSES</li>\n  <li>console_hd_class - contributed by Console.CHROME_CLASSES</li>\n  <li>console_title_class - contributed by Console.CHROME_CLASSES</li>\n  <li>str_collapse - pulled from attribute strings.collapse</li>\n  <li>str_title - pulled from attribute strings.title</li>\n</ul>"
    },
    "BODY_TEMPLATE": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_BODY_TEMPLATE",
     "!doc": "Markup template used to generate the DOM structure for the Console body\n(where the messages are inserted) when it is rendered.  The template\nincludes only the {placeholder} &quot;console_bd_class&quot;, which is\nconstributed by Console.CHROME_CLASSES."
    },
    "FOOTER_TEMPLATE": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_FOOTER_TEMPLATE",
     "!doc": "Markup template used to generate the DOM structure for the footer\nsection of the Console when it is rendered.  The template includes\nmany of the {placeholder}s from Console.CHROME_CLASSES as well as:\n\n<ul>\n  <li>id_guid - generated unique id, relates the label and checkbox</li>\n  <li>str_pause - pulled from attribute strings.pause</li>\n  <li>str_clear - pulled from attribute strings.clear</li>\n</ul>"
    },
    "ENTRY_TEMPLATE": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_ENTRY_TEMPLATE",
     "!doc": "Default markup template used to create the DOM structure for Console\nentries. The markup contains {placeholder}s for content and classes\nthat are replaced via Y.Lang.sub.  The default template contains\nthe {placeholder}s identified in Console.ENTRY_CLASSES as well as the\nfollowing placeholders that will be populated by the log entry data:\n\n<ul>\n  <li>cat_class</li>\n  <li>src_class</li>\n  <li>totalTime</li>\n  <li>elapsedTime</li>\n  <li>localTime</li>\n  <li>sourceAndDetail</li>\n  <li>message</li>\n</ul>"
    },
    "ATTRS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Console.html#property_ATTRS",
     "!doc": "Static property used to define the default attribute configuration of\nthe Widget."
    }
   }
  },
  "cookie": {
   "Cookie": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html",
    "exists": {
     "!type": "fn(name: string) -> bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_exists",
     "!doc": "Determines if the cookie with the given name exists. This is useful for\nBoolean cookies (those that do not follow the name=value convention)."
    },
    "get": {
     "!type": "fn(name: string, options: fn()) -> ?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_get",
     "!doc": "Returns the cookie value for the given name."
    },
    "getSub": {
     "!type": "fn(name: string, subName: string, converter: fn(), options: +Object) -> ?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_getSub",
     "!doc": "Returns the value of a subcookie."
    },
    "getSubs": {
     "!type": "fn(name: string, options: +Object) -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_getSubs",
     "!doc": "Returns an object containing name-value pairs stored in the cookie with the given name."
    },
    "remove": {
     "!type": "fn(name: string, options: +Object) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_remove",
     "!doc": "Removes a cookie from the machine by setting its expiration date to\nsometime in the past."
    },
    "removeSub": {
     "!type": "fn(name: string, subName: string, options: +Object) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_removeSub",
     "!doc": "Removes a sub cookie with a given name."
    },
    "set": {
     "!type": "fn(name: string, value: ?, options: +Object) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_set",
     "!doc": "Sets a cookie with a given name and value."
    },
    "setSub": {
     "!type": "fn(name: string, subName: string, value: ?, options: +Object) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_setSub",
     "!doc": "Sets a sub cookie with a given name to a particular value."
    },
    "setSubs": {
     "!type": "fn(name: string, value: +Object, options: +Object) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Cookie.html#method_setSubs",
     "!doc": "Sets a cookie with a given name to contain a hash of name-value pairs."
    }
   }
  },
  "dataschema": {
   "!data": {
    "submodules": {
     "dataschema-array": {},
     "dataschema-base": {},
     "dataschema-json": {},
     "dataschema-text": {},
     "dataschema-xml": {}
    }
   },
   "DataSchema": {
    "Array": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.Array.html",
     "prototype": {
      "!proto": "dataschema.DataSchema.Base.prototype"
     },
     "apply": {
      "!type": "fn(schema?: +config.DataSchema.ArrayApplyConfig, data: [?]) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.Array.html#method_apply",
      "!doc": "Applies a schema to an array of data, returning a normalized object\nwith results in the `results` property. The `meta` property of the\nresponse object is present for consistency, but is assigned an empty\nobject.  If the input data is absent or not an array, an `error`\nproperty will be added.\n\nThe input array is expected to contain objects, arrays, or strings.\n\nIf _schema_ is not specified or _schema.resultFields_ is not an array,\n`response.results` will be assigned the input array unchanged.\n\nWhen a _schema_ is specified, the following will occur:\n\nIf the input array contains strings, they will be copied as-is into the\n`response.results` array.\n\nIf the input array contains arrays, `response.results` will contain an\narray of objects with key:value pairs assuming the fields in\n_schema.resultFields_ are ordered in accordance with the data array\nvalues.\n\nIf the input array contains objects, the identified\n_schema.resultFields_ will be used to extract a value from those\nobjects for the output result.\n\n_schema.resultFields_ field identifiers are objects with the following properties:\n\n  * `key`   : <strong>(required)</strong> The locator name (String)\n  * `parser`: A function or the name of a function on `Y.Parsers` used\n        to convert the input value into a normalized type.  Parser\n        functions are passed the value as input and are expected to\n        return a value.\n\nIf no value parsing is needed, you can use strings as identifiers\ninstead of objects (see example below).",
      "!data": {
       "submodule": "dataschema-array"
      }
     }
    },
    "Base": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.Base.html",
     "apply": {
      "!type": "fn(schema: +Object, data: +Object) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.Base.html#method_apply",
      "!doc": "Overridable method returns data as-is.",
      "!data": {
       "submodule": "dataschema-base"
      }
     },
     "prototype": {
      "parse": {
       "!type": "fn(value: +Object, field: +Object) -> +Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.Base.html#method_parse",
       "!doc": "Applies field parser, if defined",
       "!data": {
        "submodule": "dataschema-base"
       }
      }
     }
    },
    "JSON": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.JSON.html",
     "prototype": {
      "!proto": "dataschema.DataSchema.Base.prototype"
     },
     "getPath": {
      "!type": "fn(locator: string) -> [string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.JSON.html#method_getPath",
      "!doc": "Utility function converts JSON locator strings into walkable paths",
      "!data": {
       "submodule": "dataschema-json"
      }
     },
     "getLocationValue": {
      "!type": "fn(path: [string], data: string) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.JSON.html#method_getLocationValue",
      "!doc": "Utility function to walk a path and return the value located there.",
      "!data": {
       "submodule": "dataschema-json"
      }
     },
     "apply": {
      "!type": "fn(schema?: +config.DataSchema.JSONApplyConfig, data: +Object|[?]|string) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.JSON.html#method_apply",
      "!doc": "Applies a schema to an array of data located in a JSON structure, returning\na normalized object with results in the `results` property. Additional\ninformation can be parsed out of the JSON for inclusion in the `meta`\nproperty of the response object.  If an error is encountered during\nprocessing, an `error` property will be added.\n\nThe input _data_ is expected to be an object or array.  If it is a string,\nit will be passed through `Y.JSON.parse()`.\n\nIf _data_ contains an array of data records to normalize, specify the\n_schema.resultListLocator_ as a dot separated path string just as you would\nreference it in JavaScript.  So if your _data_ object has a record array at\n_data.response.results_, use _schema.resultListLocator_ =\n\"response.results\". Bracket notation can also be used for array indices or\nobject properties (e.g. \"response[results]\");  This is called a \"path\nlocator\"\n\nField data in the result list is extracted with field identifiers in\n_schema.resultFields_.  Field identifiers are objects with the following\nproperties:\n\n  * `key`   : <strong>(required)</strong> The path locator (String)\n  * `parser`: A function or the name of a function on `Y.Parsers` used\n        to convert the input value into a normalized type.  Parser\n        functions are passed the value as input and are expected to\n        return a value.\n\nIf no value parsing is needed, you can use path locators (strings)\ninstead of field identifiers (objects) -- see example below.\n\nIf no processing of the result list array is needed, _schema.resultFields_\ncan be omitted; the `response.results` will point directly to the array.\n\nIf the result list contains arrays, `response.results` will contain an\narray of objects with key:value pairs assuming the fields in\n_schema.resultFields_ are ordered in accordance with the data array\nvalues.\n\nIf the result list contains objects, the identified _schema.resultFields_\nwill be used to extract a value from those objects for the output result.\n\nTo extract additional information from the JSON, include an array of\npath locators in _schema.metaFields_.  The collected values will be\nstored in `response.meta`.",
      "!data": {
       "submodule": "dataschema-json"
      }
     }
    },
    "Text": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.Text.html",
     "prototype": {
      "!proto": "dataschema.DataSchema.Base.prototype"
     },
     "apply": {
      "!type": "fn(schema: +config.DataSchema.TextApplyConfig, data: string) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.Text.html#method_apply",
      "!doc": "Applies a schema to a string of delimited data, returning a normalized\nobject with results in the `results` property. The `meta` property of\nthe response object is present for consistency, but is assigned an\nempty object.  If the input data is absent or not a string, an `error`\nproperty will be added.\n\nUse _schema.resultDelimiter_ and _schema.fieldDelimiter_ to instruct\n`apply` how to split up the string into an array of data arrays for\nprocessing.\n\nUse _schema.resultFields_ to specify the keys in the generated result\nobjects in `response.results`. The key:value pairs will be assigned\nin the order of the _schema.resultFields_ array, assuming the values\nin the data records are defined in the same order.\n\n_schema.resultFields_ field identifiers are objects with the following\nproperties:\n\n  * `key`   : <strong>(required)</strong> The property name you want\n        the data value assigned to in the result object (String)\n  * `parser`: A function or the name of a function on `Y.Parsers` used\n        to convert the input value into a normalized type.  Parser\n        functions are passed the value as input and are expected to\n        return a value.\n\nIf no value parsing is needed, you can use just the desired property\nname string as the field identifier instead of an object (see example\nbelow).",
      "!data": {
       "submodule": "dataschema-text"
      }
     }
    },
    "XML": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.XML.html",
     "prototype": {
      "!proto": "dataschema.DataSchema.Base.prototype"
     },
     "apply": {
      "!type": "fn(schema: +config.DataSchema.XMLApplyConfig, data: +XMLDocument) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSchema.XML.html#method_apply",
      "!doc": "Applies a schema to an XML data tree, returning a normalized object with\nresults in the `results` property. Additional information can be parsed out\nof the XML for inclusion in the `meta` property of the response object.  If\nan error is encountered during processing, an `error` property will be\nadded.\n\nField data in the nodes captured by the XPath in _schema.resultListLocator_\nis extracted with the field identifiers described in _schema.resultFields_.\nField identifiers are objects with the following properties:\n\n  * `key`    : <strong>(required)</strong> The desired property name to use\n        store the retrieved value in the result object.  If `locator` is\n        not specified, `key` is also used as the XPath locator (String)\n  * `locator`: The XPath locator to the node or attribute within each\n        result node found by _schema.resultListLocator_ containing the\n        desired field data (String)\n  * `parser` : A function or the name of a function on `Y.Parsers` used\n        to convert the input value into a normalized type.  Parser\n        functions are passed the value as input and are expected to\n        return a value.\n  * `schema` : Used to retrieve nested field data into an array for\n        assignment as the result field value.  This object follows the same\n        conventions as _schema_.\n\nIf no value parsing or nested parsing is needed, you can use XPath locators\n(strings) instead of field identifiers (objects) -- see example below.\n\n`response.results` will contain an array of objects with key:value pairs.\nThe keys are the field identifier `key`s, and the values are the data\nvalues extracted from the nodes or attributes found by the field `locator`\n(or `key` fallback).\n\nTo extract additional information from the XML, include an array of\nXPath locators in _schema.metaFields_.  The collected values will be\nstored in `response.meta` with the XPath locator as keys.",
      "!data": {
       "submodule": "dataschema-xml"
      }
     }
    }
   }
  },
  "datasource": {
   "!data": {
    "submodules": {
     "datasource-arrayschema": {},
     "datasource-cache": {},
     "datasource-function": {},
     "datasource-get": {},
     "datasource-io": {},
     "datasource-jsonschema": {},
     "datasource-local": {},
     "datasource-polling": {},
     "datasource-textschema": {},
     "datasource-xmlschema": {}
    }
   },
   "DataSourceArraySchema": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceArraySchema.html",
    "prototype": {
     "!proto": "plugin.Plugin.Base.prototype"
    },
    "NS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceArraySchema.html#property_NS",
     "!doc": "The namespace for the plugin. This will be the property on the host which\nreferences the plugin instance.",
     "!data": {
      "submodule": "datasource-arrayschema"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceArraySchema.html#property_NAME",
     "!doc": "Class name.",
     "!data": {
      "submodule": "datasource-arrayschema"
     }
    }
   },
   "DataSourceCacheExtension": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceCacheExtension.html",
    "NS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceCacheExtension.html#property_NS",
     "!doc": "The namespace for the plugin. This will be the property on the host which\nreferences the plugin instance.",
     "!data": {
      "submodule": "datasource-cache"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceCacheExtension.html#property_NAME",
     "!doc": "Class name.",
     "!data": {
      "submodule": "datasource-cache"
     }
    }
   },
   "DataSourceCache": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceCache.html",
    "!data": {
     "augments": [
      "DataSourceCachePlugin"
     ],
     "extends": [
      "plugin.Plugin.Base"
     ]
    },
    "prototype": {
     "!proto": "cache.Cache.prototype"
    },
    "NS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceCache.html#property_NS",
     "!doc": "The namespace for the plugin. This will be the property on the host which\nreferences the plugin instance.",
     "!data": {
      "submodule": "datasource-cache"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceCache.html#property_NAME",
     "!doc": "Class name.",
     "!data": {
      "submodule": "datasource-cache"
     }
    }
   },
   "DataSource": {
    "Function": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Function.html",
     "prototype": {
      "!proto": "datasource.DataSource.Local.prototype"
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Function.html#property_NAME",
      "!doc": "Class name.",
      "!data": {
       "submodule": "datasource-function"
      }
     }
    },
    "Get": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Get.html",
     "prototype": {
      "!proto": "datasource.DataSource.Local.prototype"
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Get.html#property_NAME",
      "!doc": "Class name.",
      "!data": {
       "submodule": "datasource-get"
      }
     }
    },
    "IO": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.IO.html",
     "prototype": {
      "!proto": "datasource.DataSource.Local.prototype"
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.IO.html#property_NAME",
      "!doc": "Class name.",
      "!data": {
       "submodule": "datasource-io"
      }
     }
    },
    "Local": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Local.html",
     "prototype": {
      "!proto": "base.Base.prototype",
      "sendRequest": {
       "!type": "fn(request?: +Object) -> number",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Local.html#method_sendRequest",
       "!doc": "Generates a unique transaction ID and fires <code>request</code> event.\n<strong>Note</strong>: the property <code>callback</code> is a\ndeprecated alias for the <code>on</code> transaction configuration\nproperty described below.",
       "!data": {
        "submodule": "datasource-local"
       }
      }
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Local.html#property_NAME",
      "!doc": "Class name.",
      "!data": {
       "submodule": "datasource-local"
      }
     },
     "transactions": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Local.html#property_transactions",
      "!doc": "Global in-progress transaction objects.",
      "!data": {
       "submodule": "datasource-local"
      }
     },
     "issueCallback": {
      "!type": "fn(e: +event_custom.EventFacade, caller: +DataSource)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSource.Local.html#method_issueCallback",
      "!doc": "Returns data to callback.",
      "!data": {
       "submodule": "datasource-local"
      }
     }
    }
   },
   "DataSourceJSONSchema": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceJSONSchema.html",
    "prototype": {
     "!proto": "plugin.Plugin.Base.prototype"
    },
    "NS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceJSONSchema.html#property_NS",
     "!doc": "The namespace for the plugin. This will be the property on the host which\nreferences the plugin instance.",
     "!data": {
      "submodule": "datasource-jsonschema"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceJSONSchema.html#property_NAME",
     "!doc": "Class name.",
     "!data": {
      "submodule": "datasource-jsonschema"
     }
    }
   },
   "Pollable": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Pollable.html",
    "prototype": {
     "!proto": "datasource.DataSource.Local.prototype",
     "setInterval": {
      "!type": "fn(msec: number, request?: +Object) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Pollable.html#method_setInterval",
      "!doc": "Sets up a polling mechanism to send requests at set intervals and\nforward responses to given callback.",
      "!data": {
       "submodule": "datasource-polling"
      }
     },
     "clearInterval": {
      "!type": "fn(id: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Pollable.html#method_clearInterval",
      "!doc": "Disables polling mechanism associated with the given interval ID.",
      "!data": {
       "submodule": "datasource-polling"
      }
     },
     "clearAllIntervals": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Pollable.html#method_clearAllIntervals",
      "!doc": "Clears all intervals.",
      "!data": {
       "submodule": "datasource-polling"
      }
     }
    }
   },
   "DataSourceTextSchema": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceTextSchema.html",
    "prototype": {
     "!proto": "plugin.Plugin.Base.prototype"
    },
    "NS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceTextSchema.html#property_NS",
     "!doc": "The namespace for the plugin. This will be the property on the host which\nreferences the plugin instance.",
     "!data": {
      "submodule": "datasource-textschema"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceTextSchema.html#property_NAME",
     "!doc": "Class name.",
     "!data": {
      "submodule": "datasource-textschema"
     }
    }
   },
   "DataSourceXMLSchema": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceXMLSchema.html",
    "prototype": {
     "!proto": "plugin.Plugin.Base.prototype"
    },
    "NS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceXMLSchema.html#property_NS",
     "!doc": "The namespace for the plugin. This will be the property on the host which\nreferences the plugin instance.",
     "!data": {
      "submodule": "datasource-xmlschema"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataSourceXMLSchema.html#property_NAME",
     "!doc": "Class name.",
     "!data": {
      "submodule": "datasource-xmlschema"
     }
    }
   }
  },
  "datatable": {
   "!data": {
    "submodules": {
     "datatable-base": {},
     "datatable-body": {},
     "datatable-column-widths": {},
     "datatable-core": {},
     "datatable-datasource": {},
     "datatable-foot": {},
     "datatable-head": {},
     "datatable-highlight": {},
     "datatable-keynav": {},
     "datatable-message": {},
     "datatable-mutable": {},
     "datatable-paginator": {},
     "datatable-sort": {},
     "datatable-table": {}
    }
   },
   "DataTable": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html",
    "prototype": {
     "!proto": "datatable.DataTable.Base.prototype",
     "COL_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property_COL_TEMPLATE",
      "!doc": "The HTML template used to create the tables `<col>`s.",
      "!data": {
       "submodule": "datatable-column-widths"
      }
     },
     "COLGROUP_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property_COLGROUP_TEMPLATE",
      "!doc": "The HTML template used to create the tables `<colgroup>`.",
      "!data": {
       "submodule": "datatable-column-widths"
      }
     },
     "setColumnWidth": {
      "!type": "fn(id: number|string|+Object, width: number|string) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_setColumnWidth",
      "!doc": "Assigns the style width of the `<col>` representing the column identifed by\n`id` and updates the column configuration.\n\nPass the empty string for `width` to return a column to auto sizing.\n\nThis does not trigger a `columnsChange` event today, but I can be convinced\nthat it should.",
      "!data": {
       "submodule": "datatable-column-widths"
      }
     },
     "data": {
      "!type": "+app.ModelList",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property_data",
      "!doc": "The ModelList that manages the tables data.",
      "!data": {
       "submodule": "datatable-core"
      }
     },
     "getColumn": {
      "!type": "fn(name: string|number|[number]) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_getColumn",
      "!doc": "Gets the column configuration object for the given key, name, or index.  For\nnested columns, `name` can be an array of indexes, each identifying the index\nof that column in the respective parents \"children\" array.\n\nIf you pass a column object, it will be returned.\n\nFor columns with keys, you can also fetch the column with\n`instance.get(columns.foo)`.",
      "!data": {
       "submodule": "datatable-core"
      }
     },
     "getRecord": {
      "!type": "fn(seed: number|string|+node.Node) -> +app.Model",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_getRecord",
      "!doc": "Returns the Model associated to the record `id`, `clientId`, or index (not\nrow index).  If none of those yield a Model from the `data` ModelList, the\narguments will be passed to the `view` instances `getRecord` method\nif it has one.\n\nIf no Model can be found, `null` is returned.",
      "!data": {
       "submodule": "datatable-core"
      }
     },
     "_getRecordType": {
      "!type": "fn(val: +app.Model) -> +app.Model",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method__getRecordType",
      "!doc": "Returns the Model class of the instances `data` attribute ModelList.  If\nnot set, returns the explicitly configured value.",
      "!data": {
       "submodule": "datatable-core"
      }
     },
     "keyActions": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property_keyActions",
      "!doc": "Table of actions to be performed for each key.  It is loaded with a clone\nof [ARIA_ACTIONS](#property_ARIA_ACTIONS) by default.\n\nThe key for each entry is either a key-code or an alias from the\n[KEY_NAMES](#property_KEY_NAMES) table. They can be prefixed with any combination\nof the modifier keys `alt`, `ctrl`, `meta` or `shift` each followed by a hyphen,\nsuch as `\"ctrl-shift-up\"` (modifiers, if more than one, should appear in alphabetical order).\n\nThe value for each entry should be a function or the name of a method in\nthe DataTable instance.  The method will receive the original keyboard\nEventFacade as its only argument.\n\nIf the value is a string and it cannot be resolved into a method,\nit will be assumed to be the name of an event to fire. The listener for that\nevent will receive an EventFacade containing references to the cell that has the focus,\nthe row, column and, unless it is a header row, the record it corresponds to.\nThe second argument will be the original EventFacade for the keyboard event.",
      "!data": {
       "submodule": "datatable-keynav"
      }
     },
     "MESSAGE_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property_MESSAGE_TEMPLATE",
      "!doc": "Template used to generate the node that will be used to report messages.",
      "!data": {
       "submodule": "datatable-message"
      }
     },
     "hideMessage": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_hideMessage",
      "!doc": "Hides the message node.",
      "!data": {
       "submodule": "datatable-message"
      }
     },
     "showMessage": {
      "!type": "fn(message: string) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_showMessage",
      "!doc": "Display the message node and set its content to `message`.  If there is a\nlocalized `strings` entry for the value of `message`, that string will be\nused.",
      "!data": {
       "submodule": "datatable-message"
      }
     },
     "_messageNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property__messageNode",
      "!doc": "Node used to display messages from `showMessage`.",
      "!data": {
       "submodule": "datatable-message"
      }
     },
     "addColumn": {
      "!type": "fn(config: +Object|string, index?: number|[number]) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_addColumn",
      "!doc": "Adds the column configuration to the DataTables `columns` configuration.\nIf the `index` parameter is supplied, it is injected at that index.  If the\ntable has nested headers, inject a subcolumn by passing an array of indexes\nto identify the new columns final location.\n\nThe `index` parameter is required if adding a nested column.\n\nThis method is a convienience method for fetching the DataTables `columns`\nattribute, updating it, and calling\n`table.set(columns, _updatedColumnsDefs_)`\n\nFor example:\n\n<pre><code>// Becomes last column\ntable.addColumn(name);\n\n// Inserted after the current second column, moving the current third column\n// to index 4\ntable.addColumn({ key: price, formatter: currencyFormatter }, 2 );\n\n// Insert a new column in a set of headers three rows deep.  The index array\n// translates to\n// [ 2, --  in the third columns children\n//   1, --  in the second childs children\n//   3 ] -- as the fourth child column\ntable.addColumn({ key: age, sortable: true }, [ 2, 1, 3 ]);\n</code></pre>",
      "!data": {
       "submodule": "datatable-mutable"
      }
     },
     "modifyColumn": {
      "!type": "fn(name: string|number|[number]|+Object, config: +config.DataTableModifyColumnConfig) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_modifyColumn",
      "!doc": "Updates an existing column definition. Fires the `modifyColumn` event.\n\nFor example:\n\n<pre><code>// Add a formatter to the existing price column definition\ntable.modifyColumn(price, { formatter: currencyFormatter });\n\n// Change the label on a header cell in a set of nested headers three rows\n// deep.  The index array translates to\n// [ 2,  -- in the third columns children\n//   1,  -- the second child\n//   3 ] -- the fourth child column\ntable.modifyColumn([2, 1, 3], { label: Experience });\n</code></pre>",
      "!data": {
       "submodule": "datatable-mutable"
      }
     },
     "moveColumn": {
      "!type": "fn(name: string|number|[number]|+Object, index: number|[number]) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_moveColumn",
      "!doc": "Moves an existing column to a new location. Fires the `moveColumn` event.\n\nThe destination index can be a number or array of numbers to place a column\nheader in a nested header row.",
      "!data": {
       "submodule": "datatable-mutable"
      }
     },
     "removeColumn": {
      "!type": "fn(name: string|number|[number]|+Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_removeColumn",
      "!doc": "Removes an existing column. Fires the `removeColumn` event.",
      "!data": {
       "submodule": "datatable-mutable"
      }
     },
     "addRow": {
      "!type": "fn(data: +Object, config?: +config.DataTableAddRowConfig, callback?: fn(err: +Error, response: ?)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_addRow",
      "!doc": "Adds a new record to the DataTables `data` ModelList.  Record data can be\nan object of field values or an instance of the DataTables configured\n`recordType` class.\n\nThis relays all parameters to the `data` ModelLists `add` method.\n\nIf a configuration object is passed as a second argument, and that object\nhas `sync: true` set, the underlying Model will be `save()`d.\n\nIf the DataTables `autoSync` attribute is set to `true`, the additional\nargument is not needed.\n\nIf syncing and the last argument is a function, that function will be used\nas a callback to the Models `save()` method.",
      "!data": {
       "submodule": "datatable-mutable"
      }
     },
     "removeRow": {
      "!type": "fn(id: +Object|string|number, config?: +config.DataTableRemoveRowConfig, callback?: fn(err: +Error, response: ?)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_removeRow",
      "!doc": "Removes a record from the DataTables `data` ModelList.  The record can be\nprovided explicitly or targeted by its `id` (see ModelLists `getById`\nmethod), `clientId`, or index in the ModelList.\n\nAfter locating the target Model, this relays the Model and all other passed\narguments to the `data` ModelLists `remove` method.\n\nIf a configuration object is passed as a second argument, and that object\nhas `sync: true` set, the underlying Model will be destroyed, passing\n`{ delete: true }` to trigger calling the Models sync layer.\n\nIf the DataTables `autoSync` attribute is set to `true`, the additional\nargument is not needed.\n\nIf syncing and the last argument is a function, that function will be used\nas a callback to the Models `destroy()` method.",
      "!data": {
       "submodule": "datatable-mutable"
      }
     },
     "modifyRow": {
      "!type": "fn(id: +Object|string|number, data: +Object, config?: +config.DataTableModifyRowConfig, callback?: fn(err: +Error, response: ?)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_modifyRow",
      "!doc": "Updates an existing record in the DataTables `data` ModelList.  The record\ncan be provided explicitly or targeted by its `id` (see ModelLists\n`getById` method), `clientId`, or index in the ModelList.\n\nAfter locating the target Model, this relays the all other passed\narguments to the Models `setAttrs` method.\n\nIf a configuration object is passed as a second argument, and that object\nhas `sync: true` set, the underlying Model will be `save()`d.\n\nIf the DataTables `autoSync` attribute is set to `true`, the additional\nargument is not needed.\n\nIf syncing and the last argument is a function, that function will be used\nas a callback to the Models `save()` method.",
      "!data": {
       "submodule": "datatable-mutable"
      }
     },
     "addRows": {
      "!type": "fn(data: [+Object], config?: +config.DataTableAddRowsConfig, callback?: fn(err: +Error, response: ?)) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_addRows",
      "!doc": "Adds an array of new records to the DataTables `data` ModelList.  Record data\ncan be an array of objects containing field values or an array of instance of\nthe DataTables configured `recordType` class.\n\nThis relays all parameters to the `data` ModelLists `add` method.\n\nTechnically, this is an alias to `addRow`, but please use the appropriately\nnamed method for readability.\n\nIf a configuration object is passed as a second argument, and that object\nhas `sync: true` set, the underlying Models will be `save()`d.\n\nIf the DataTables `autoSync` attribute is set to `true`, the additional\nargument is not needed.\n\nIf syncing and the last argument is a function, that function will be used\nas a callback to each Models `save()` method.",
      "!data": {
       "submodule": "datatable-mutable"
      }
     },
     "sort": {
      "!type": "fn(fields: string|[string]|+Object|[+Object], payload?: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_sort",
      "!doc": "Sort the data in the `data` ModelList and refresh the table with the new\norder.\n\nAcceptable values for `fields` are `key` strings or objects with a single\nproperty, the column `key`, with a value of 1, -1, \"asc\", or \"desc\".  E.g.\n`{ username: asc }`.  String values are assumed to be ascending.\n\nExample values would be:\n\n * `\"username\"` - sort by the datas `username` field or the `key`\n   associated to a column with that `name`.\n * `{ username: \"desc\" }` - sort by `username` in descending order.\n   Alternately, use values \"asc\", 1 (same as \"asc\"), or -1 (same as \"desc\").\n * `[\"lastName\", \"firstName\"]` - ascending sort by `lastName`, but for\n   records with the same `lastName`, ascending subsort by `firstName`.\n   Array can have as many items as you want.\n * `[{ lastName: -1 }, \"firstName\"]` - descending sort by `lastName`,\n   ascending subsort by `firstName`. Mixed types are ok.",
      "!data": {
       "submodule": "datatable-sort"
      }
     },
     "SORTABLE_HEADER_TEMPLATE": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property_SORTABLE_HEADER_TEMPLATE",
      "!doc": "Template for the node that will wrap the header content for sortable\ncolumns.",
      "!data": {
       "submodule": "datatable-sort"
      }
     },
     "toggleSort": {
      "!type": "fn(fields: string|[string], payload?: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_toggleSort",
      "!doc": "Reverse the current sort direction of one or more fields currently being\nsorted by.\n\nPass the `key` of the column or columns you want the sort order reversed\nfor.",
      "!data": {
       "submodule": "datatable-sort"
      }
     }
    },
    "Base": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html",
     "!data": {
      "extends": [
       "datatable.DataTable.Core"
      ]
     },
     "prototype": {
      "!proto": "widget.Widget.prototype",
      "delegate": {
       "!type": "fn(type: string, fn: fn(), spec: string, context: +Object, args: ?) -> +event_custom.EventHandle",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#method_delegate",
       "!doc": "Pass through to `delegate()` called from the `contentBox`.",
       "!data": {
        "submodule": "datatable-base"
       }
      },
      "getCell": {
       "!type": "fn(seed: [number]|+node.Node, shift?: [number]|string) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#method_getCell",
       "!doc": "Returns the `<td>` Node from the given row and column index.  Alternately,\nthe `seed` can be a Node.  If so, the nearest ancestor cell is returned.\nIf the `seed` is a cell, it is returned.  If there is no cell at the given\ncoordinates, `null` is returned.\n\nOptionally, include an offset array or string to return a cell near the\ncell identified by the `seed`.  The offset can be an array containing the\nnumber of rows to shift followed by the number of columns to shift, or one\nof \"above\", \"below\", \"next\", or \"previous\".\n\n<pre><code>// Previous cell in the previous row\nvar cell = table.getCell(e.target, [-1, -1]);\n\n// Next cell\nvar cell = table.getCell(e.target, next);\nvar cell = table.getCell(e.taregt, [0, 1];</pre></code>\n\nThis is actually just a pass through to the `view` instances method\nby the same name.",
       "!data": {
        "submodule": "datatable-base"
       }
      },
      "getRow": {
       "!type": "fn(id: number|string|+app.Model) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#method_getRow",
       "!doc": "Returns the `<tr>` Node from the given row index, Model, or Models\n`clientId`.  If the rows havent been rendered yet, or if the row cant be\nfound by the input, `null` is returned.\n\nThis is actually just a pass through to the `view` instances method\nby the same name.",
       "!data": {
        "submodule": "datatable-base"
       }
      },
      "_displayColumns": {
       "!type": "[+Object]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#property__displayColumns",
       "!doc": "Array of the columns that correspond to those with value cells in the\ndata rows. Excludes colspan header columns (configured with `children`).",
       "!data": {
        "submodule": "datatable-base"
       }
      },
      "syncUI": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Base.html#method_syncUI",
       "!doc": "Fires the `renderView` event, delegating UI updates to the configured View.",
       "!data": {
        "submodule": "datatable-base"
       }
      }
     }
    },
    "BodyView": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html",
     "prototype": {
      "!proto": "app.View.prototype",
      "CELL_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#property_CELL_TEMPLATE",
       "!doc": "HTML template used to create table cells.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "CLASS_EVEN": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#property_CLASS_EVEN",
       "!doc": "CSS class applied to even rows.  This is assigned at instantiation.\n\nFor DataTable, this will be `yui3-datatable-even`.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "CLASS_ODD": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#property_CLASS_ODD",
       "!doc": "CSS class applied to odd rows.  This is assigned at instantiation.\n\nWhen used by DataTable instances, this will be `yui3-datatable-odd`.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "ROW_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#property_ROW_TEMPLATE",
       "!doc": "HTML template used to create table rows.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "host": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#property_host",
       "!doc": "The object that serves as the source of truth for column and row data.\nThis property is assigned at instantiation from the `host` property of\nthe configuration object passed to the constructor.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "TBODY_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#property_TBODY_TEMPLATE",
       "!doc": "HTML templates used to create the `<tbody>` containing the table rows.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "getCell": {
       "!type": "fn(seed: [number]|+node.Node, shift?: [number]|string) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#method_getCell",
       "!doc": "Returns the `<td>` Node from the given row and column index.  Alternately,\nthe `seed` can be a Node.  If so, the nearest ancestor cell is returned.\nIf the `seed` is a cell, it is returned.  If there is no cell at the given\ncoordinates, `null` is returned.\n\nOptionally, include an offset array or string to return a cell near the\ncell identified by the `seed`.  The offset can be an array containing the\nnumber of rows to shift followed by the number of columns to shift, or one\nof \"above\", \"below\", \"next\", or \"previous\".\n\n<pre><code>// Previous cell in the previous row\nvar cell = table.getCell(e.target, [-1, -1]);\n\n// Next cell\nvar cell = table.getCell(e.target, next);\nvar cell = table.getCell(e.target, [0, 1];</pre></code>",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "getRecord": {
       "!type": "fn(seed: string|+node.Node) -> +app.Model",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#method_getRecord",
       "!doc": "Returns the Model associated to the row Node or id provided. Passing the\nNode or id for a descendant of the row also works.\n\nIf no Model can be found, `null` is returned.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "getRow": {
       "!type": "fn(id: number|string|+app.Model) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#method_getRow",
       "!doc": "Returns the `<tr>` Node from the given row index, Model, or Models\n`clientId`.  If the rows havent been rendered yet, or if the row cant be\nfound by the input, `null` is returned.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "render": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#method_render",
       "!doc": "Creates the tables `<tbody>` content by assembling markup generated by\npopulating the `ROW\\_TEMPLATE`, and `CELL\\_TEMPLATE` templates with content\nfrom the `columns` and `modelList` attributes.\n\nThe rendering process happens in three stages:\n\n1. A row template is assembled from the `columns` attribute (see\n   `_createRowTemplate`)\n\n2. An HTML string is built up by concatenating the application of the data in\n   each Model in the `modelList` to the row template. For cells with\n   `formatter`s, the function is called to generate cell content. Cells\n   with `nodeFormatter`s are ignored. For all other cells, the data value\n   from the Model attribute for the given column key is used.  The\n   accumulated row markup is then inserted into the container.\n\n3. If any column is configured with a `nodeFormatter`, the `modelList` is\n   iterated again to apply the `nodeFormatter`s.\n\nSupported properties of the column objects include:\n\n  * `key` - Used to link a column to an attribute in a Model.\n  * `name` - Used for columns that dont relate to an attribute in the Model\n    (`formatter` or `nodeFormatter` only) if the implementer wants a\n    predictable name to refer to in their CSS.\n  * `cellTemplate` - Overrides the instances `CELL_TEMPLATE` for cells in\n    this column only.\n  * `formatter` - Used to customize or override the content value from the\n    Model.  These do not have access to the cell or row Nodes and should\n    return string (HTML) content.\n  * `nodeFormatter` - Used to provide content for a cell as well as perform\n    any custom modifications on the cell or row Node that could not be\n    performed by `formatter`s.  Should be used sparingly for better\n    performance.\n  * `emptyCellValue` - String (HTML) value to use if the Model data for a\n    column, or the content generated by a `formatter`, is the empty string,\n    `null`, or `undefined`.\n  * `allowHTML` - Set to `true` if a column value, `formatter`, or\n    `emptyCellValue` can contain HTML.  This defaults to `false` to protect\n    against XSS.\n  * `className` - Space delimited CSS classes to add to all `<td>`s in a\n    column.\n\nColumn `formatter`s are passed an object (`o`) with the following\nproperties:\n\n  * `value` - The current value of the columns associated attribute, if\n    any.\n  * `data` - An object map of Model keys to their current values.\n  * `record` - The Model instance.\n  * `column` - The column configuration object for the current column.\n  * `className` - Initially empty string to allow `formatter`s to add CSS\n    classes to the cells `<td>`.\n  * `rowIndex` - The zero-based row number.\n  * `rowClass` - Initially empty string to allow `formatter`s to add CSS\n    classes to the cells containing row `<tr>`.\n\nThey may return a value or update `o.value` to assign specific HTML\ncontent.  A returned value has higher precedence.\n\nColumn `nodeFormatter`s are passed an object (`o`) with the following\nproperties:\n\n  * `value` - The current value of the columns associated attribute, if\n    any.\n  * `td` - The `<td>` Node instance.\n  * `cell` - The `<div>` liner Node instance if present, otherwise, the\n    `<td>`.  When adding content to the cell, prefer appending into this\n    property.\n  * `data` - An object map of Model keys to their current values.\n  * `record` - The Model instance.\n  * `column` - The column configuration object for the current column.\n  * `rowIndex` - The zero-based row number.\n\nThey are expected to inject content into the cells Node directly, including\nany \"empty\" cell content.  Each `nodeFormatter` will have access through the\nNode API to all cells and rows in the `<tbody>`, but not to the `<table>`,\nas it will not be attached yet.\n\nIf a `nodeFormatter` returns `false`, the `o.td` and `o.cell` Nodes will be\n`destroy()`ed to remove them from the Node cache and free up memory.  The\nDOM elements will remain as will any content added to them.  _It is highly\nadvisable to always return `false` from your `nodeFormatter`s_.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "refreshRow": {
       "!type": "fn(row: +node.Node, model: +app.Model, colKeys: [string]) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#method_refreshRow",
       "!doc": "Refreshes the provided row against the provided model and the Array of\ncolumns to be updated.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "refreshCell": {
       "!type": "fn(cell: +node.Node, model?: +app.Model, col?: +Object) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#method_refreshCell",
       "!doc": "Refreshes the given cell with the provided model data and the provided\ncolumn configuration.\n\nUses the provided column formatter if aviable.",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "getColumn": {
       "!type": "fn(name: string|+node.Node) -> +Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html#method_getColumn",
       "!doc": "Returns column data from this.get(columns). If a Y.Node is provided as\nthe key, will try to determine the key from the classname",
       "!data": {
        "submodule": "datatable-body"
       }
      },
      "undefined": {
       "!type": "fn(row: +node.Node) -> number",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.html",
       "!doc": "Locates the row within the tbodyNode and returns the found index, or Null\nif it is not found in the tbodyNode",
       "!data": {
        "submodule": "datatable-body"
       }
      }
     },
     "Formatters": {
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html",
      "!data": {
       "for": "datatable_formatters.DataTable.BodyView.Formatters"
      },
      "prototype": {
       "TFOOT_TEMPLATE": {
        "!type": "string",
        "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#property_TFOOT_TEMPLATE",
        "!doc": "HTML templates used to create the `<tfoot>` containing the table footers.",
        "!data": {
         "submodule": "datatable-foot"
        }
       },
       "render": {
        "!type": "fn() -> !this",
        "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_render",
        "!doc": "Creates the `<tfoot>` Node and inserts it after the `<thead>` Node.",
        "!data": {
         "submodule": "datatable-foot"
        }
       }
      }
     }
    },
    "Column": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html",
     "prototype": {
      "width": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_width",
       "!doc": "Adds a style `width` setting to an associated `<col>`\nelement for the column.\n\nNote, the assigned width will not truncate cell content, and\nit will not preserve the configured width if doing so would\ncompromise either the instances `width` configuration or\nthe natural width of the tables containing DOM elements.\n\nIf absolute widths are required, it can be accomplished with\nsome custom CSS and the use of a `cellTemplate`, or\n`formatter`.\n\nSee the description of\n[datatable-column-widths](DataTable.ColumnWidths.html)\nfor an example of how to do this.\n\n    { key: a, width: 400px },\n    { key: b, width: 10em }",
       "!data": {
        "submodule": "datatable-column-widths"
       }
      },
      "key": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_key",
       "!doc": "Binds the column values to the named property in the [data](DataTable.html#attr_data).\n\nOptional if [formatter](#property_formatter), [nodeFormatter](#property_nodeFormatter),\nor [cellTemplate](#property_cellTemplate) is used to populate the content.\n\nIt should not be set if [children](#property_children) is set.\n\nThe value is used for the [\\_id](#property__id) property unless the [name](#property_name)\nproperty is also set.\n\n    { key: username }\n\nThe above column definition can be reduced to this:\n\n    username",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "name": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_name",
       "!doc": "An identifier that can be used to locate a column via\n[getColumn](DataTable.html#method_getColumn)\nor style columns with class `yui3-datatable-col-NAME` after dropping characters\nthat are not valid for CSS class names.\n\nIt defaults to the [key](#property_key).\n\nThe value is used for the [\\_id](#property__id) property.\n\n    { name: fullname, formatter: ... }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "field": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_field",
       "!doc": "An alias for [name](#property_name) for backward compatibility.\n\n    { field: fullname, formatter: ... }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "id": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_id",
       "!doc": "Overrides the default unique id assigned `<th id=\"HERE\">`.\n\n__Use this with caution__, since it can result in\nduplicate ids in the DOM.\n\n    {\n        name: checkAll,\n        id: check-all,\n        label: ...\n        formatter: ...\n    }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "label": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_label",
       "!doc": "HTML to populate the header `<th>` for the column.\nIt defaults to the value of the [key](#property_key) property or the text\n`Column n` where _n_ is an ordinal number.\n\n    { key: MfgvaPrtNum, label: Part Number }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "children": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_children",
       "!doc": "Used to create stacked headers.\n\nChild columns may also contain `children`. There is no limit\nto the depth of nesting.\n\nColumns configured with `children` are for display only and\n<strong>should not</strong> be configured with a [key](#property_key).\nConfigurations relating to the display of data, such as\n[formatter](#property_formatter), [nodeFormatter](#property_nodeFormatter),\n[emptyCellValue](#property_emptyCellValue), etc. are ignored.\n\n    { label: Name, children: [\n        { key: firstName, label: First`},\n        { key: lastName, label: Last`}\n    ]}",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "abbr": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_abbr",
       "!doc": "Assigns the value `<th abbr=\"HERE\">`.\n\n    {\n      key  : forecast,\n      label: 1yr Target Forecast,\n      abbr : Forecast\n    }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "title": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_title",
       "!doc": "Assigns the value `<th title=\"HERE\">`.\n\n    {\n      key  : forecast,\n      label: 1yr Target Forecast,\n      title: Target Forecast for the Next 12 Months\n    }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "headerTemplate": {
       "!type": "+HTML",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_headerTemplate",
       "!doc": "Overrides the default [CELL_TEMPLATE](DataTable.HeaderView.html#property_CELL_TEMPLATE)\nused by `Y.DataTable.HeaderView` to render the header cell\nfor this column.  This is necessary when more control is\nneeded over the markup for the header itself, rather than\nits content.\n\nUse the [label](#property_label) configuration if you dont need to\ncustomize the `<th>` iteself.\n\nImplementers are strongly encouraged to preserve at least\nthe `{id}` and `{_id}` placeholders in the custom value.\n\n    {\n        headerTemplate:\n            <th id=\"{id}\"  +\n                title=\"Unread\"  +\n                class=\"{className}\"  +\n                {_id}>&#9679;</th>\n    }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "cellTemplate": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_cellTemplate",
       "!doc": "Overrides the default [CELL_TEMPLATE](DataTable.BodyView.html#property_CELL_TEMPLATE)\nused by `Y.DataTable.BodyView` to render the data cells\nfor this column.  This is necessary when more control is\nneeded over the markup for the `<td>` itself, rather than\nits content.\n\n    {\n        key: id,\n        cellTemplate:\n            <td class=\"{className}\"> +\n              <input type=\"checkbox\"  +\n                     id=\"{content}\"> +\n            </td>\n    }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "formatter": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_formatter",
       "!doc": "String or function used to translate the raw record data for each cell in a\ngiven column into a format better suited to display.\n\nIf it is a string, it will initially be assumed to be the name of one of the\nformatting functions in\n[Y.DataTable.BodyView.Formatters](DataTable.BodyView.Formatters.html).\nIf one such formatting function exists, it will be used.\n\nIf no such named formatter is found, it will be assumed to be a template\nstring and will be expanded.  The placeholders can contain the key to any\nfield in the record or the placeholder `{value}` which represents the value\nof the current field.\n\nIf the value is a function, it will be assumed to be a formatting function.\nA formatting function receives a single argument, an object with the following properties:\n\n* __value__ The raw value from the record Model to populate this cell.\n  Equivalent to `o.record.get(o.column.key)` or `o.data[o.column.key]`.\n* __data__ The Model data for this row in simple object format.\n* __record__ The Model for this row.\n* __column__ The column configuration object.\n* __className__ A string of class names to add `<td class=\"HERE\">` in addition to\n  the column class and any classes in the columns className configuration.\n* __rowIndex__ The index of the current Model in the ModelList.\n  Typically correlates to the row index as well.\n* __rowClass__ A string of css classes to add `<tr class=\"HERE\"><td....`\n  This is useful to avoid the need for nodeFormatters to add classes to the containing row.\n\nThe formatter function may return a string value that will be used for the cell\ncontents or it may change the value of the `value`, `className` or `rowClass`\nproperties which well then be used to format the cell.  If the value for the cell\nis returned in the `value` property of the input argument, no value should be returned.\n\n    {\n        key: name,\n        formatter: link,  // named formatter\n        linkFrom: website // extra column property for link formatter\n    },\n    {\n        key: cost,\n        formatter: {value} // formatter template string\n      //formatter: {cost}  // same result but less portable\n    },\n    {\n        name: Name,          // column does not have associated field value\n                               // thus, it uses name instead of key\n        formatter: {firstName} {lastName} // template references other fields\n    },\n    {\n        key: price,\n        formatter: function (o) { // function both returns a string to show\n            if (o.value > 3) {    // and a className to apply to the cell\n                o.className += expensive;\n            }\n\n            return  + o.value.toFixed(2);\n        }\n    },",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "nodeFormatter": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_nodeFormatter",
       "!doc": "Used to customize the content of the data cells for this column.\n\n`nodeFormatter` is significantly slower than [formatter](#property_formatter)\nand should be avoided if possible. Unlike [formatter](#property_formatter),\n`nodeFormatter` has access to the `<td>` element and its ancestors.\n\nThe function provided is expected to fill in the `<td>` element itself.\n__Node formatters should return `false`__ except in certain conditions as described\nin the users guide.\n\nThe function receives a single object\nargument with the following properties:\n\n* __td__\tThe `<td>` Node for this cell.\n* __cell__\tIf the cell `<td> contains an element with class `yui3-datatable-liner,\n  this will refer to that Node. Otherwise, it is equivalent to `td` (default behavior).\n* __value__\tThe raw value from the record Model to populate this cell.\n  Equivalent to `o.record.get(o.column.key)` or `o.data[o.column.key]`.\n* __data__\tThe Model data for this row in simple object format.\n* __record__\tThe Model for this row.\n* __column__\tThe column configuration object.\n* __rowIndex__\tThe index of the current Model in the ModelList.\n _Typically_ correlates to the row index as well.",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "emptyCellValue": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_emptyCellValue",
       "!doc": "Provides the default value to populate the cell if the data\nfor that cell is `undefined`, `null`, or an empty string.\n\n    {\n        key: price,\n        emptyCellValue: ???\n    }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "allowHTML": {
       "!type": "bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_allowHTML",
       "!doc": "Skips the security step of HTML escaping the value for cells\nin this column.\n\nThis is also necessary if [emptyCellValue](#property_emptyCellValue)\nis set with an HTML string.\n`nodeFormatter`s ignore this configuration.  If using a\n`nodeFormatter`, it is recommended to use\n[Y.Escape.html()](Escape.html#method_html)\non any user supplied content that is to be displayed.\n\n    {\n        key: preview,\n        allowHTML: true\n    }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "className": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_className",
       "!doc": "A string of CSS classes that will be added to the `<td>`s\n`class` attribute.\n\nNote, all cells will automatically have a class in the\nform of \"yui3-datatable-col-XXX\" added to the `<td>`, where\nXXX is the columns configured `name`, `key`, or `id` (in\nthat order of preference) sanitized from invalid characters.\n\n    {\n        key: symbol,\n        className: no-hide\n    }",
       "!data": {
        "submodule": "datatable-core"
       }
      },
      "sortable": {
       "!type": "bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_sortable",
       "!doc": "Used when the instances `sortable` attribute is set to\n\"auto\" (the default) to determine which columns will support\nuser sorting by clicking on the header.\n\nIf the instances `key` attribute is not set, this\nconfiguration is ignored.\n\n    { key: lastLogin, sortable: true }",
       "!data": {
        "submodule": "datatable-sort"
       }
      },
      "caseSensitive": {
       "!type": "bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_caseSensitive",
       "!doc": "When the instances `caseSensitive` attribute is set to\n`true` the sort order is case sensitive (relevant to string columns only).\n\nCase sensitive sort is marginally more efficient and should be considered\nfor large data sets when case insensitive sort is not required.\n\n    { key: lastLogin, sortable: true, caseSensitive: true }",
       "!data": {
        "submodule": "datatable-sort"
       }
      },
      "sortFn": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_sortFn",
       "!doc": "Allows a column to be sorted using a custom algorithm.  The\nfunction receives three parameters, the first two being the\ntwo record Models to compare, and the third being a boolean\n`true` if the sort order should be descending.\n\nThe function should return `1` to sort `a` above `b`, `-1`\nto sort `a` below `b`, and `0` if they are equal.  Keep in\nmind that the order should be reversed when `desc` is\n`true`.\n\nThe `desc` parameter is provided to allow `sortFn`s to\nalways sort certain values above or below others, such as\nalways sorting `null`s on top.\n\n    {\n      label: Name,\n      sortFn: function (a, b, desc) {\n        var an = a.get(lname) + b.get(fname),\n            bn = a.get(lname) + b.get(fname),\n            order = (an > bn) ? 1 : -(an < bn);\n\n        return desc ? -order : order;\n      },\n      formatter: function (o) {\n        return o.data.lname + ,  + o.data.fname;\n      }\n    }",
       "!data": {
        "submodule": "datatable-sort"
       }
      },
      "sortDir": {
       "!type": "number",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_sortDir",
       "!doc": "(__read-only__) If a column is sorted, this\nwill be set to `1` for ascending order or `-1` for\ndescending.  This configuration is public for inspection,\nbut cant be used during DataTable instantiation to set the\nsort direction of the column.  Use the tables\n[sortBy](DataTable.html#attr_sortBy)\nattribute for that.",
       "!data": {
        "submodule": "datatable-sort"
       }
      }
     }
    },
    "HeaderView": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.HeaderView.html",
     "prototype": {
      "!proto": "app.View.prototype",
      "CELL_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.HeaderView.html#property_CELL_TEMPLATE",
       "!doc": "Template used to create the tables header cell markup.  Override this to\ncustomize how header cell markup is created.",
       "!data": {
        "submodule": "datatable-head"
       }
      },
      "columns": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.HeaderView.html#property_columns",
       "!doc": "The data representation of the header rows to render.  This is assigned by\nparsing the `columns` configuration array, and is used by the render()\nmethod.",
       "!data": {
        "submodule": "datatable-head"
       }
      },
      "ROW_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.HeaderView.html#property_ROW_TEMPLATE",
       "!doc": "Template used to create the tables header row markup.  Override this to\ncustomize the row markup.",
       "!data": {
        "submodule": "datatable-head"
       }
      },
      "source": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.HeaderView.html#property_source",
       "!doc": "The object that serves as the source of truth for column and row data.\nThis property is assigned at instantiation from the `source` property of\nthe configuration object passed to the constructor.",
       "!data": {
        "submodule": "datatable-head"
       }
      },
      "THEAD_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.HeaderView.html#property_THEAD_TEMPLATE",
       "!doc": "HTML templates used to create the `<thead>` containing the table headers.",
       "!data": {
        "submodule": "datatable-head"
       }
      },
      "render": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.HeaderView.html#method_render",
       "!doc": "Creates the `<thead>` Node content by assembling markup generated by\npopulating the `ROW_TEMPLATE` and `CELL_TEMPLATE` templates with content\nfrom the `columns` property.",
       "!data": {
        "submodule": "datatable-head"
       }
      }
     }
    },
    "Highlight": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Highlight.html",
     "prototype": {
      "highlightClassNames": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Highlight.html#property_highlightClassNames",
       "!doc": "An object consisting of classnames for a `row`, a `col` and a `cell` to\nbe applied to their respective objects when the user moves the mouse over\nthe item and the attribute is set to true.",
       "!data": {
        "submodule": "datatable-highlight"
       }
      }
     }
    },
    "KEY_NAMES": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property_KEY_NAMES",
     "!doc": "Mapping of key codes to friendly key names that can be used in the\n[keyActions](#property_keyActions) property and [ARIA_ACTIONS](#property_ARIA_ACTIONS)\nproperty.\n\nIt contains aliases for the following keys:\n    <ul>\n    <li>backspace</li>\n    <li>tab</li>\n    <li>enter</li>\n    <li>esc</li>\n    <li>space</li>\n    <li>pgup</li>\n    <li>pgdown</li>\n    <li>end</li>\n    <li>home</li>\n    <li>left</li>\n    <li>up</li>\n    <li>right</li>\n    <li>down</li>\n    <li>f1 .. f12</li>\n    </ul>",
     "!data": {
      "submodule": "datatable-keynav"
     }
    },
    "ARIA_ACTIONS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#property_ARIA_ACTIONS",
     "!doc": "Mapping of key codes to actions according to the WAI-ARIA suggestion for the\n[Grid Widget](http://www.w3.org/WAI/PF/aria-practices/#grid).\n\nThe key for each entry is a key-code or [keyName](#property_KEY_NAMES) while the\nvalue can be a function that performs the action or a string.  If a string,\nit can either correspond to the name of a method in this module (or  any\nmethod in a DataTable instance) or the name of an event to fire.",
     "!data": {
      "submodule": "datatable-keynav"
     }
    },
    "Paginator": {
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Paginator.html",
     "!data": {
      "for": "paginator.Paginator"
     },
     "View": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.View.html",
      "prototype": {
       "!proto": "app.View.prototype",
       "containerTemplate": {
        "!type": "string",
        "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.View.html#property_containerTemplate",
        "!doc": "Template for this views container.",
        "!data": {
         "submodule": "datatable-paginator"
        }
       },
       "contentTemplate": {
        "!type": "string",
        "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.View.html#property_contentTemplate",
        "!doc": "Template for content. Helps maintain order of controls.",
        "!data": {
         "submodule": "datatable-paginator"
        }
       },
       "initializer": {
        "!type": "fn()",
        "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.View.html#method_initializer",
        "!doc": "Sets classnames on the templates and bind events",
        "!data": {
         "submodule": "datatable-paginator"
        }
       },
       "render": {
        "!type": "fn() -> !this",
        "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.View.html#method_render",
        "!data": {
         "submodule": "datatable-paginator"
        }
       },
       "attachEvents": {
        "!type": "fn()",
        "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.View.html#method_attachEvents",
        "!data": {
         "submodule": "datatable-paginator"
        }
       }
      }
     },
     "prototype": {
      "firstPage": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#method_firstPage",
       "!doc": "Sets the `paginatorModel` to the first page.",
       "!data": {
        "submodule": "datatable-paginator"
       }
      },
      "lastPage": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#method_lastPage",
       "!doc": "Sets the `paginatorModel` to the last page.",
       "!data": {
        "submodule": "datatable-paginator"
       }
      },
      "previousPage": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#method_previousPage",
       "!doc": "Sets the `paginatorModel` to the previous page.",
       "!data": {
        "submodule": "datatable-paginator"
       }
      },
      "nextPage": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Paginator.html#method_nextPage",
       "!doc": "Sets the `paginatorModel` to the next page.",
       "!data": {
        "submodule": "datatable-paginator"
       }
      }
     }
    },
    "TableView": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html",
     "prototype": {
      "!proto": "app.View.prototype",
      "CAPTION_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#property_CAPTION_TEMPLATE",
       "!doc": "The HTML template used to create the caption Node if the `caption`\nattribute is set.",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "TABLE_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#property_TABLE_TEMPLATE",
       "!doc": "The HTML template used to create the table Node.",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "body": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#property_body",
       "!doc": "The object or instance of the class assigned to `bodyView` that is\nresponsible for rendering and managing the tables `<tbody>`(s) and its\ncontent.",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "foot": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#property_foot",
       "!doc": "The object or instance of the class assigned to `footerView` that is\nresponsible for rendering and managing the tables `<tfoot>` and its\ncontent.",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "head": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#property_head",
       "!doc": "The object or instance of the class assigned to `headerView` that is\nresponsible for rendering and managing the tables `<thead>` and its\ncontent.",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "getCell": {
       "!type": "fn(seed: [number]|+node.Node, shift?: [number]|string) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#method_getCell",
       "!doc": "Returns the `<td>` Node from the given row and column index.  Alternately,\nthe `seed` can be a Node.  If so, the nearest ancestor cell is returned.\nIf the `seed` is a cell, it is returned.  If there is no cell at the given\ncoordinates, `null` is returned.\n\nOptionally, include an offset array or string to return a cell near the\ncell identified by the `seed`.  The offset can be an array containing the\nnumber of rows to shift followed by the number of columns to shift, or one\nof \"above\", \"below\", \"next\", or \"previous\".\n\n<pre><code>// Previous cell in the previous row\nvar cell = table.getCell(e.target, [-1, -1]);\n\n// Next cell\nvar cell = table.getCell(e.target, next);\nvar cell = table.getCell(e.taregt, [0, 1];</pre></code>\n\nThis is actually just a pass through to the `bodyView` instances method\nby the same name.",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "getRecord": {
       "!type": "fn(seed: string|+node.Node) -> +app.Model",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#method_getRecord",
       "!doc": "Relays call to the `bodyView`s `getRecord` method if it has one.",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "getRow": {
       "!type": "fn(id: number|string|+app.Model) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#method_getRow",
       "!doc": "Returns the `<tr>` Node from the given row index, Model, or Models\n`clientId`.  If the rows havent been rendered yet, or if the row cant be\nfound by the input, `null` is returned.\n\nThis is actually just a pass through to the `bodyView` instances method\nby the same name.",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "displayColumns": {
       "!type": "[+Object]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#property_displayColumns",
       "!doc": "Array of the columns that correspond to those with value cells in the\ndata rows. Excludes colspan header columns (configured with `children`).",
       "!data": {
        "submodule": "datatable-table"
       }
      },
      "render": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.TableView.html#method_render",
       "!doc": "Creates the UI in the configured `container`.",
       "!data": {
        "submodule": "datatable-table"
       }
      }
     }
    }
   },
   "Plugin": {
    "DataTableDataSource": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DataTableDataSource.html",
     "prototype": {
      "!proto": "plugin.Plugin.Base.prototype",
      "load": {
       "!type": "fn(config: +config.Plugin.DataTableDataSourceLoadConfig)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DataTableDataSource.html#method_load",
       "!doc": "Load data by calling DataSources sendRequest() method under the hood.",
       "!data": {
        "submodule": "datatable-datasource"
       }
      },
      "onDataReturnInitializeTable": {
       "!type": "fn(e: +event_custom.EventFacade)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DataTableDataSource.html#method_onDataReturnInitializeTable",
       "!doc": "Callback function passed to DataSources sendRequest() method populates\nan entire DataTable with new data, clearing previous data, if any.",
       "!data": {
        "submodule": "datatable-datasource"
       }
      }
     },
     "NS": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DataTableDataSource.html#property_NS",
      "!doc": "The namespace for the plugin. This will be the property on the host which\nreferences the plugin instance.",
      "!data": {
       "submodule": "datatable-datasource"
      }
     },
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DataTableDataSource.html#property_NAME",
      "!doc": "Class name.",
      "!data": {
       "submodule": "datatable-datasource"
      }
     }
    }
   }
  },
  "datatable_formatters": {
   "!data": {
    "module": "datatable-formatters"
   },
   "DataTable": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html",
    "!data": {
     "for": "datatable.DataTable"
    },
    "prototype": {
     "!proto": "datatable.DataTable.Base.prototype"
    },
    "BodyView": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/BodyView.html",
     "prototype": {
      "!proto": "app.View.prototype"
     },
     "Formatters": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html",
      "button": {
       "!type": "fn(col: +Object) -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_button",
       "!doc": "Returns a formatter that produces a BUTTON element using the value of\nthe [buttonLabel](DataTable.Column.html#property_buttonLabel)\ncolumn definition attribute as its label or the text\n`Click` if not found.\n\nApplies the CSS className `yui3-datatable-button` to the cell."
      },
      "boolean": {
       "!type": "fn(col: +Object) -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_boolean",
       "!doc": "Returns a formatter function that returns the texts `\"true\"` or `\"false\"`\nand assigns the CSS classNames `yui3-datatable-true` or `yui3-datatable-false`\nbased on the value of the cell.\n\nIf either a [booleanLabels](DataTable.Column.html#property_booleanLabels)\n configuration object is defined for the column\nor a [booleanLabels](DataTable.html#attr_booleanLabels)\nconfiguration attribute is defined for the datatable,\nthe formatter will use the values for the properties `true` or `false`\nof either of those objects as the text to show.\n\nIt returns `null`s or `undefined`s unchanged so that the `emptyCellValue`\nconfiguration attribute will eventually apply.\n\n     {key:\"active\", formatter: \"boolean\", booleanLabels: {\n         \"true\": \"yes\",\n         \"false\": \"no\"\n     }}"
      },
      "currency": {
       "!type": "fn(col: +Object) -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_currency",
       "!doc": "Returns a formatter function that formats values as currency using\nthe [Number.format](Number.html#method_format) method.\nIt looks for the format to apply in the\n[currencyFormat](DataTable.Column.html#property_currencyFormat) property\nof the column or in the\n[currencyFormat](DataTable.html#attr_currencyFormat)\n attribute of the whole table.\n\n    {key: \"amount\", formatter: \"currency\", currencyFormat: {\n        decimalPlaces:2,\n        decimalSeparator: \",\",\n        thousandsSeparator: \".\",\n        suffix: \"&euro;\"\n    }}\n\nSee [Number.format](Number.html#method_format) for the available format specs.\n\nAnything that cannot be parsed as a number will be returned unchanged.\n\nApplies the CSS className `yui3-datatable-currency` to the cell."
      },
      "date": {
       "!type": "fn(col: +Object) -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_date",
       "!doc": "Returns a date formatting function.\nIt looks for the format to apply in the\n[dateFormat](DataTable.Column.html#property_dateFormat)\nproperty of the column or in the\n[dateFormat](DataTable.html#attr_dateFormat)\n attribute of the whole table.\n\n    {key: \"DOB\", formatter: \"date\", dateFormat: \"%I:%M:%S %p\"}\n\nSee [Date.format](Date.html#method_format) for the available format specs.\n\nAnything that is not a date is returned unchanged.\n\nApplies the CSS className `yui3-datatable-date` to the cell."
      },
      "localDate": {
       "!type": "fn() -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_localDate",
       "!doc": "Returns a date-only (no time part) formatting function using the current locale.\n\n    {key: \"DOB\", formatter: \"localDate\"}\n\nAnything that is not a date is returned unchanged.\n\nApplies the CSS className `yui3-datatable-date` to the cell."
      },
      "localTime": {
       "!type": "fn() -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_localTime",
       "!doc": "Returns a time-only (no date part) formatting function using the current locale.\n\n    {key: \"startTime\", formatter: \"localTime\"}\n\nAnything that is not a date is returned unchanged.\n\nApplies the CSS className `yui3-datatable-date` to the cell."
      },
      "localDateTime": {
       "!type": "fn() -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_localDateTime",
       "!doc": "Returns a date formatting function using the current locale.\n\n    {key: \"DOB\", formatter: \"localDateTime\"}\n\nAnything that is not a date is returned unchanged.\n\nApplies the CSS className `yui3-datatable-date` to the cell."
      },
      "email": {
       "!type": "fn(col: +Object) -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_email",
       "!doc": "Returns a function that produces email links.\nIf the column definition contains a property\n[linkFrom](DataTable.Column.html#property_linkFrom) it will use the value\nin that field for the link, otherwise, the same column value will be used for both\nlink and text.\n\n    {key: \"contact\", formatter: \"email\", linkFrom: \"contactEmail\"}\n\nIt will use the respective\n[emptyCellValue](DataTable.Column.html#property_emptyCellValue)\ncolumn configuration attribute\nfor each of the value and the link if either is empty.\nIf the link value is still empty, it will return the value with no link.\n\nApplies the CSS className `yui3-datatable-email` to the cell."
      },
      "link": {
       "!type": "fn(col: +Object) -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_link",
       "!doc": "Returns a function that produces links.\nIf the column definition contains a property\n[linkFrom](DataTable.Column.html#property_linkFrom) it will use the value\nin that field for the link, otherwise, the same column value will be used for both\nlink and text.\n\n    {key: \"company\", formatter: \"link\", linkFrom: \"webSite\"}\n\nIt will use the respective\n[emptyCellValue](DataTable.Column.html#property_emptyCellValue)\n column configuration attribute\nfor each of the value and the link if either is empty.\nIf the link value is still empty, it will return the value with no link.\n\nApplies the CSS className `yui3-datatable-link` to the cell."
      },
      "number": {
       "!type": "fn(col: +Object) -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_number",
       "!doc": "Returns a formatter function that formats values using\nthe [Number.format](Number.html#method_format) method.\nIt looks for the format to apply in the\n[numberFormat](DataTable.Column.html#property_numberFormat)\nproperty of the column or in the\n[numberFormat](DataTable.html#attr_numberFormat)\nattribute of the whole table.\n\n     {key: \"weight\", formatter: \"number\", numberFormat: {\n         decimalPlaces:2,\n         decimalSeparator: \",\",\n         thousandsSeparator: \",\",\n         suffix: \"kg\"\n     }}\n\nSee [Number.format](Number.html#method_format) for the available format specs.\n\nAnything that cannot be parsed as a number will be returned unchanged.\n\nApplies the CSS className `yui3-datatable-number` to the cell."
      },
      "lookup": {
       "!type": "fn(col: +Object) -> fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.BodyView.Formatters.html#method_lookup",
       "!doc": "Returns a formatter function that returns texts from a lookup table\nbased on the stored value.\n\nIt looks for the translation to apply in the\n[lookupTable](DataTable.Column.html#property_lookupTable) property of the\ncolumn in either of these two formats:\n\n    {key: \"status\", formatter: \"lookup\", lookupTable: {\n        0: \"unknown\",\n        1: \"requested\",\n        2: \"approved\",\n        3: \"delivered\"\n    }},\n    {key: \"otherStatus\", formatter: \"lookup\", lookupTable: [\n        {value:0, text: \"unknown\"},\n        {value:1, text: \"requested\"},\n        {value:2, text: \"approved\"},\n        {value:3, text: \"delivered\"}\n    ]}\n\nApplies the CSS className `yui3-datatable-lookup` to the cell."
      }
     }
    },
    "Column": {
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html",
     "!data": {
      "for": "datatable.DataTable.Column"
     },
     "prototype": {
      "buttonLabel": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_buttonLabel",
       "!doc": "Label to be shown in the face of a button produced by the\n[button](DataTable.BodyView.Formatters.html#method_button) formatter"
      },
      "booleanLabels": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_booleanLabels",
       "!doc": "Determines the texts to be shown to represent Boolean values when the\n[boolean](DataTable.BodyView.Formatters.html#method_boolean) formatter\nis used.\n\nThe attribute is an object with text values for properties `true` and `false`.\n\n    {key:\"active\", formatter: \"boolean\", booleanLabels: {\n        \"true\": \"yes\",\n        \"false\": \"no\"\n    }}"
      },
      "currencyFormat": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_currencyFormat",
       "!doc": "Format specification for columns using the\n[currency](DataTable.BodyView.Formatters.html#method_currency) formatter.\nIt contains an object as described in\n[Number.format](Number.html#method_format)."
      },
      "dateFormat": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_dateFormat",
       "!doc": "Format specification for columns using the\n[date](DataTable.BodyView.Formatters.html#method_date) formatter.\nIt contains a string as described in\n[Date.format](Date.html#method_format)."
      },
      "linkFrom": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_linkFrom",
       "!doc": "Name of the field that is to provide the link for a column using the\n[email](DataTable.BodyView.Formatters.html#method_email) or\n[link](DataTable.BodyView.Formatters.html#method_link)\nformatters."
      },
      "numberFormat": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_numberFormat",
       "!doc": "Format specification for columns using the\n[number](DataTable.BodyView.Formatters.html#method_number) formatter.\nIt contains an object as described in\n[Number.format](Number.html#method_format)."
      },
      "lookupTable": {
       "!type": "+Object|[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.Column.html#property_lookupTable",
       "!doc": "Map of values to text used to translate internal values to human readable text\nin columns using the [lookup](DataTable.BodyView.Formatters.html#method_lookup)\nformatter.\n\nThe map can be given in either of two formats:\n\n    {key: \"status\", formatter: \"lookup\", lookupTable: {\n        0: \"unknown\",\n        1: \"requested\",\n        2: \"approved\",\n        3: \"delivered\"\n    }},\n    {key: \"otherStatus\", formatter: \"lookup\", lookupTable: [\n        {value:0, text: \"unknown\"},\n        {value:1, text: \"requested\"},\n        {value:2, text: \"approved\"},\n        {value:3, text: \"delivered\"}\n    ]}\n\nThe last format is compatible with the [dropdown](DataTable.Editors.html#property_dropdown)\nand autocomplete-based editors, where the order of the items in the dropdown matters."
      }
     }
    }
   }
  },
  "datatable_scroll": {
   "!data": {
    "module": "datatable-scroll"
   },
   "DataTable": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html",
    "!data": {
     "for": "datatable.DataTable"
    },
    "prototype": {
     "!proto": "datatable.DataTable.Base.prototype",
     "scrollTo": {
      "!type": "fn(id: string|number|[number]|+node.Node) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DataTable.html#method_scrollTo",
      "!doc": "Scrolls a given row or cell into view if the table is scrolling.  Pass the\n`clientId` of a Model from the DataTables `data` ModelList or its row\nindex to scroll to a row or a [row index, column index] array to scroll to\na cell.  Alternately, to scroll to any element contained within the tables\nscrolling areas, pass its ID, or the Node itself (though you could just as\nwell call `node.scrollIntoView()` yourself, but hey, whatever)."
     }
    }
   }
  },
  "datatype_date": {
   "!data": {
    "module": "datatype-date",
    "submodules": {
     "datatype-date-format": {},
     "datatype-date-math": {},
     "datatype-date-parse": {}
    }
   },
   "Date": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html",
    "prototype": {
     "format": {
      "!type": "fn(oDate: +datatype_date.Date, oConfig: +Object) -> +HTML",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_format",
      "!doc": "Takes a native JavaScript Date and formats it as a string for display to user.",
      "!data": {
       "submodule": "datatype-date-format"
      }
     },
     "isValidDate": {
      "!type": "fn(oDate: +datatype_date.Date) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_isValidDate",
      "!doc": "Checks whether a native JavaScript Date contains a valid value.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "areEqual": {
      "!type": "fn(aDate: +datatype_date.Date, bDate: +datatype_date.Date) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_areEqual",
      "!doc": "Checks whether two dates correspond to the same date and time.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "isGreater": {
      "!type": "fn(aDate: +datatype_date.Date, bDate: +datatype_date.Date) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_isGreater",
      "!doc": "Checks whether the first date comes later than the second.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "isGreaterOrEqual": {
      "!type": "fn(aDate: +datatype_date.Date, bDate: +datatype_date.Date) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_isGreaterOrEqual",
      "!doc": "Checks whether the first date comes later than or is the same as\nthe second.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "isInRange": {
      "!type": "fn(aDate: +datatype_date.Date, bDate: +datatype_date.Date, cDate: +datatype_date.Date) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_isInRange",
      "!doc": "Checks whether the date is between two other given dates.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "addDays": {
      "!type": "fn(oDate: +datatype_date.Date, numDays: number) -> +datatype_date.Date",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_addDays",
      "!doc": "Adds a specified number of days to the given date.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "addMonths": {
      "!type": "fn(oDate: +datatype_date.Date, numMonths: number) -> +datatype_date.Date",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_addMonths",
      "!doc": "Adds a specified number of months to the given date.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "addYears": {
      "!type": "fn(oDate: +datatype_date.Date, numYears: number) -> +datatype_date.Date",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_addYears",
      "!doc": "Adds a specified number of years to the given date.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "listOfDatesInMonth": {
      "!type": "fn(oDate: +datatype_date.Date) -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_listOfDatesInMonth",
      "!doc": "Lists all dates in a given month.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "daysInMonth": {
      "!type": "fn(oDate: +datatype_date.Date) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_daysInMonth",
      "!doc": "Takes a native JavaScript Date and returns the number of days\nin the month that the given date belongs to.",
      "!data": {
       "submodule": "datatype-date-math"
      }
     },
     "parse": {
      "!type": "fn(data: +datatype_date.Date|number|string) -> +datatype_date.Date",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Date.html#method_parse",
      "!doc": "Converts data to type Date.",
      "!data": {
       "submodule": "datatype-date-parse"
      }
     }
    }
   }
  },
  "dd": {
   "!data": {
    "submodules": {
     "dd-constrain": {},
     "dd-drop-plugin": {},
     "dd-plugin": {},
     "dd-ddm-base": {},
     "dd-ddm-drop": {},
     "dd-ddm": {},
     "dd-delegate": {},
     "dd-gestures": {},
     "dd-drag": {},
     "dd-drop": {},
     "dd-proxy": {},
     "dd-scroll": {}
    }
   },
   "Plugin": {
    "DDConstrained": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html",
     "prototype": {
      "!proto": "base.Base.prototype",
      "resetCache": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#method_resetCache",
       "!doc": "Reset the internal region cache.",
       "!data": {
        "submodule": "dd-constrain"
       }
      },
      "getRegion": {
       "!type": "fn(inc: bool) -> +Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#method_getRegion",
       "!doc": "Get the active region: viewport, node, custom region",
       "!data": {
        "submodule": "dd-constrain"
       }
      },
      "inRegion": {
       "!type": "fn(xy: [?]) -> bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#method_inRegion",
       "!doc": "Checks if the XY passed or the dragNode is inside the active region.",
       "!data": {
        "submodule": "dd-constrain"
       }
      },
      "align": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#method_align",
       "!doc": "Modifies the Drag.actXY method from the after drag:align event. This is where the constraining happens.",
       "!data": {
        "submodule": "dd-constrain"
       }
      },
      "drag": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.DDConstrained.html#method_drag",
       "!doc": "Fires after drag:drag. Handle the tickX and tickX align events.",
       "!data": {
        "submodule": "dd-constrain"
       }
      }
     }
    },
    "Drop": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Drop.html",
     "prototype": {
      "!proto": "dd.DD.Drop.prototype",
      "NAME": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Drop.html#property_NAME",
       "!doc": "dd-drop-plugin",
       "!data": {
        "submodule": "dd-drop-plugin"
       }
      },
      "NS": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Drop.html#property_NS",
       "!doc": "The Drop instance will be placed on the Node instance under the drop namespace. It can be accessed via Node.drop;",
       "!data": {
        "submodule": "dd-drop-plugin"
       }
      }
     }
    },
    "Drag": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Drag.html",
     "prototype": {
      "!proto": "dd.DD.Drag.prototype",
      "NAME": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Drag.html#property_NAME",
       "!doc": "dd-plugin",
       "!data": {
        "submodule": "dd-plugin"
       }
      },
      "NS": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Drag.html#property_NS",
       "!doc": "The Drag instance will be placed on the Node instance under the dd namespace. It can be accessed via Node.dd;",
       "!data": {
        "submodule": "dd-plugin"
       }
      }
     }
    }
   },
   "DD": {
    "DDM": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html",
     "prototype": {
      "!proto": "base.Base.prototype",
      "_active": {
       "!type": "bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#property__active",
       "!doc": "flag set when we activate our first drag, so DDM can start listening for events.",
       "!data": {
        "submodule": "dd-ddm-base"
       }
      },
      "CSS_PREFIX": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#property_CSS_PREFIX",
       "!doc": "The PREFIX to attach to all DD CSS class names",
       "!data": {
        "submodule": "dd-ddm-base"
       }
      },
      "activeDrag": {
       "!type": "+Drag",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#property_activeDrag",
       "!doc": "A reference to the currently active draggable object.",
       "!data": {
        "submodule": "dd-ddm-base"
       }
      },
      "stopDrag": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_stopDrag",
       "!doc": "Method will forcefully stop a drag operation. For example calling this from inside an ESC keypress handler will stop this drag.",
       "!data": {
        "submodule": "dd-ddm-base"
       }
      },
      "getDrag": {
       "!type": "fn(node: string|+Object) -> +Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_getDrag",
       "!doc": "Get a valid Drag instance back from a Node or a selector string, false otherwise",
       "!data": {
        "submodule": "dd-ddm-base"
       }
      },
      "swapPosition": {
       "!type": "fn(n1: +node.Node, n2: +node.Node) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_swapPosition",
       "!doc": "Swap the position of 2 nodes based on their CSS positioning.",
       "!data": {
        "submodule": "dd-ddm-base"
       }
      },
      "getNode": {
       "!type": "fn(n: +node.Node|+Object|string) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_getNode",
       "!doc": "Return a node instance from the given node, selector string or Y.Base extended object.",
       "!data": {
        "submodule": "dd-ddm-base"
       }
      },
      "swapNode": {
       "!type": "fn(n1: +node.Node, n2: +node.Node) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_swapNode",
       "!doc": "Swap the position of 2 nodes based on their DOM location.",
       "!data": {
        "submodule": "dd-ddm-base"
       }
      },
      "syncActiveShims": {
       "!type": "fn(force: bool)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_syncActiveShims",
       "!doc": "This method will sync the position of the shims on the Drop Targets that are currently active.",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "useHash": {
       "!type": "bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#property_useHash",
       "!doc": "Should we only check targets that are in the viewport on drags (for performance), default: true",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "activeDrop": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#property_activeDrop",
       "!doc": "A reference to the active Drop Target",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "validDrops": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#property_validDrops",
       "!doc": "An array of the valid Drop Targets for this interaction.",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "otherDrops": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#property_otherDrops",
       "!doc": "An object literal of Other Drop Targets that we encountered during this interaction (in the case of overlapping Drop Targets)",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "targets": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#property_targets",
       "!doc": "All of the Targets",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "isOverTarget": {
       "!type": "fn(drop: +Object) -> bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_isOverTarget",
       "!doc": "Check to see if the Drag element is over the target, method varies on current mode",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "clearCache": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_clearCache",
       "!doc": "Clears the cache data used for this interaction.",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "getBestMatch": {
       "!type": "fn(drops: [?], all: bool) -> +Object or Array",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_getBestMatch",
       "!doc": "This method will gather the area for all potential targets and see which has the hightest covered area and return it.",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "getDrop": {
       "!type": "fn(node: string|+Object) -> +Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_getDrop",
       "!doc": "Get a valid Drop instance back from a Node or a selector string, false otherwise",
       "!data": {
        "submodule": "dd-ddm-drop"
       }
      },
      "regDelegate": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_regDelegate",
       "!doc": "Register a Delegate with the DDM",
       "!data": {
        "submodule": "dd-delegate"
       }
      },
      "getDelegate": {
       "!type": "fn() -> ?",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.DDM.html#method_getDelegate",
       "!doc": "Get a delegate instance from a container node",
       "!data": {
        "submodule": "dd-delegate"
       }
      }
     }
    },
    "Delegate": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html",
     "prototype": {
      "!proto": "base.Base.prototype",
      "dd": {
       "!type": "?",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#property_dd",
       "!doc": "A reference to the temporary dd instance used under the hood.",
       "!data": {
        "submodule": "dd-delegate"
       }
      },
      "syncTargets": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#method_syncTargets",
       "!doc": "Applies the Y.Plugin.Drop to all nodes matching the cont + nodes selector query.",
       "!data": {
        "submodule": "dd-delegate"
       }
      },
      "createDrop": {
       "!type": "fn(node: +node.Node, groups: [?]) -> ?",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Delegate.html#method_createDrop",
       "!doc": "Apply the Drop plugin to this node",
       "!data": {
        "submodule": "dd-delegate"
       }
      }
     }
    },
    "Drag": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html",
     "prototype": {
      "!proto": "base.Base.prototype",
      "addToGroup": {
       "!type": "fn(g: string) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_addToGroup",
       "!doc": "Add this Drag instance to a group, this should be used for on-the-fly group additions.",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "removeFromGroup": {
       "!type": "fn(g: string) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_removeFromGroup",
       "!doc": "Remove this Drag instance from a group, this should be used for on-the-fly group removals.",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "target": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_target",
       "!doc": "This will be a reference to the Drop instance associated with this drag if the target: true config attribute is set..",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "deltaXY": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_deltaXY",
       "!doc": "The offset of the mouse position to the elements position",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "startXY": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_startXY",
       "!doc": "The initial mouse position",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "nodeXY": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_nodeXY",
       "!doc": "The initial element position",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "lastXY": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_lastXY",
       "!doc": "The position of the element as its moving (for offset calculations)",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "actXY": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_actXY",
       "!doc": "The xy that the node will be set to. Changing this will alter the position as its dragged.",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "realXY": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_realXY",
       "!doc": "The real xy position of the node.",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "mouseXY": {
       "!type": "[?]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_mouseXY",
       "!doc": "The XY coords of the mousemove",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "region": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_region",
       "!doc": "A region object associated with this drag, used for checking regions while dragging.",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "validClick": {
       "!type": "fn(ev: +event_custom.EventFacade) -> bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_validClick",
       "!doc": "Method first checks to see if we have handles, if so it validates the click\nagainst the handle. Then if it finds a valid handle, it checks it against\nthe invalid handles list. Returns true if a good handle was used, false otherwise.",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "removeHandle": {
       "!type": "fn(str: string) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_removeHandle",
       "!doc": "Remove a Selector added by addHandle",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "addHandle": {
       "!type": "fn(str: string) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_addHandle",
       "!doc": "Add a handle to a drag element. Drag only initiates when a mousedown happens on this element.",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "removeInvalid": {
       "!type": "fn(str: string) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_removeInvalid",
       "!doc": "Remove an invalid handle added by addInvalid",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "addInvalid": {
       "!type": "fn(str: string) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_addInvalid",
       "!doc": "Add a selector string to test the handle against. If the test passes the drag operation will not continue.",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "start": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_start",
       "!doc": "Starts the drag operation",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "end": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_end",
       "!doc": "Ends the drag operation",
       "!data": {
        "submodule": "dd-drag"
       }
      },
      "stopDrag": {
       "!type": "fn() -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#method_stopDrag",
       "!doc": "Method will forcefully stop a drag operation. For example calling this from inside an ESC keypress handler will stop this drag.",
       "!data": {
        "submodule": "dd-drag"
       }
      }
     },
     "START_EVENT": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drag.html#property_START_EVENT",
      "!doc": "This property defaults to \"mousedown\", but when drag-gestures is loaded, it is changed to \"gesturemovestart\"",
      "!data": {
       "submodule": "dd-drag"
      }
     }
    },
    "Drop": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html",
     "prototype": {
      "!proto": "base.Base.prototype",
      "addToGroup": {
       "!type": "fn(g: string) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#method_addToGroup",
       "!doc": "Add this Drop instance to a group, this should be used for on-the-fly group additions.",
       "!data": {
        "submodule": "dd-drop"
       }
      },
      "removeFromGroup": {
       "!type": "fn(g: string) -> !this",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#method_removeFromGroup",
       "!doc": "Remove this Drop instance from a group, this should be used for on-the-fly group removals.",
       "!data": {
        "submodule": "dd-drop"
       }
      },
      "shim": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#property_shim",
       "!doc": "Node reference to the targets shim",
       "!data": {
        "submodule": "dd-drop"
       }
      },
      "region": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#property_region",
       "!doc": "A region object associated with this target, used for checking regions while dragging.",
       "!data": {
        "submodule": "dd-drop"
       }
      },
      "overTarget": {
       "!type": "bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#property_overTarget",
       "!doc": "This flag is tripped when a drag element is over this target.",
       "!data": {
        "submodule": "dd-drop"
       }
      },
      "inGroup": {
       "!type": "fn(groups: [?]) -> ?",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#method_inGroup",
       "!doc": "Check if this target is in one of the supplied groups.",
       "!data": {
        "submodule": "dd-drop"
       }
      },
      "sizeShim": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Drop.html#method_sizeShim",
       "!doc": "Positions and sizes the shim with the raw data from the node,\nthis can be used to programatically adjust the Targets shim for Animation..",
       "!data": {
        "submodule": "dd-drop"
       }
      }
     }
    },
    "Scroll": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html",
     "prototype": {
      "!proto": "base.Base.prototype",
      "align": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#method_align",
       "!doc": "Called from the drag:align event to determine if we need to scroll.",
       "!data": {
        "submodule": "dd-scroll"
       }
      },
      "start": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#method_start",
       "!doc": "Called from the drag:start event",
       "!data": {
        "submodule": "dd-scroll"
       }
      },
      "end": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/DD.Scroll.html#method_end",
       "!doc": "Called from the drag:end event",
       "!data": {
        "submodule": "dd-scroll"
       }
      }
     }
    }
   }
  },
  "dial": {
   "Dial": {
    "!type": "fn(config: +config.DialConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html",
    "prototype": {
     "!proto": "widget.Widget.prototype",
     "syncUI": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Dial.html#method_syncUI",
      "!doc": "Synchronizes the DOM state with the attribute settings."
     }
    }
   }
  },
  "dom": {
   "!data": {
    "submodules": {
     "dom-base": {},
     "dom-screen": {},
     "dom-style": {},
     "selector-css2": {},
     "selector-css3": {},
     "selector-native": {}
    }
   },
   "DOM": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html",
    "prototype": {
     "getText": {
      "!type": "fn(element: +HTMLElement) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_getText",
      "!doc": "Returns the text content of the HTMLElement."
     },
     "setText": {
      "!type": "fn(element: +HTMLElement, content: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setText",
      "!doc": "Sets the text content of the HTMLElement."
     },
     "setAttribute": {
      "!type": "fn(el: +HTMLElement, attr: string, val: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setAttribute",
      "!doc": "Provides a normalized attribute interface."
     },
     "getAttribute": {
      "!type": "fn(el: +HTMLElement, attr: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_getAttribute",
      "!doc": "Provides a normalized attribute interface."
     },
     "hasClass": {
      "!type": "fn(element: +HTMLElement, className: string) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_hasClass",
      "!doc": "Determines whether a DOM element has the given className."
     },
     "addClass": {
      "!type": "fn(element: +HTMLElement, className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_addClass",
      "!doc": "Adds a class name to a given DOM element."
     },
     "removeClass": {
      "!type": "fn(element: +HTMLElement, className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_removeClass",
      "!doc": "Removes a class name from a given element."
     },
     "replaceClass": {
      "!type": "fn(element: +HTMLElement, oldClassName: string, newClassName: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_replaceClass",
      "!doc": "Replace a class with another class for a given element.\nIf no oldClassName is present, the newClassName is simply added."
     },
     "toggleClass": {
      "!type": "fn(element: +HTMLElement, className: string, addClass: bool)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_toggleClass",
      "!doc": "If the className exists on the node it is removed, if it doesnt exist it is added."
     },
     "byId": {
      "!type": "fn(id: string, doc: +Object) -> +HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_byId",
      "!doc": "Returns the HTMLElement with the given ID (Wrapper for document.getElementById).",
      "!data": {
       "submodule": "dom-base"
      }
     },
     "elementByAxis": {
      "!type": "fn(element: +HTMLElement, axis: string, fn?: fn(), all?: bool) -> +HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_elementByAxis",
      "!doc": "Searches the element by the given axis for the first matching element.",
      "!data": {
       "submodule": "dom-base"
      }
     },
     "contains": {
      "!type": "fn(element: +HTMLElement, needle: +HTMLElement) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_contains",
      "!doc": "Determines whether or not one HTMLElement is or contains another HTMLElement.",
      "!data": {
       "submodule": "dom-base"
      }
     },
     "inDoc": {
      "!type": "fn(element: +HTMLElement, doc: +HTMLElement) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_inDoc",
      "!doc": "Determines whether or not the HTMLElement is part of the document.",
      "!data": {
       "submodule": "dom-base"
      }
     },
     "create": {
      "!type": "fn(html: string, doc: +HTMLDocument) -> +HTMLElement|+DocumentFragment",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_create",
      "!doc": "Creates a new dom node using the provided markup string."
     },
     "addHTML": {
      "!type": "fn(node: +HTMLElement, content: +HTMLElement|[?]|+HTMLCollection, where: +HTMLElement)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_addHTML",
      "!doc": "Inserts content in a node at the given location"
     },
     "region": {
      "!type": "fn(element: +HTMLElement) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_region",
      "!doc": "Returns an Object literal containing the following about this element: (top, right, bottom, left)"
     },
     "intersect": {
      "!type": "fn(element: +HTMLElement, element2: +HTMLElement|+Object, altRegion: +Object) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_intersect",
      "!doc": "Find the intersect information for the passed nodes."
     },
     "inRegion": {
      "!type": "fn(node: +Object, node2: +Object, all: bool, altRegion: +Object) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_inRegion",
      "!doc": "Check if any part of this node is in the passed region"
     },
     "inViewportRegion": {
      "!type": "fn(element: +HTMLElement, all: bool, altRegion: +Object) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_inViewportRegion",
      "!doc": "Check if any part of this element is in the viewport"
     },
     "viewportRegion": {
      "!type": "fn() -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_viewportRegion",
      "!doc": "Returns an Object literal containing the following about the visible region of viewport: (top, right, bottom, left)"
     },
     "winHeight": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_winHeight",
      "!doc": "Returns the inner height of the viewport (exludes scrollbar).",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "winWidth": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_winWidth",
      "!doc": "Returns the inner width of the viewport (exludes scrollbar).",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "docHeight": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_docHeight",
      "!doc": "Document height",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "docWidth": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_docWidth",
      "!doc": "Document width",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "docScrollX": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_docScrollX",
      "!doc": "Amount page has been scroll horizontally",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "docScrollY": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_docScrollY",
      "!doc": "Amount page has been scroll vertically",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "getXY": {
      "!type": "fn(element: ?) -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_getXY",
      "!doc": "Gets the current position of an element based on page coordinates.\nElement must be part of the DOM tree to have page coordinates\n(display:none or elements not appended return false).",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "getScrollbarWidth": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_getScrollbarWidth",
      "!doc": "Gets the width of vertical scrollbars on overflowed containers in the body\ncontent.",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "getX": {
      "!type": "fn(element: ?) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_getX",
      "!doc": "Gets the current X position of an element based on page coordinates.\nElement must be part of the DOM tree to have page coordinates\n(display:none or elements not appended return false).",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "getY": {
      "!type": "fn(element: ?) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_getY",
      "!doc": "Gets the current Y position of an element based on page coordinates.\nElement must be part of the DOM tree to have page coordinates\n(display:none or elements not appended return false).",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "setXY": {
      "!type": "fn(element: ?, xy: [?], noRetry: bool)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setXY",
      "!doc": "Set the position of an html element in page coordinates.\nThe element must be part of the DOM tree to have page coordinates (display:none or elements not appended return false).",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "setX": {
      "!type": "fn(element: ?, x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setX",
      "!doc": "Set the X position of an html element in page coordinates, regardless of how the element is positioned.\nThe element(s) must be part of the DOM tree to have page coordinates (display:none or elements not appended return false).",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "setY": {
      "!type": "fn(element: ?, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setY",
      "!doc": "Set the Y position of an html element in page coordinates, regardless of how the element is positioned.\nThe element(s) must be part of the DOM tree to have page coordinates (display:none or elements not appended return false).",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "swapXY": {
      "!type": "fn(node: +node.Node, otherNode: +node.Node) -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_swapXY",
      "!doc": "Swap the xy position with another node",
      "!data": {
       "submodule": "dom-screen"
      }
     },
     "setWidth": {
      "!type": "fn(element: +HTMLElement, size: string|number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setWidth",
      "!doc": "Sets the width of the element to the given size, regardless\nof box model, border, padding, etc."
     },
     "setHeight": {
      "!type": "fn(element: +HTMLElement, size: string|number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setHeight",
      "!doc": "Sets the height of the element to the given size, regardless\nof box model, border, padding, etc."
     },
     "setStyle": {
      "!type": "fn(node: +HTMLElement, att: string, val: string|number, style?: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setStyle",
      "!doc": "Sets a style property for a given element.",
      "!data": {
       "submodule": "dom-style"
      }
     },
     "getStyle": {
      "!type": "fn(node: +HTMLElement, att: string, style?: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_getStyle",
      "!doc": "Returns the current style value for the given property.",
      "!data": {
       "submodule": "dom-style"
      }
     },
     "setStyles": {
      "!type": "fn(node: +HTMLElement, hash: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_setStyles",
      "!doc": "Sets multiple style properties.",
      "!data": {
       "submodule": "dom-style"
      }
     },
     "getComputedStyle": {
      "!type": "fn(node: +HTMLElement, att: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOM.html#method_getComputedStyle",
      "!doc": "Returns the computed style for the given node.",
      "!data": {
       "submodule": "dom-style"
      }
     }
    }
   },
   "Selector": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Selector.html",
    "prototype": {
     "shorthand": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Selector.html#property_shorthand",
      "!doc": "Mapping of shorthand tokens to corresponding attribute selector",
      "!data": {
       "submodule": "selector-css2"
      }
     },
     "operators": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Selector.html#property_operators",
      "!doc": "List of operators and corresponding boolean functions.\nThese functions are passed the attribute and the current nodes value of the attribute.",
      "!data": {
       "submodule": "selector-css2"
      }
     }
    },
    "useNative": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Selector.html#property_useNative",
     "!doc": "Use the native version of `querySelectorAll`, if it exists.",
     "!data": {
      "submodule": "selector-native"
     }
    },
    "query": {
     "!type": "fn(selector: string, root: +HTMLElement, firstOnly: bool) -> [+HTMLElement]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Selector.html#method_query",
     "!doc": "Retrieves a set of nodes based on a given CSS selector.",
     "!data": {
      "submodule": "selector-native"
     }
    },
    "filter": {
     "!type": "fn(nodes: [+HTMLElement], selector: string) -> [+HTMLElement]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Selector.html#method_filter",
     "!doc": "Filters out nodes that do not match the given CSS selector.",
     "!data": {
      "submodule": "selector-native"
     }
    },
    "test": {
     "!type": "fn(node: +HTMLElement, selector: string, root: +HTMLElement) -> bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Selector.html#method_test",
     "!doc": "Determines whether or not the given node matches the given CSS selector.",
     "!data": {
      "submodule": "selector-native"
     }
    },
    "ancestor": {
     "!type": "fn(node: +HTMLElement, selector: string, testSelf: bool) -> +HTMLElement",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Selector.html#method_ancestor",
     "!doc": "A convenience method to emulate Y.Nodes aNode.ancestor(selector).",
     "!data": {
      "submodule": "selector-native"
     }
    }
   }
  },
  "dump": {
   "YUI": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html",
    "!data": {
     "extends": [
      "event_custom.EventTarget"
     ],
     "for": "yui.YUI"
    },
    "prototype": {
     "dump": {
      "!type": "fn(o: +Object, d: number) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_dump",
      "!doc": "Returns a simple string representation of the object or array.\nOther types of objects will be returned unprocessed.  Arrays\nare expected to be indexed."
     }
    }
   }
  },
  "editor": {
   "!data": {
    "submodules": {
     "content-editable": {},
     "createlink-base": {},
     "editor-base": {},
     "editor-bidi": {},
     "editor-br": {},
     "editor-inline": {},
     "editor-para-base": {},
     "editor-para-ie": {},
     "editor-para": {},
     "selection": {},
     "editor-tab": {},
     "exec-command": {},
     "frame": {},
     "editor-lists": {}
    }
   },
   "ContentEditable": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html",
    "prototype": {
     "!proto": "Y.Plugin.Base.prototype",
     "use": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#method_use",
      "!doc": "This is a scoped version of the normal YUI.use method & is bound to the ContentEditable element\nAt setup, the inst.use method is mapped to this method.",
      "!data": {
       "submodule": "content-editable"
      }
     },
     "delegate": {
      "!type": "fn(type: string, fn: fn(), cont: +String, Node, sel: string) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#method_delegate",
      "!doc": "A delegate method passed to the instances delegate method",
      "!data": {
       "submodule": "content-editable"
      }
     },
     "getInstance": {
      "!type": "fn() -> +yui.YUI",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#method_getInstance",
      "!doc": "Get a reference to the internal YUI instance.",
      "!data": {
       "submodule": "content-editable"
      }
     },
     "render": {
      "!type": "fn(node: string|+HTMLElement|+node.Node) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#method_render",
      "!data": {
       "submodule": "content-editable"
      }
     },
     "focus": {
      "!type": "fn(fn: fn()) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#method_focus",
      "!doc": "Set the focus to the container",
      "!data": {
       "submodule": "content-editable"
      }
     },
     "show": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#method_show",
      "!doc": "Show the iframe instance",
      "!data": {
       "submodule": "content-editable"
      }
     },
     "hide": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#method_hide",
      "!doc": "Hide the iframe instance",
      "!data": {
       "submodule": "content-editable"
      }
     }
    },
    "THROTTLE_TIME": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#property_THROTTLE_TIME",
     "!doc": "The throttle time for key events in IE",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "DOM_EVENTS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#property_DOM_EVENTS",
     "!doc": "The DomEvents that the frame automatically attaches and bubbles",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "HTML": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#property_HTML",
     "!doc": "The template string used to create the ContentEditable element",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#property_NAME",
     "!doc": "The name of the class (contentEditable)",
     "!data": {
      "submodule": "content-editable"
     }
    },
    "NS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ContentEditable.html#property_NS",
     "!doc": "The namespace on which ContentEditable plugin will reside.",
     "!data": {
      "submodule": "content-editable"
     }
    }
   },
   "Plugin": {
    "CreateLinkBase": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CreateLinkBase.html",
     "STRINGS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CreateLinkBase.html#property_STRINGS",
      "!doc": "Strings used by the plugin"
     },
     "PROMPT": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CreateLinkBase.html#property_PROMPT",
      "!doc": "String used for the Prompt"
     },
     "DEFAULT": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.CreateLinkBase.html#property_DEFAULT",
      "!doc": "String used as the default value of the Prompt"
     }
    },
    "ExecCommand": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ExecCommand.html",
     "prototype": {
      "!proto": "base.Base.prototype",
      "bidi": {
       "!type": "?",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.html#property_bidi",
       "!doc": "bidi execCommand override for setting the text direction of a node.\nThis property is added to the `Y.Plugin.ExecCommands.COMMANDS`\ncollection.",
       "!data": {
        "submodule": "editor-bidi"
       }
      },
      "command": {
       "!type": "fn(action: string, value: string) -> +node.Node|+node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.html#method_command",
       "!doc": "Execute a command on the frames document.",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "getInstance": {
       "!type": "fn() -> +yui.YUI",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.html#method_getInstance",
       "!doc": "Gets the instance of YUI bound to the parent frame",
       "!data": {
        "submodule": "exec-command"
       }
      }
     },
     "COMMANDS": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html",
      "createlink": {
       "!type": "fn(cmd: string) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_createlink",
       "!doc": "Override for the createlink method from the <a href=\"Plugin.CreateLinkBase.html\">CreateLinkBase</a> plugin."
      },
      "wrap": {
       "!type": "fn(cmd: string, tag: string) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_wrap",
       "!doc": "Wraps the content with a new element of type (tag)",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "inserthtml": {
       "!type": "fn(cmd: string, html: string) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_inserthtml",
       "!doc": "Inserts the provided HTML at the cursor, should be a single element.",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "insertandfocus": {
       "!type": "fn(cmd: string, html: string) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_insertandfocus",
       "!doc": "Inserts the provided HTML at the cursor, and focuses the cursor afterwards.",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "insertbr": {
       "!type": "fn(cmd: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_insertbr",
       "!doc": "Inserts a BR at the current cursor position",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "insertimage": {
       "!type": "fn(cmd: string, img: string) -> +node.Node",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_insertimage",
       "!doc": "Inserts an image at the cursor position",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "addclass": {
       "!type": "fn(cmd: string, cls: string) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_addclass",
       "!doc": "Add a class to all of the elements in the selection",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "removeclass": {
       "!type": "fn(cmd: string, cls: string) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_removeclass",
       "!doc": "Remove a class from all of the elements in the selection",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "forecolor": {
       "!type": "fn(cmd: string, val: string) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_forecolor",
       "!doc": "Adds a forecolor to the current selection, or creates a new element and applies it",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "backcolor": {
       "!type": "fn(cmd: string, val: string) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_backcolor",
       "!doc": "Adds a background color to the current selection, or creates a new element and applies it",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "hilitecolor": {
       "!type": "fn(cmd: string, val: string) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_hilitecolor",
       "!doc": "Sugar method, calles backcolor",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "fontname2": {
       "!type": "fn(cmd: string, val: string) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_fontname2",
       "!doc": "Adds a font name to the current selection, or creates a new element and applies it",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "fontsize2": {
       "!type": "fn(cmd: string, val: string) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_fontsize2",
       "!doc": "Adds a fontsize to the current selection, or creates a new element and applies it",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "insertorderedlist": {
       "!type": "fn(cmd: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_insertorderedlist",
       "!doc": "Overload for list",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "insertunorderedlist": {
       "!type": "fn(cmd: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_insertunorderedlist",
       "!doc": "Overload for list",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "list": {
       "!type": "fn(cmd: string, tag: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_list",
       "!doc": "Noramlizes lists creation/destruction for IE. All others pass through to native calls",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "justify": {
       "!type": "fn(cmd: string, val: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_justify",
       "!doc": "Noramlizes alignment for Webkit Browsers",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "justifycenter": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_justifycenter",
       "!doc": "Override method for justify",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "justifyleft": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_justifyleft",
       "!doc": "Override method for justify",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "justifyright": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_justifyright",
       "!doc": "Override method for justify",
       "!data": {
        "submodule": "exec-command"
       }
      },
      "justifyfull": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.COMMANDS.html#method_justifyfull",
       "!doc": "Override method for justify",
       "!data": {
        "submodule": "exec-command"
       }
      }
     },
     "NAME": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.html#property_NAME",
      "!doc": "execCommand",
      "!data": {
       "submodule": "exec-command"
      }
     },
     "NS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ExecCommand.html#property_NS",
      "!doc": "exec",
      "!data": {
       "submodule": "exec-command"
      }
     }
    },
    "EditorBidi": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html",
     "prototype": {
      "!proto": "base.Base.prototype"
     },
     "EVENTS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#property_EVENTS",
      "!doc": "The events to check for a direction change on",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "BLOCKS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#property_BLOCKS",
      "!doc": "More elements may be needed. BODY *must* be in the list to take care of the special case.\n\nblockParent could be changed to use inst.EditorSelection.BLOCKS\ninstead, but that would make Y.Plugin.EditorBidi.blockParent\nunusable in non-RTE contexts (it being usable is a nice\nside-effect).",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "DIV_WRAPPER": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#property_DIV_WRAPPER",
      "!doc": "Template for creating a block element",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "blockParent": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#method_blockParent",
      "!doc": "Returns a block parent for a given element",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "_NODE_SELECTED": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#property__NODE_SELECTED",
      "!doc": "The data key to store on the node.",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "addParents": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#method_addParents",
      "!doc": "Generates a list of all the block parents of the current NodeList",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "NAME": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#property_NAME",
      "!doc": "editorBidi",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "NS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#property_NS",
      "!doc": "editorBidi",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "RE_TEXT_ALIGN": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#property_RE_TEXT_ALIGN",
      "!doc": "Regex for testing/removing text-align style from an element",
      "!data": {
       "submodule": "editor-bidi"
      }
     },
     "removeTextAlign": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBidi.html#method_removeTextAlign",
      "!doc": "Method to test a nodes style attribute for text-align and removing it.",
      "!data": {
       "submodule": "editor-bidi"
      }
     }
    },
    "EditorBR": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBR.html",
     "prototype": {
      "!proto": "base.Base.prototype"
     },
     "NAME": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBR.html#property_NAME",
      "!doc": "editorBR",
      "!data": {
       "submodule": "editor-br"
      }
     },
     "NS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorBR.html#property_NS",
      "!doc": "editorBR",
      "!data": {
       "submodule": "editor-br"
      }
     }
    },
    "EditorParaBase": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorParaBase.html",
     "prototype": {
      "!proto": "base.Base.prototype"
     },
     "NAME": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorParaBase.html#property_NAME",
      "!doc": "editorPara",
      "!data": {
       "submodule": "editor-para-base"
      }
     },
     "NS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorParaBase.html#property_NS",
      "!doc": "editorPara",
      "!data": {
       "submodule": "editor-para-base"
      }
     }
    },
    "EditorParaIE": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorParaIE.html",
     "prototype": {
      "!proto": "editor.Plugin.EditorParaBase.prototype"
     },
     "NAME": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorParaIE.html#property_NAME",
      "!doc": "editorPara",
      "!data": {
       "submodule": "editor-para-ie"
      }
     },
     "NS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorParaIE.html#property_NS",
      "!doc": "editorPara",
      "!data": {
       "submodule": "editor-para-ie"
      }
     }
    },
    "EditorPara": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorPara.html",
     "prototype": {
      "!proto": "editor.Plugin.EditorParaBase.prototype"
     },
     "NAME": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorPara.html#property_NAME",
      "!doc": "editorPara",
      "!data": {
       "submodule": "editor-para"
      }
     },
     "NS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorPara.html#property_NS",
      "!doc": "editorPara",
      "!data": {
       "submodule": "editor-para"
      }
     }
    },
    "EditorTab": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorTab.html",
     "prototype": {
      "!proto": "base.Base.prototype"
     },
     "NAME": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorTab.html#property_NAME",
      "!doc": "editorTab",
      "!data": {
       "submodule": "editor-tab"
      }
     },
     "NS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorTab.html#property_NS",
      "!doc": "tab",
      "!data": {
       "submodule": "editor-tab"
      }
     }
    },
    "EditorLists": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorLists.html",
     "prototype": {
      "!proto": "base.Base.prototype"
     },
     "NON": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorLists.html#property_NON",
      "!doc": "The non element placeholder, used for positioning the cursor and filling empty items",
      "!data": {
       "submodule": "editor-lists"
      }
     },
     "NONSEL": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorLists.html#property_NONSEL",
      "!doc": "The selector query to get all non elements",
      "!data": {
       "submodule": "editor-lists"
      }
     },
     "REMOVE": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorLists.html#property_REMOVE",
      "!doc": "The items to removed from a list when a list item is moved, currently removes BR nodes",
      "!data": {
       "submodule": "editor-lists"
      }
     },
     "NAME": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorLists.html#property_NAME",
      "!doc": "editorLists",
      "!data": {
       "submodule": "editor-lists"
      }
     },
     "NS": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.EditorLists.html#property_NS",
      "!doc": "lists",
      "!data": {
       "submodule": "editor-lists"
      }
     }
    }
   },
   "EditorBase": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "frame": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#property_frame",
      "!doc": "Internal reference to the Y.ContentEditable instance",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "copyStyles": {
      "!type": "fn(from: +node.Node, to: +node.Node)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_copyStyles",
      "!doc": "Copy certain styles from one node instance to another (used for new paragraph creation mainly)",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "getDomPath": {
      "!type": "fn(node: +node.Node)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_getDomPath",
      "!doc": "Walk the dom tree from this node up to body, returning a reversed array of parents.",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "execCommand": {
      "!type": "fn(cmd: string, val: string) -> +node.Node|+node.NodeList",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_execCommand",
      "!doc": "Pass through to the frame.execCommand method",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "getInstance": {
      "!type": "fn() -> +yui.YUI",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_getInstance",
      "!doc": "Get the YUI instance of the frame",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "render": {
      "!type": "fn(node: +dom.Selector|+HTMLElement|+node.Node) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_render",
      "!doc": "Renders the Y.ContentEditable to the passed node.",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "focus": {
      "!type": "fn(fn: fn()) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_focus",
      "!doc": "Focus the contentWindow of the iframe",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "show": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_show",
      "!doc": "Handles the showing of the Editor instance. Currently only handles the iframe",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "hide": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_hide",
      "!doc": "Handles the hiding of the Editor instance. Currently only handles the iframe",
      "!data": {
       "submodule": "editor-base"
      }
     },
     "getContent": {
      "!type": "fn() -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_getContent",
      "!doc": "(Un)Filters the content of the Editor, cleaning YUI related code. //TODO better filtering",
      "!data": {
       "submodule": "editor-base"
      }
     }
    },
    "NORMALIZE_FONTSIZE": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_NORMALIZE_FONTSIZE",
     "!doc": "Pulls the fontSize from a node, then checks for string values (x-large, x-small)\nand converts them to pixel sizes. If the parsed size is different from the original, it calls\nnode.setStyle to update the node with a pixel size for normalization.",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "TABKEY": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#property_TABKEY",
     "!doc": "The HTML markup to use for the tabkey",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "FILTER_RGB": {
     "!type": "fn(String: ?) -> ?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#method_FILTER_RGB",
     "!doc": "Converts an RGB color string to a hex color, example: rgb(0, 255, 0) converts to #00ff00",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "TAG2CMD": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#property_TAG2CMD",
     "!doc": "A hash table of tags to their execcomands",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "NC_KEYS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#property_NC_KEYS",
     "!doc": "Hash table of keys to fire a nodeChange event for.",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "USE": {
     "!type": "[?]",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#property_USE",
     "!doc": "The default modules to use inside the Frame",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "NAME": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#property_NAME",
     "!doc": "The Class Name: editorBase",
     "!data": {
      "submodule": "editor-base"
     }
    },
    "STRINGS": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorBase.html#property_STRINGS",
     "!doc": "Editor Strings.  By default contains only the `title` property for the\nTitle of frame document (default \"Rich Text Editor\").",
     "!data": {
      "submodule": "editor-base"
     }
    }
   },
   "EditorSelection": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html",
    "removeFontFamily": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_removeFontFamily",
     "!doc": "Utility method to remove dead font-family styles from an element.",
     "!data": {
      "submodule": "selection"
     }
    },
    "filter": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_filter",
     "!doc": "Performs a prefilter on all nodes in the editor. Looks for nodes with a style: fontFamily or font face\nIt then creates a dynamic class assigns it and removed the property. This is so that we dont lose\nthe fontFamily when selecting nodes.",
     "!data": {
      "submodule": "selection"
     }
    },
    "filterBlocks": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_filterBlocks",
     "!doc": "Method attempts to replace all \"orphined\" text nodes in the main body by wrapping them with a <p>. Called from filter.",
     "!data": {
      "submodule": "selection"
     }
    },
    "REG_FONTFAMILY": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_REG_FONTFAMILY",
     "!doc": "Regular Expression used to find dead font-family styles",
     "!data": {
      "submodule": "selection"
     }
    },
    "REG_CHAR": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_REG_CHAR",
     "!doc": "Regular Expression to determine if a string has a character in it",
     "!data": {
      "submodule": "selection"
     }
    },
    "REG_NON": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_REG_NON",
     "!doc": "Regular Expression to determine if a string has a non-character in it",
     "!data": {
      "submodule": "selection"
     }
    },
    "REG_NOHTML": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_REG_NOHTML",
     "!doc": "Regular Expression to remove all HTML from a string",
     "!data": {
      "submodule": "selection"
     }
    },
    "unfilter": {
     "!type": "fn() -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_unfilter",
     "!doc": "Undoes what filter does enough to return the HTML from the Editor, then re-applies the filter.",
     "!data": {
      "submodule": "selection"
     }
    },
    "resolve": {
     "!type": "fn(n: +HTMLElement) -> +node.Node",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_resolve",
     "!doc": "Resolve a node from the selection object and return a Node instance",
     "!data": {
      "submodule": "selection"
     }
    },
    "getText": {
     "!type": "fn(node: +node.Node) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_getText",
     "!doc": "Returns the innerHTML of a node with all HTML tags removed.",
     "!data": {
      "submodule": "selection"
     }
    },
    "ALL": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_ALL",
     "!doc": "The selector to use when looking for Nodes to cache the value of: [style],font[face]",
     "!data": {
      "submodule": "selection"
     }
    },
    "BLOCKS": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_BLOCKS",
     "!doc": "The selector to use when looking for block level items.",
     "!data": {
      "submodule": "selection"
     }
    },
    "TMP": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_TMP",
     "!doc": "The temporary fontname applied to a selection to retrieve their values: yui-tmp",
     "!data": {
      "submodule": "selection"
     }
    },
    "DEFAULT_TAG": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_DEFAULT_TAG",
     "!doc": "The default tag to use when creating elements: span",
     "!data": {
      "submodule": "selection"
     }
    },
    "CURID": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_CURID",
     "!doc": "The id of the outer cursor wrapper",
     "!data": {
      "submodule": "selection"
     }
    },
    "CUR_WRAPID": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_CUR_WRAPID",
     "!doc": "The id used to wrap the inner space of the cursor position",
     "!data": {
      "submodule": "selection"
     }
    },
    "CURSOR": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_CURSOR",
     "!doc": "The default HTML used to focus the cursor..",
     "!data": {
      "submodule": "selection"
     }
    },
    "ROOT": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_ROOT",
     "!doc": "The default HTML element from which data will be retrieved. Default: body",
     "!data": {
      "submodule": "selection"
     }
    },
    "cleanCursor": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_cleanCursor",
     "!doc": "Called from Editor keydown to remove the \"extra\" space before the cursor.",
     "!data": {
      "submodule": "selection"
     }
    },
    "prototype": {
     "text": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_text",
      "!doc": "Range text value",
      "!data": {
       "submodule": "selection"
      }
     },
     "isCollapsed": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_isCollapsed",
      "!doc": "Flag to show if the range is collapsed or not",
      "!data": {
       "submodule": "selection"
      }
     },
     "anchorNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_anchorNode",
      "!doc": "A Node instance of the parentNode of the anchorNode of the range",
      "!data": {
       "submodule": "selection"
      }
     },
     "anchorOffset": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_anchorOffset",
      "!doc": "The offset from the range object",
      "!data": {
       "submodule": "selection"
      }
     },
     "anchorTextNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_anchorTextNode",
      "!doc": "A Node instance of the actual textNode of the range.",
      "!data": {
       "submodule": "selection"
      }
     },
     "focusNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_focusNode",
      "!doc": "A Node instance of the parentNode of the focusNode of the range",
      "!data": {
       "submodule": "selection"
      }
     },
     "focusOffset": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_focusOffset",
      "!doc": "The offset from the range object",
      "!data": {
       "submodule": "selection"
      }
     },
     "focusTextNode": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#property_focusTextNode",
      "!doc": "A Node instance of the actual textNode of the range.",
      "!data": {
       "submodule": "selection"
      }
     },
     "getSelected": {
      "!type": "fn() -> +node.NodeList",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_getSelected",
      "!doc": "Get all the nodes in the current selection. This method will actually perform a filter first.\nThen it calls doc.execCommand(fontname, null, yui-tmp) to touch all nodes in the selection.\nThe it compiles a list of all nodes affected by the execCommand and builds a NodeList to return.",
      "!data": {
       "submodule": "selection"
      }
     },
     "insertContent": {
      "!type": "fn(html: string) -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_insertContent",
      "!doc": "Insert HTML at the current cursor position and return a Node instance of the newly inserted element.",
      "!data": {
       "submodule": "selection"
      }
     },
     "insertAtCursor": {
      "!type": "fn(html: string, node: +node.Node, offset: number, collapse: bool) -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_insertAtCursor",
      "!doc": "Insert HTML at the current cursor position, this method gives you control over the text node to insert into and the offset where to put it.",
      "!data": {
       "submodule": "selection"
      }
     },
     "wrapContent": {
      "!type": "fn(tag: string) -> +node.NodeList",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_wrapContent",
      "!doc": "Get all elements inside a selection and wrap them with a new element and return a NodeList of all elements touched.",
      "!data": {
       "submodule": "selection"
      }
     },
     "replace": {
      "!type": "fn(se: string, re: string) -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_replace",
      "!doc": "Find and replace a string inside a text node and replace it with HTML focusing the node after\nto allow you to continue to type.",
      "!data": {
       "submodule": "selection"
      }
     },
     "remove": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_remove",
      "!doc": "Destroy the range.",
      "!data": {
       "submodule": "selection"
      }
     },
     "createRange": {
      "!type": "fn() -> +Range",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_createRange",
      "!doc": "Wrapper for the different range creation methods.",
      "!data": {
       "submodule": "selection"
      }
     },
     "selectNode": {
      "!type": "fn(node: +node.Node, collapse: bool) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_selectNode",
      "!doc": "Select a Node (hilighting it).",
      "!data": {
       "submodule": "selection"
      }
     },
     "setCursor": {
      "!type": "fn() -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_setCursor",
      "!doc": "Put a placeholder in the DOM at the current cursor position.",
      "!data": {
       "submodule": "selection"
      }
     },
     "getCursor": {
      "!type": "fn() -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_getCursor",
      "!doc": "Get the placeholder in the DOM at the current cursor position.",
      "!data": {
       "submodule": "selection"
      }
     },
     "removeCursor": {
      "!type": "fn(keep: bool) -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_removeCursor",
      "!doc": "Remove the cursor placeholder from the DOM.",
      "!data": {
       "submodule": "selection"
      }
     },
     "focusCursor": {
      "!type": "fn() -> +node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_focusCursor",
      "!doc": "Gets a stored cursor and focuses it for editing, must be called sometime after setCursor",
      "!data": {
       "submodule": "selection"
      }
     },
     "toString": {
      "!type": "fn() -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_toString",
      "!doc": "Generic toString for logging.",
      "!data": {
       "submodule": "selection"
      }
     },
     "getEditorOffset": {
      "!type": "fn(node?: +Y.Node) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EditorSelection.html#method_getEditorOffset",
      "!doc": "Gets the offset of the selection for the selection within the current\neditor",
      "!data": {
       "submodule": "selection"
      }
     }
    }
   },
   "Frame": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "use": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#method_use",
      "!doc": "This is a scoped version of the normal YUI.use method & is bound to this frame/window.\nAt setup, the inst.use method is mapped to this method.",
      "!data": {
       "submodule": "frame"
      }
     },
     "delegate": {
      "!type": "fn(type: string, fn: fn(), cont: string, sel: string) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#method_delegate",
      "!doc": "A delegate method passed to the instances delegate method",
      "!data": {
       "submodule": "frame"
      }
     },
     "getInstance": {
      "!type": "fn() -> +yui.YUI",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#method_getInstance",
      "!doc": "Get a reference to the internal YUI instance.",
      "!data": {
       "submodule": "frame"
      }
     },
     "render": {
      "!type": "fn(node: string|+HTMLElement|+node.Node) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#method_render",
      "!doc": "Render the iframe into the container config option or open the window.",
      "!data": {
       "submodule": "frame"
      }
     },
     "focus": {
      "!type": "fn(fn: fn()) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#method_focus",
      "!doc": "Set the focus to the iframe",
      "!data": {
       "submodule": "frame"
      }
     },
     "show": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#method_show",
      "!doc": "Show the iframe instance",
      "!data": {
       "submodule": "frame"
      }
     },
     "hide": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#method_hide",
      "!doc": "Hide the iframe instance",
      "!data": {
       "submodule": "frame"
      }
     }
    },
    "THROTTLE_TIME": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_THROTTLE_TIME",
     "!doc": "The throttle time for key events in IE",
     "!data": {
      "submodule": "frame"
     }
    },
    "DOM_EVENTS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_DOM_EVENTS",
     "!doc": "The DomEvents that the frame automatically attaches and bubbles",
     "!data": {
      "submodule": "frame"
     }
    },
    "DEFAULT_CSS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_DEFAULT_CSS",
     "!doc": "The default css used when creating the document.",
     "!data": {
      "submodule": "frame"
     }
    },
    "HTML": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_HTML",
     "!doc": "The template string used to create the iframe, deprecated to use DOM instead of innerHTML",
     "!data": {
      "submodule": "frame"
     }
    },
    "IFRAME_ATTRS": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_IFRAME_ATTRS",
     "!doc": "Attributes to auto add to the dynamic iframe under the hood",
     "!data": {
      "submodule": "frame"
     }
    },
    "PAGE_HTML": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_PAGE_HTML",
     "!doc": "The template used to create the page when created dynamically.",
     "!data": {
      "submodule": "frame"
     }
    },
    "getDocType": {
     "!type": "fn() -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#method_getDocType",
     "!doc": "Parses document.doctype and generates a DocType to match the parent page, if supported.\nFor IE8, it grabs document.all[0].nodeValue and uses that. For IE < 8, it falls back to Frame.DOC_TYPE.",
     "!data": {
      "submodule": "frame"
     }
    },
    "DOC_TYPE": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_DOC_TYPE",
     "!doc": "The DOCTYPE to prepend to the new document when created. Should match the one on the page being served.",
     "!data": {
      "submodule": "frame"
     }
    },
    "META": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_META",
     "!doc": "The meta-tag for Content-Type to add to the dynamic document",
     "!data": {
      "submodule": "frame"
     }
    },
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_NAME",
     "!doc": "The name of the class (frame)",
     "!data": {
      "submodule": "frame"
     }
    },
    "NS": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Frame.html#property_NS",
     "!doc": "The namespace on which Frame plugin will reside.",
     "!data": {
      "submodule": "frame"
     }
    }
   }
  },
  "escape": {
   "Escape": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Escape.html",
    "html": {
     "!type": "fn(string: string) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Escape.html#method_html",
     "!doc": "Returns a copy of the specified string with special HTML characters\nescaped. The following characters will be converted to their\ncorresponding character entities:\n\n    & < > \"  / `\n\nThis implementation is based on the [OWASP HTML escaping\nrecommendations][1]. In addition to the characters in the OWASP\nrecommendations, we also escape the <code>&#x60;</code> character, since IE\ninterprets it as an attribute delimiter.\n\nIf _string_ is not already a string, it will be coerced to a string.\n\n[1]: http://www.owasp.org/index.php/XSS_(Cross_Site_Scripting)_Prevention_Cheat_Sheet"
    },
    "regex": {
     "!type": "fn(string: string) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Escape.html#method_regex",
     "!doc": "Returns a copy of the specified string with special regular expression\ncharacters escaped, allowing the string to be used safely inside a regex.\nThe following characters, and all whitespace characters, are escaped:\n\n    -  ^ * ( ) + [ ] { } | \\ , . ?\n\nIf _string_ is not already a string, it will be coerced to a string."
    }
   }
  },
  "event_custom": {
   "!data": {
    "module": "event-custom",
    "submodules": {
     "event-custom-base": {},
     "event-custom-complex": {}
    }
   },
   "CustomEvent": {
    "!type": "fn(type: string, defaults: +Object)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html",
    "prototype": {
     "type": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_type",
      "!doc": "The type of event, returned to subscribers when the event fires",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "silent": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_silent",
      "!doc": "By default all custom events are logged in the debug build, set silent\nto true to disable debug outpu for this event.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "subscribers": {
      "!type": "+Subscriber {}",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_subscribers",
      "!doc": "The subscribers to this event",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "afters": {
      "!type": "+Subscriber {}",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_afters",
      "!doc": "After subscribers",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "monitored": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_monitored",
      "!doc": "Monitor when an event is attached or detached.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "broadcast": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_broadcast",
      "!doc": "If 0, this event does not broadcast.  If 1, the YUI instance is notified\nevery time this event fires.  If 2, the YUI instance and the YUI global\n(if event is enabled on the global) are notified every time this event\nfires.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "queuable": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_queuable",
      "!doc": "Specifies whether this event should be queued when the host is actively\nprocessing an event.  This will effect exectution order of the callbacks\nfor the various events.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "fired": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_fired",
      "!doc": "This event has fired if true",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "firedWith": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_firedWith",
      "!doc": "An array containing the arguments the custom event\nwas last fired with.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "fireOnce": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_fireOnce",
      "!doc": "This event should only fire one time if true, and if\nit has fired, any new subscribers should be notified\nimmediately.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "async": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_async",
      "!doc": "fireOnce listeners will fire syncronously unless async\nis set to true",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "stopped": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_stopped",
      "!doc": "Flag for stopPropagation that is modified during fire()\n1 means to stop propagation to bubble targets.  2 means\nto also stop additional subscribers on this target.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "prevented": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_prevented",
      "!doc": "Flag for preventDefault that is modified during fire().\nif it is not 0, the default behavior for this event",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "host": {
      "!type": "+event_custom.EventTarget",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_host",
      "!doc": "Specifies the host for this custom event.  This is used\nto enable event bubbling",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "defaultFn": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_defaultFn",
      "!doc": "The default function to execute after event listeners\nhave fire, but only if the default action was not\nprevented.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "defaultTargetOnly": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_defaultTargetOnly",
      "!doc": "Flag for the default function to execute only if the\nfiring event is the current target. This happens only\nwhen using custom event delegation and setting the\nflag to `true` mimics the behavior of event delegation\nin the DOM.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "stoppedFn": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_stoppedFn",
      "!doc": "The function to execute if a subscriber calls\nstopPropagation or stopImmediatePropagation",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "preventedFn": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_preventedFn",
      "!doc": "The function to execute if a subscriber calls\npreventDefault",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "emitFacade": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_emitFacade",
      "!doc": "If set to true, the custom event will deliver an EventFacade object\nthat is similar to a DOM event object.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "signature": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_signature",
      "!doc": "Supports multiple options for listener signatures in order to\nport YUI 2 apps.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "context": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_context",
      "!doc": "The context the the event will fire from by default.  Defaults to the YUI\ninstance.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "preventable": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_preventable",
      "!doc": "Specifies whether or not this events default function\ncan be cancelled by a subscriber by executing preventDefault()\non the event facade",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "bubbles": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_bubbles",
      "!doc": "Specifies whether or not a subscriber can stop the event propagation\nvia stopPropagation(), stopImmediatePropagation(), or halt()\n\nEvents can only bubble if emitFacade is true.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "hasSubs": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_hasSubs",
      "!doc": "Returns the number of subscribers for this event as the sum of the on()\nsubscribers and after() subscribers.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "monitor": {
      "!type": "fn(what: string) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_monitor",
      "!doc": "Monitor the event state for the subscribed event.  The first parameter\nis what should be monitored, the rest are the normal parameters when\nsubscribing to an event.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "getSubs": {
      "!type": "fn() -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_getSubs",
      "!doc": "Get all of the subscribers to this event and any sibling event",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "applyConfig": {
      "!type": "fn(o: ?, force: bool)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_applyConfig",
      "!doc": "Apply configuration properties.  Only applies the CONFIG whitelist",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "subscribe": {
      "!type": "fn(fn: fn()) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_subscribe",
      "!doc": "Listen for this event",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "on": {
      "!type": "fn(fn: fn(), context: +Object, arg: +Mixed) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_on",
      "!doc": "Listen for this event",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "after": {
      "!type": "fn(fn: fn(), context: +Object, arg: +Mixed) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_after",
      "!doc": "Listen for this event after the normal subscribers have been notified and\nthe default behavior has been applied.  If a normal subscriber prevents the\ndefault behavior, it also prevents after listeners from firing.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "detach": {
      "!type": "fn(fn: fn(), context: +Object) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_detach",
      "!doc": "Detach listeners.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "unsubscribe": {
      "!type": "fn(fn: fn(), context: +Object) -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_unsubscribe",
      "!doc": "Detach listeners.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "log": {
      "!type": "fn(msg: string, cat: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_log",
      "!doc": "Logger abstraction to centralize the application of the silent flag",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "fire": {
      "!type": "fn(arguments: +Object) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_fire",
      "!doc": "Notifies the subscribers.  The callback functions will be executed\nfrom the context specified when the event was created, and with the\nfollowing parameters:\n  <ul>\n  <li>The type of event</li>\n  <li>All of the arguments fire() was executed with as an array</li>\n  <li>The custom object (if any) that was passed into the subscribe()\n      method</li>\n  </ul>",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "unsubscribeAll": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_unsubscribeAll",
      "!doc": "Removes all listeners",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "detachAll": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_detachAll",
      "!doc": "Removes all listeners",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "stopPropagation": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_stopPropagation",
      "!doc": "Stop propagation to bubble targets",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "stopImmediatePropagation": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_stopImmediatePropagation",
      "!doc": "Stops propagation to bubble targets, and prevents any remaining\nsubscribers on the current target from executing.",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "preventDefault": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_preventDefault",
      "!doc": "Prevents the execution of this events defaultFn",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "halt": {
      "!type": "fn(immediate: bool)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#method_halt",
      "!doc": "Stops the event propagation and prevents the default\nevent behavior.",
      "!data": {
       "submodule": "event-custom-complex"
      }
     }
    },
    "keepDeprecatedSubs": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/CustomEvent.html#property_keepDeprecatedSubs",
     "!doc": "Static flag to enable population of the <a href=\"#property_subscribers\">`subscribers`</a>\nand  <a href=\"#property_subscribers\">`afters`</a> properties held on a `CustomEvent` instance.\n\nThese properties were changed to private properties (`_subscribers` and `_afters`), and\nconverted from objects to arrays for performance reasons.\n\nSetting this property to true will populate the deprecated `subscribers` and `afters`\nproperties for people who may be using them (which is expected to be rare). There will\nbe a performance hit, compared to the new array based implementation.\n\nIf you are using these deprecated properties for a use case which the public API\ndoes not support, please file an enhancement request, and we can provide an alternate\npublic implementation which doesnt have the performance cost required to maintiain the\nproperties as objects.",
     "!data": {
      "submodule": "event-custom-base"
     }
    }
   },
   "Do": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.html",
    "objs": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.html#property_objs",
     "!doc": "Cache of objects touched by the utility",
     "!data": {
      "submodule": "event-custom-base"
     }
    },
    "before": {
     "!type": "fn(fn: fn(), obj: ?, sFn: string, c: ?, arg: +Mixed) -> +event_custom.EventHandle",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.html#method_before",
     "!doc": "<p>Execute the supplied method before the specified function.  Wrapping\nfunction may optionally return an instance of the following classes to\nfurther alter runtime behavior:</p>\n<dl>\n    <dt></code>Y.Do.Halt(message, returnValue)</code></dt>\n        <dd>Immediatly stop execution and return\n        <code>returnValue</code>.  No other wrapping functions will be\n        executed.</dd>\n    <dt></code>Y.Do.AlterArgs(message, newArgArray)</code></dt>\n        <dd>Replace the arguments that the original function will be\n        called with.</dd>\n    <dt></code>Y.Do.Prevent(message)</code></dt>\n        <dd>Dont execute the wrapped function.  Other before phase\n        wrappers will be executed.</dd>\n</dl>",
     "!data": {
      "submodule": "event-custom-base"
     }
    },
    "after": {
     "!type": "fn(fn: fn(), obj: ?, sFn: string, c: ?, arg: +Mixed) -> +event_custom.EventHandle",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.html#method_after",
     "!doc": "<p>Execute the supplied method after the specified function.  Wrapping\nfunction may optionally return an instance of the following classes to\nfurther alter runtime behavior:</p>\n<dl>\n    <dt></code>Y.Do.Halt(message, returnValue)</code></dt>\n        <dd>Immediatly stop execution and return\n        <code>returnValue</code>.  No other wrapping functions will be\n        executed.</dd>\n    <dt></code>Y.Do.AlterReturn(message, returnValue)</code></dt>\n        <dd>Return <code>returnValue</code> instead of the wrapped\n        methods original return value.  This can be further altered by\n        other after phase wrappers.</dd>\n</dl>\n\n<p>The static properties <code>Y.Do.originalRetVal</code> and\n<code>Y.Do.currentRetVal</code> will be populated for reference.</p>",
     "!data": {
      "submodule": "event-custom-base"
     }
    },
    "detach": {
     "!type": "fn(handle: +event_custom.EventHandle)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.html#method_detach",
     "!doc": "Detach a before or after subscription.",
     "!data": {
      "submodule": "event-custom-base"
     }
    },
    "originalRetVal": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.html#property_originalRetVal",
     "!doc": "Contains the return value from the wrapped method, accessible\nby after event listeners.",
     "!data": {
      "submodule": "event-custom-base"
     }
    },
    "currentRetVal": {
     "!type": "?",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.html#property_currentRetVal",
     "!doc": "Contains the current state of the return value, consumable by\nafter event listeners, and updated if an after subscriber\nchanges the return value generated by the wrapped function.",
     "!data": {
      "submodule": "event-custom-base"
     }
    },
    "Method": {
     "!type": "fn(obj: ?, sFn: ?)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.Method.html",
     "prototype": {
      "register": {
       "!type": "fn(sid: string, fn: fn(), when: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.Method.html#method_register",
       "!doc": "Register a aop subscriber",
       "!data": {
        "submodule": "event-custom-base"
       }
      },
      "delete": {
       "!type": "fn(sid: string, fn: fn(), when: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.Method.html#method_delete",
       "!doc": "Unregister a aop subscriber",
       "!data": {
        "submodule": "event-custom-base"
       }
      },
      "exec": {
       "!type": "fn(arg: ?) -> ?",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Do.Method.html#method_exec",
       "!doc": "<p>Execute the wrapped method.  All arguments are passed into the wrapping\nfunctions.  If any of the before wrappers return an instance of\n<code>Y.Do.Halt</code> or <code>Y.Do.Prevent</code>, neither the wrapped\nfunction nor any after phase subscribers will be executed.</p>\n\n<p>The return value will be the return value of the wrapped function or one\nprovided by a wrapper function via an instance of <code>Y.Do.Halt</code> or\n<code>Y.Do.AlterReturn</code>.",
       "!data": {
        "submodule": "event-custom-base"
       }
      }
     }
    }
   },
   "EventFacade": {
    "!type": "fn(e: +event.Event, currentTarget: +HTMLElement)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html",
    "prototype": {
     "details": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#property_details",
      "!doc": "The arguments passed to fire",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "type": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#property_type",
      "!doc": "The event type, this can be overridden by the fire() payload",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "target": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#property_target",
      "!doc": "Node reference for the targeted eventtarget",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "currentTarget": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#property_currentTarget",
      "!doc": "Node reference for the element that the listener was attached to.",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "relatedTarget": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#property_relatedTarget",
      "!doc": "Node reference to the relatedTarget",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "stopPropagation": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#method_stopPropagation",
      "!doc": "Stops the propagation to the next bubble target",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "stopImmediatePropagation": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#method_stopImmediatePropagation",
      "!doc": "Stops the propagation to the next bubble target and\nprevents any additional listeners from being exectued\non the current target.",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "preventDefault": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#method_preventDefault",
      "!doc": "Prevents the events default behavior",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "halt": {
      "!type": "fn(immediate: bool)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventFacade.html#method_halt",
      "!doc": "Stops the event propagation and prevents the default\nevent behavior.",
      "!data": {
       "submodule": "event-custom-complex"
      }
     }
    }
   },
   "EventTarget": {
    "!type": "fn(opts: ?)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html",
    "prototype": {
     "addTarget": {
      "!type": "fn(o: +event_custom.EventTarget) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_addTarget",
      "!doc": "Registers another EventTarget as a bubble target.  Bubble order\nis determined by the order registered.  Multiple targets can\nbe specified.\n\nEvents can only bubble if emitFacade is true.\n\nIncluded in the event-custom-complex submodule.",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "getTargets": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_getTargets",
      "!doc": "Returns an array of bubble targets for this object.",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "removeTarget": {
      "!type": "fn(o: +event_custom.EventTarget) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_removeTarget",
      "!doc": "Removes a bubble target",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "bubble": {
      "!type": "fn(evt: +event_custom.CustomEvent) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_bubble",
      "!doc": "Propagate an event.  Requires the event-custom-complex module.",
      "!data": {
       "submodule": "event-custom-complex"
      }
     },
     "once": {
      "!type": "fn(type: string, fn: fn(), context?: +Object, arg?: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_once",
      "!doc": "Listen to a custom event hosted by this object one time.\nThis is the equivalent to <code>on</code> except the\nlistener is immediatelly detached when it is executed.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "onceAfter": {
      "!type": "fn(type: string, fn: fn(), context?: +Object, arg?: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_onceAfter",
      "!doc": "Listen to a custom event hosted by this object one time.\nThis is the equivalent to <code>after</code> except the\nlistener is immediatelly detached when it is executed.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "parseType": {
      "!type": "fn(type: string, pre?: string) -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_parseType",
      "!doc": "Takes the type parameter passed to on and parses out the\nvarious pieces that could be included in the type.  If the\nevent type is passed without a prefix, it will be expanded\nto include the prefix one is supplied or the event target\nis configured with a default prefix.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "on": {
      "!type": "fn(type: string, fn: fn(), context?: +Object, arg?: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_on",
      "!doc": "Subscribe a callback function to a custom event fired by this object or\nfrom an object that bubbles its events to this object.\n\n     this.on(\"change\", this._onChange, this);\n\nCallback functions for events published with `emitFacade = true` will\nreceive an `EventFacade` as the first argument (typically named \"e\").\nThese callbacks can then call `e.preventDefault()` to disable the\nbehavior published to that events `defaultFn`.  See the `EventFacade`\nAPI for all available properties and methods. Subscribers to\nnon-`emitFacade` events will receive the arguments passed to `fire()`\nafter the event name.\n\nTo subscribe to multiple events at once, pass an object as the first\nargument, where the key:value pairs correspond to the eventName:callback.\n\n     this.on({\n         \"attrChange\" : this._onAttrChange,\n         \"change\"     : this._onChange\n     });\n\nYou can also pass an array of event names as the first argument to\nsubscribe to all listed events with the same callback.\n\n     this.on([ \"change\", \"attrChange\" ], this._onChange);\n\nReturning `false` from a callback is supported as an alternative to\ncalling `e.preventDefault(); e.stopPropagation();`.  However, it is\nrecommended to use the event methods whenever possible.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "subscribe": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_subscribe",
      "!doc": "subscribe to an event",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "detach": {
      "!type": "fn(type: string|+Object, fn: fn(), context: +Object) -> +event_custom.EventTarget",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_detach",
      "!doc": "Detach one or more listeners the from the specified event",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "unsubscribe": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_unsubscribe",
      "!doc": "detach a listener",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "detachAll": {
      "!type": "fn(type: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_detachAll",
      "!doc": "Removes all listeners from the specified event.  If the event type\nis not specified, all listeners from all hosted custom events will\nbe removed.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "unsubscribeAll": {
      "!type": "fn(type: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_unsubscribeAll",
      "!doc": "Removes all listeners from the specified event.  If the event type\nis not specified, all listeners from all hosted custom events will\nbe removed.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "publish": {
      "!type": "fn(type: string, opts: +config.EventTargetPublishConfig) -> +event_custom.CustomEvent",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_publish",
      "!doc": "Creates a new custom event of the specified type.  If a custom event\nby that name already exists, it will not be re-created.  In either\ncase the custom event is returned.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "fire": {
      "!type": "fn(type: string|+Object, arguments: +Object) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_fire",
      "!doc": "Fire a custom event by name.  The callback functions will be executed\nfrom the context specified when the event was created, and with the\nfollowing parameters.\n\nThe first argument is the event type, and any additional arguments are\npassed to the listeners as parameters.  If the first of these is an\nobject literal, and the event is configured to emit an event facade,\nthat object is mixed into the event facade and the facade is provided\nin place of the original object.\n\nIf the custom event object hasnt been created, then the event hasnt\nbeen published and it has no subscribers.  For performance sake, we\nimmediate exit in this case.  This means the event wont bubble, so\nif the intention is that a bubble target be notified, the event must\nbe published on this object first.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "getEvent": {
      "!type": "fn(type: string, prefixed: string) -> +event_custom.CustomEvent",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_getEvent",
      "!doc": "Returns the custom event of the provided type has been created, a\nfalsy value otherwise",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "after": {
      "!type": "fn(type: string, fn: fn(), context?: +Object, arg?: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_after",
      "!doc": "Subscribe to a custom event hosted by this object.  The\nsupplied callback will execute after any listeners add\nvia the subscribe method, and after the default function,\nif configured for the event, has executed.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "before": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventTarget.html#method_before",
      "!doc": "Executes the callback before a DOM event, custom event\nor method.  If the first argument is a function, it\nis assumed the target is a method.  For DOM and custom\nevents, this is an alias for Y.on.\n\nFor DOM and custom events:\ntype, callback, context, 0-n arguments\n\nFor methods:\ncallback, object (method host), methodName, context, 0-n arguments",
      "!data": {
       "submodule": "event-custom-base"
      }
     }
    }
   },
   "YUI": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html",
    "!data": {
     "extends": [
      "event_custom.EventTarget"
     ],
     "for": "yui.YUI"
    },
    "prototype": {
     "Global": {
      "!type": "+event_custom.EventTarget",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#property_Global",
      "!doc": "Hosts YUI page level events.  This is where events bubble to\nwhen the broadcast config is set to 2.  This property is\nonly available if the custom event module is loaded.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "on": {
      "!type": "fn(type: string, fn: fn(), context?: +Object, arg?: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_on",
      "!doc": "`Y.on()` can do many things:\n\n<ul>\n    <li>Subscribe to custom events `publish`ed and `fire`d from Y</li>\n    <li>Subscribe to custom events `publish`ed with `broadcast` 1 or 2 and\n        `fire`d from any object in the YUI instance sandbox</li>\n    <li>Subscribe to DOM events</li>\n    <li>Subscribe to the execution of a method on any object, effectively\n    treating that method as an event</li>\n</ul>\n\nFor custom event subscriptions, pass the custom event name as the first argument\nand callback as the second. The `this` object in the callback will be `Y` unless\nan override is passed as the third argument.\n\n    Y.on(io:complete, function () {\n        Y.MyApp.updateStatus(Transaction complete);\n    });\n\nTo subscribe to DOM events, pass the name of a DOM event as the first argument\nand a CSS selector string as the third argument after the callback function.\nAlternately, the third argument can be a `Node`, `NodeList`, `HTMLElement`,\narray, or simply omitted (the default is the `window` object).\n\n    Y.on(click, function (e) {\n        e.preventDefault();\n\n        // proceed with ajax form submission\n        var url = this.get(action);\n        ...\n    }, #my-form);\n\nThe `this` object in DOM event callbacks will be the `Node` targeted by the CSS\nselector or other identifier.\n\n`on()` subscribers for DOM events or custom events `publish`ed with a\n`defaultFn` can prevent the default behavior with `e.preventDefault()` from the\nevent object passed as the first parameter to the subscription callback.\n\nTo subscribe to the execution of an object method, pass arguments corresponding to the call signature for\n<a href=\"../classes/Do.html#methods_before\">`Y.Do.before(...)`</a>.\n\nNOTE: The formal parameter list below is for events, not for function\ninjection.  See `Y.Do.before` for that signature.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "once": {
      "!type": "fn(type: string, fn: fn(), context?: +Object, arg?: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_once",
      "!doc": "Listen for an event one time.  Equivalent to `on()`, except that\nthe listener is immediately detached when executed.\n\nSee the <a href=\"#methods_on\">`on()` method</a> for additional subscription\noptions.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "onceAfter": {
      "!type": "fn(type: string, fn: fn(), context?: +Object, arg?: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_onceAfter",
      "!doc": "Listen for an event one time.  Equivalent to `once()`, except, like `after()`,\nthe subscription callback executes after all `on()` subscribers and the events\n`defaultFn` (if configured) have executed.  Like `after()` if any `on()` phase\nsubscriber calls `e.preventDefault()`, neither the `defaultFn` nor the `after()`\nsubscribers will execute.\n\nThe listener is immediately detached when executed.\n\nSee the <a href=\"#methods_on\">`on()` method</a> for additional subscription\noptions.",
      "!data": {
       "submodule": "event-custom-base"
      }
     },
     "after": {
      "!type": "fn(type: string, fn: fn(), context?: +Object, args?: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_after",
      "!doc": "Like `on()`, this method creates a subscription to a custom event or to the\nexecution of a method on an object.\n\nFor events, `after()` subscribers are executed after the events\n`defaultFn` unless `e.preventDefault()` was called from an `on()` subscriber.\n\nSee the <a href=\"#methods_on\">`on()` method</a> for additional subscription\noptions.\n\nNOTE: The subscription signature shown is for events, not for function\ninjection.  See <a href=\"../classes/Do.html#methods_after\">`Y.Do.after`</a>\nfor that signature.",
      "!data": {
       "submodule": "event-custom-base"
      }
     }
    }
   },
   "EventHandle": {
    "!type": "fn(evt: +event_custom.CustomEvent, sub: +event_custom.Subscriber)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/EventHandle.html",
    "prototype": {
     "evt": {
      "!type": "+event_custom.CustomEvent",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventHandle.html#property_evt",
      "!doc": "The custom event"
     },
     "sub": {
      "!type": "+event_custom.Subscriber",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventHandle.html#property_sub",
      "!doc": "The subscriber object"
     },
     "detach": {
      "!type": "fn() -> number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventHandle.html#method_detach",
      "!doc": "Detaches this subscriber"
     },
     "monitor": {
      "!type": "fn(what: string) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/EventHandle.html#method_monitor",
      "!doc": "Monitor the event state for the subscribed event.  The first parameter\nis what should be monitored, the rest are the normal parameters when\nsubscribing to an event."
     }
    }
   },
   "Subscriber": {
    "!type": "fn(fn: fn(), context: +Object, args: [?])",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html",
    "prototype": {
     "fn": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html#property_fn",
      "!doc": "The callback that will be execute when the event fires\nThis is wrapped by Y.rbind if obj was supplied."
     },
     "context": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html#property_context",
      "!doc": "Optional this keyword for the listener"
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html#property_id",
      "!doc": "Unique subscriber id"
     },
     "args": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html#property_args",
      "!doc": "Additional arguments to propagate to the subscriber"
     },
     "events": {
      "!type": "+event_custom.EventTarget",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html#property_events",
      "!doc": "Custom events for a given fire transaction."
     },
     "once": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html#property_once",
      "!doc": "This listener only reacts to the event once"
     },
     "notify": {
      "!type": "fn(args: [?], ce: +event_custom.CustomEvent)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html#method_notify",
      "!doc": "Executes the subscriber."
     },
     "contains": {
      "!type": "fn(fn: fn(), context: +Object) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Subscriber.html#method_contains",
      "!doc": "Returns true if the fn and obj match this objects properties.\nUsed by the unsubscribe method to match the right subscriber."
     }
    }
   }
  },
  "event_simulate": {
   "!data": {
    "module": "event-simulate"
   },
   "Event": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html",
    "!data": {
     "for": "event.Event"
    },
    "simulate": {
     "!type": "fn(target: +HTMLElement, type: string, options: +Object)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_simulate",
     "!doc": "Simulates the event or gesture with the given name on a target."
    }
   }
  },
  "event_valuechange": {
   "!data": {
    "module": "event-valuechange"
   },
   "ValueChange": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ValueChange.html",
    "POLL_INTERVAL": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ValueChange.html#property_POLL_INTERVAL",
     "!doc": "Interval (in milliseconds) at which to poll for changes to the value of an\nelement with one or more `valuechange` subscribers when the user is likely\nto be interacting with it."
    },
    "TIMEOUT": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/ValueChange.html#property_TIMEOUT",
     "!doc": "Timeout (in milliseconds) after which to stop polling when there hasnt been\nany new activity (keypresses, mouse clicks, etc.) on an element."
    }
   }
  },
  "event": {
   "!data": {
    "submodules": {
     "event-base": {},
     "event-delegate": {},
     "event-touch": {},
     "event-focus": {},
     "event-hover": {},
     "event-key": {},
     "event-mouseenter": {},
     "event-mousewheel": {},
     "event-outside": {},
     "event-resize": {},
     "event-synthetic": {},
     "event-tap": {}
    }
   },
   "Event": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html",
    "delegate": {
     "!type": "fn(node: +node.Node, subscription: [?], notifier: bool, filter: string)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_delegate",
     "!doc": "Event delegation for the tap event. The delegated event will use a\nsupplied selector or filtering function to test if the event references at least one\nnode that should trigger the subscription callback.\n\nUsage:\n\n    node.delegate(tap, function (e) {\n        Y.log(li a inside node was tapped.);\n    }, li a);",
     "!data": {
      "submodule": "event-tap"
     }
    },
    "compileFilter": {
     "!type": "fn(selector: string) -> fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_compileFilter",
     "!doc": "Compiles a selector string into a filter function to identify whether\nNodes along the parent axis of an events target should trigger event\nnotification.\n\nThis function is memoized, so previously compiled filter functions are\nreturned if the same selector string is provided.\n\nThis function may be useful when defining synthetic events for delegate\nhandling.\n\nHosted as a property of the `delegate` method (e.g. `Y.delegate.compileFilter`).",
     "!data": {
      "submodule": "event-delegate"
     }
    },
    "POLL_RETRYS": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#property_POLL_RETRYS",
     "!doc": "The number of times we should look for elements that are not\nin the DOM at the time the event is requested after the document\nhas been loaded.  The default is 1000@amp;40 ms, so it will poll\nfor 40 seconds or until all outstanding handlers are bound\n(whichever comes first).",
     "!data": {
      "submodule": "event-base"
     }
    },
    "POLL_INTERVAL": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#property_POLL_INTERVAL",
     "!doc": "The poll interval in milliseconds",
     "!data": {
      "submodule": "event-base"
     }
    },
    "lastError": {
     "!type": "+Error",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#property_lastError",
     "!doc": "addListener/removeListener can throw errors in unexpected scenarios.\nThese errors are suppressed, the method returns false, and this property\nis set",
     "!data": {
      "submodule": "event-base"
     }
    },
    "DOMReady": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#property_DOMReady",
     "!doc": "True when the document is initially usable",
     "!data": {
      "submodule": "event-base"
     }
    },
    "onAvailable": {
     "!type": "fn(id: string|[string], fn: fn(), p_obj: +Object, p_override: bool|+Object, checkContent: bool)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_onAvailable",
     "!doc": "Executes the supplied callback when the item with the supplied\nid is found.  This is meant to be used to execute behavior as\nsoon as possible as the page loads.  If you use this after the\ninitial page load it will poll for a fixed time for the element.\nThe number of times it will poll and the frequency are\nconfigurable.  By default it will poll for 10 seconds.\n\n<p>The callback is executed with a single parameter:\nthe custom object parameter, if provided.</p>",
     "!data": {
      "submodule": "event-base"
     }
    },
    "onContentReady": {
     "!type": "fn(id: string, fn: fn(), obj: +Object, override: bool|+Object)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_onContentReady",
     "!doc": "Works the same way as onAvailable, but additionally checks the\nstate of sibling elements to determine if the content of the\navailable element is safe to modify.\n\n<p>The callback is executed with a single parameter:\nthe custom object parameter, if provided.</p>",
     "!data": {
      "submodule": "event-base"
     }
    },
    "attach": {
     "!type": "fn(type: string, fn: fn(), el: string|+HTMLElement|[?]|+node.NodeList, context: +Object, args: bool|+Object) -> +event_custom.EventHandle",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_attach",
     "!doc": "Adds an event listener",
     "!data": {
      "submodule": "event-base"
     }
    },
    "detach": {
     "!type": "fn(node: +node.Node, subscription: [?], notifier: bool)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_detach",
     "!doc": "Detaches all event subscriptions set up by the event-tap module",
     "!data": {
      "submodule": "event-tap"
     }
    },
    "getEvent": {
     "!type": "fn(e: +event.Event, el: +HTMLElement) -> +event.Event",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_getEvent",
     "!doc": "Finds the event in the window object, the callers arguments, or\nin the arguments of another method in the callstack.  This is\nexecuted automatically for events registered through the event\nmanager, so the implementer should not normally need to execute\nthis function at all.",
     "!data": {
      "submodule": "event-base"
     }
    },
    "generateId": {
     "!type": "fn(el: ?) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_generateId",
     "!doc": "Generates an unique ID for the element if it does not already\nhave one.",
     "!data": {
      "submodule": "event-base"
     }
    },
    "purgeElement": {
     "!type": "fn(el: +HTMLElement, recurse: bool, type: string)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_purgeElement",
     "!doc": "Removes all listeners attached to the given element via addListener.\nOptionally, the nodes children can also be purged.\nOptionally, you can specify a specific type of event to remove.",
     "!data": {
      "submodule": "event-base"
     }
    },
    "getListeners": {
     "!type": "fn(el: +HTMLElement|string, type: string) -> +event_custom.CustomEvent",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_getListeners",
     "!doc": "Returns all listeners attached to the given element via addListener.\nOptionally, you can specify a specific type of event to return.",
     "!data": {
      "submodule": "event-base"
     }
    },
    "defineOutside": {
     "!type": "fn(event: string, name: string)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_defineOutside",
     "!doc": "Defines a new outside event to correspond with the given DOM event.\n\nBy default, the created synthetic event name will be the name of the event\nwith \"outside\" appended (e.g. \"click\" becomes \"clickoutside\"). If you want\na different name for the created Event, pass it as a second argument like so:\n<code>Y.Event.defineOutside(eventType, \"yonderclick\")</code>.",
     "!data": {
      "submodule": "event-outside"
     }
    },
    "define": {
     "!type": "fn(type: string, config: +config.EventDefineConfig, force: bool) -> +event.SyntheticEvent",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_define",
     "!doc": "<p>Defines a new event in the DOM event system.  Implementers are\nresponsible for monitoring for a scenario whereby the event is fired.  A\nnotifier object is provided to the functions identified below.  When the\ncriteria defining the event are met, call notifier.fire( [args] ); to\nexecute event subscribers.</p>\n\n<p>The first parameter is the name of the event.  The second parameter is a\nconfiguration object which define the behavior of the event system when the\nnew event is subscribed to or detached from.  The methods that should be\ndefined in this configuration object are <code>on</code>,\n<code>detach</code>, <code>delegate</code>, and <code>detachDelegate</code>.\nYou are free to define any other methods or properties needed to define your\nevent.  Be aware, however, that since the object is used to subclass\nSyntheticEvent, you should avoid method names used by SyntheticEvent unless\nyour intention is to override the default behavior.</p>\n\n<p>This is a list of properties and methods that you can or should specify\nin the configuration object:</p>\n\n<dl>\n  <dt><code>on</code></dt>\n      <dd><code>function (node, subscription, notifier)</code> The\n      implementation logic for subscription.  Any special setup you need to\n      do to create the environment for the event being fired--E.g. native\n      DOM event subscriptions.  Store subscription related objects and\n      state on the <code>subscription</code> object.  When the\n      criteria have been met to fire the synthetic event, call\n      <code>notifier.fire(e)</code>.  See Notifiers <code>fire()</code>\n      method for details about what to pass as parameters.</dd>\n\n  <dt><code>detach</code></dt>\n      <dd><code>function (node, subscription, notifier)</code> The\n      implementation logic for cleaning up a detached subscription. E.g.\n      detach any DOM subscriptions added in <code>on</code>.</dd>\n\n  <dt><code>delegate</code></dt>\n      <dd><code>function (node, subscription, notifier, filter)</code> The\n      implementation logic for subscription via <code>Y.delegate</code> or\n      <code>node.delegate</code>.  The filter is typically either a selector\n      string or a function.  You can use\n      <code>Y.delegate.compileFilter(selectorString)</code> to create a\n      filter function from a selector string if needed.  The filter function\n      expects an event object as input and should output either null, a\n      matching Node, or an array of matching Nodes.  Otherwise, this acts\n      like <code>on</code> DOM event subscriptions.  Store subscription\n      related objects and information on the <code>subscription</code>\n      object.  When the criteria have been met to fire the synthetic event,\n      call <code>notifier.fire(e)</code> as noted above.</dd>\n\n  <dt><code>detachDelegate</code></dt>\n      <dd><code>function (node, subscription, notifier)</code> The\n      implementation logic for cleaning up a detached delegate subscription.\n      E.g. detach any DOM delegate subscriptions added in\n      <code>delegate</code>.</dd>\n\n  <dt><code>publishConfig</code></dt>\n      <dd>(Object) The configuration object that will be used to instantiate\n      the underlying CustomEvent. See Notifiers <code>fire</code> method\n      for details.</dd>\n\n  <dt><code>processArgs</code></dt\n      <dd>\n         <p><code>function (argArray, fromDelegate)</code> Optional method\n         to extract any additional arguments from the subscription\n         signature.  Using this allows <code>on</code> or\n         <code>delegate</code> signatures like\n         <code>node.on(&quot;hover&quot;, overCallback,\n         outCallback)</code>.</p>\n         <p>When processing an atypical argument signature, make sure the\n         args array is returned to the normal signature before returning\n         from the function.  For example, in the &quot;hover&quot; example\n         above, the <code>outCallback</code> needs to be <code>splice</code>d\n         out of the array.  The expected signature of the args array for\n         <code>on()</code> subscriptions is:</p>\n         <pre>\n             <code>[type, callback, target, contextOverride, argN...]</code>\n         </pre>\n         <p>And for <code>delegate()</code>:</p>\n         <pre>\n             <code>[type, callback, target, filter, contextOverride, argN...]</code>\n         </pre>\n         <p>where <code>target</code> is the node the event is being\n         subscribed for.  You can see these signatures documented for\n         <code>Y.on()</code> and <code>Y.delegate()</code> respectively.</p>\n         <p>Whatever gets returned from the function will be stored on the\n         <code>subscription</code> object under\n         <code>subscription._extra</code>.</p></dd>\n  <dt><code>subMatch</code></dt>\n      <dd>\n          <p><code>function (sub, args)</code>  Compares a set of\n          subscription arguments against a Subscription object to determine\n          if they match.  The default implementation compares the callback\n          function against the second argument passed to\n          <code>Y.on(...)</code> or <code>node.detach(...)</code> etc.</p>\n      </dd>\n</dl>",
     "!data": {
      "submodule": "event-synthetic"
     }
    },
    "on": {
     "!type": "fn(node: +node.Node, subscription: [?], notifier: bool)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_on",
     "!doc": "This function should set up the node that will eventually fire the event.\n\nUsage:\n\n    node.on(tap, function (e) {\n        Y.log(the node was tapped on);\n    });",
     "!data": {
      "submodule": "event-tap"
     }
    },
    "detachDelegate": {
     "!type": "fn(node: +node.Node, subscription: [?], notifier: bool)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_detachDelegate",
     "!doc": "Detaches the delegated event subscriptions set up by the event-tap module.\nOnly used if you use node.delegate(...) instead of node.on(...);",
     "!data": {
      "submodule": "event-tap"
     }
    }
   },
   "YUI": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html",
    "!data": {
     "extends": [
      "event_custom.EventTarget"
     ],
     "for": "yui.YUI"
    },
    "prototype": {
     "delegate": {
      "!type": "fn(type: string, fn: fn(), el: string|+node, filter: string, context: ?, args: ?) -> +event_custom.EventHandle",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_delegate",
      "!doc": "Sets up event delegation on a container element.  The delegated event\nwill use a supplied filter to test if the callback should be executed.\nThis filter can be either a selector string or a function that returns\na Node to use as the currentTarget for the event.\n\nThe event object for the delegated event is supplied to the callback\nfunction.  It is modified slightly in order to support all properties\nthat may be needed for event delegation.  currentTarget is set to\nthe element that matched the selector string filter or the Node returned\nfrom the filter function.  container is set to the element that the\nlistener is delegated from (this normally would be the currentTarget).\n\nFilter functions will be called with the arguments that would be passed to\nthe callback function, including the event object as the first parameter.\nThe function should return false (or a falsey value) if the success criteria\narent met, and the Node to use as the events currentTarget and this\nobject if they are.",
      "!data": {
       "submodule": "event-delegate"
      }
     }
    }
   },
   "DOMEventFacade": {
    "!type": "fn(ev: +event.Event, currentTarget: +HTMLElement, wrapper: +event_custom.CustomEvent)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html",
    "prototype": {
     "touches": {
      "!type": "[+event.DOMEventFacade]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_touches",
      "!doc": "Array of individual touch events for touch points that are still in\ncontact with the touch surface.",
      "!data": {
       "submodule": "event-touch"
      }
     },
     "targetTouches": {
      "!type": "[+event.DOMEventFacade]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_targetTouches",
      "!doc": "Array of individual touch events still in contact with the touch\nsurface and whose `touchstart` event occurred inside the same taregt\nelement as the current target element.",
      "!data": {
       "submodule": "event-touch"
      }
     },
     "changedTouches": {
      "!type": "[+event.DOMEventFacade]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_changedTouches",
      "!doc": "An array of event-specific touch events.\n\nFor `touchstart`, the touch points that became active with the current\nevent.\n\nFor `touchmove`, the touch points that have changed since the last\nevent.\n\nFor `touchend`, the touch points that have been removed from the touch\nsurface.",
      "!data": {
       "submodule": "event-touch"
      }
     },
     "type": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_type",
      "!doc": "The name of the event (e.g. \"click\")",
      "!data": {
       "submodule": "event-base"
      }
     },
     "altKey": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_altKey",
      "!doc": "`true` if the \"alt\" or \"option\" key is pressed.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "shiftKey": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_shiftKey",
      "!doc": "`true` if the shift key is pressed.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "metaKey": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_metaKey",
      "!doc": "`true` if the \"Windows\" key on a Windows keyboard, \"command\" key on an\nApple keyboard, or \"meta\" key on other keyboards is pressed.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "ctrlKey": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_ctrlKey",
      "!doc": "`true` if the \"Ctrl\" or \"control\" key is pressed.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "pageX": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_pageX",
      "!doc": "The X location of the event on the page (including scroll)",
      "!data": {
       "submodule": "event-base"
      }
     },
     "pageY": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_pageY",
      "!doc": "The Y location of the event on the page (including scroll)",
      "!data": {
       "submodule": "event-base"
      }
     },
     "clientX": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_clientX",
      "!doc": "The X location of the event in the viewport",
      "!data": {
       "submodule": "event-base"
      }
     },
     "clientY": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_clientY",
      "!doc": "The Y location of the event in the viewport",
      "!data": {
       "submodule": "event-base"
      }
     },
     "keyCode": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_keyCode",
      "!doc": "The keyCode for key events.  Uses charCode if keyCode is not available",
      "!data": {
       "submodule": "event-base"
      }
     },
     "charCode": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_charCode",
      "!doc": "The charCode for key events.  Same as keyCode",
      "!data": {
       "submodule": "event-base"
      }
     },
     "button": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_button",
      "!doc": "The button that was pushed. 1 for left click, 2 for middle click, 3 for\nright click.  This is only reliably populated on `mouseup` events.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "which": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_which",
      "!doc": "The button that was pushed.  Same as button.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "target": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_target",
      "!doc": "Node reference for the targeted element",
      "!data": {
       "submodule": "event-base"
      }
     },
     "currentTarget": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_currentTarget",
      "!doc": "Node reference for the element that the listener was attached to.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "relatedTarget": {
      "!type": "+node.Node",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_relatedTarget",
      "!doc": "Node reference to the relatedTarget",
      "!data": {
       "submodule": "event-base"
      }
     },
     "wheelDelta": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property_wheelDelta",
      "!doc": "Number representing the direction and velocity of the movement of the mousewheel.\nNegative is down, the higher the number, the faster.  Applies to the mousewheel event.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "stopPropagation": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#method_stopPropagation",
      "!doc": "Stops the propagation to the next bubble target",
      "!data": {
       "submodule": "event-base"
      }
     },
     "stopImmediatePropagation": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#method_stopImmediatePropagation",
      "!doc": "Stops the propagation to the next bubble target and\nprevents any additional listeners from being exectued\non the current target.",
      "!data": {
       "submodule": "event-base"
      }
     },
     "preventDefault": {
      "!type": "fn(returnValue: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#method_preventDefault",
      "!doc": "Prevents the events default behavior",
      "!data": {
       "submodule": "event-base"
      }
     },
     "halt": {
      "!type": "fn(immediate: bool)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#method_halt",
      "!doc": "Stops the event propagation and prevents the default\nevent behavior.",
      "!data": {
       "submodule": "event-base"
      }
     }
    },
    "_GESTURE_MAP": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/DOMEventFacade.html#property__GESTURE_MAP",
     "!doc": "A object literal with keys \"start\", \"end\", and \"move\". The value for each key is a\nstring representing the event for that environment. For touch environments, the respective\nvalues are \"touchstart\", \"touchend\" and \"touchmove\". Mouse and MSPointer environments are also\nsupported via feature detection.",
     "!data": {
      "submodule": "event-touch"
     }
    }
   },
   "SynthRegistry": {
    "!type": "fn(el: +HTMLElement, yuid: string, key: string)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SynthRegistry.html",
    "prototype": {
     "register": {
      "!type": "fn(handle: +event_custom.EventHandle)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SynthRegistry.html#method_register",
      "!doc": "Adds a subscription from the Notifier registry.",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "_unregisterSub": {
      "!type": "fn(sub: +Subscription)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SynthRegistry.html#method__unregisterSub",
      "!doc": "Removes the subscription from the Notifier registry.",
      "!data": {
       "submodule": "event-synthetic"
      }
     }
    }
   },
   "SyntheticEvent": {
    "!type": "fn(cfg: +Object)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html",
    "prototype": {
     "processArgs": {
      "!type": "fn(args: [?], delegate: bool) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#method_processArgs",
      "!doc": "<p>Implementers MAY provide this method definition.</p>\n\n<p>Implement this function if the event supports a different\nsubscription signature.  This function is used by both\n<code>on()</code> and <code>delegate()</code>.  The second parameter\nindicates that the event is being subscribed via\n<code>delegate()</code>.</p>\n\n<p>Implementations must remove extra arguments from the args list\nbefore returning.  The required args for <code>on()</code>\nsubscriptions are</p>\n<pre><code>[type, callback, target, context, argN...]</code></pre>\n\n<p>The required args for <code>delegate()</code>\nsubscriptions are</p>\n\n<pre><code>[type, callback, target, filter, context, argN...]</code></pre>\n\n<p>The return value from this function will be stored on the\nsubscription in the _extra property for reference elsewhere.</p>",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "preventDups": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#property_preventDups",
      "!doc": "<p>Implementers MAY override this property.</p>\n\n<p>Whether to prevent multiple subscriptions to this event that are\nclassified as being the same.  By default, this means the subscribed\ncallback is the same function.  See the <code>subMatch</code>\nmethod.  Setting this to true will impact performance for high volume\nevents.</p>",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "on": {
      "!type": "fn(node: +node.Node, sub: +Subscription, notifier: +event.SyntheticEvent.Notifier)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#method_on",
      "!doc": "<p>Implementers SHOULD provide this method definition.</p>\n\nImplementation logic for subscriptions done via <code>node.on(type,\nfn)</code> or <code>Y.on(type, fn, target)</code>.  This\nfunction should set up the monitor(s) that will eventually fire the\nevent.  Typically this involves subscribing to at least one DOM\nevent.  It is recommended to store detach handles from any DOM\nsubscriptions to make for easy cleanup in the <code>detach</code>\nmethod.  Typically these handles are added to the <code>sub</code>\nobject.  Also for SyntheticEvents that leverage a single DOM\nsubscription under the hood, it is recommended to pass the DOM event\nobject to <code>notifier.fire(e)</code>.  (The event name on the\nobject will be updated).",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "detach": {
      "!type": "fn(node: +node.Node, sub: +Subscription, notifier: +event.SyntheticEvent.Notifier)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#method_detach",
      "!doc": "<p>Implementers SHOULD provide this method definition.</p>\n\n<p>Implementation logic for detaching subscriptions done via\n<code>node.on(type, fn)</code>.  This function should clean up any\nsubscriptions made in the <code>on()</code> phase.</p>",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "delegate": {
      "!type": "fn(node: +node.Node, sub: +Subscription, notifier: +event.SyntheticEvent.Notifier, filter: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#method_delegate",
      "!doc": "<p>Implementers SHOULD provide this method definition.</p>\n\n<p>Implementation logic for subscriptions done via\n<code>node.delegate(type, fn, filter)</code> or\n<code>Y.delegate(type, fn, container, filter)</code>.  Like with\n<code>on()</code> above, this function should monitor the environment\nfor the event being fired, and trigger subscription execution by\ncalling <code>notifier.fire(e)</code>.</p>\n\n<p>This function receives a fourth argument, which is the filter\nused to identify which Nodes are of interest to the subscription.\nThe filter will be either a boolean function that accepts a target\nNode for each hierarchy level as the event bubbles, or a selector\nstring.  To translate selector strings into filter functions, use\n<code>Y.delegate.compileFilter(filter)</code>.</p>",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "detachDelegate": {
      "!type": "fn(node: +node.Node, sub: +Subscription, notifier: +event.SyntheticEvent.Notifier, filter: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#method_detachDelegate",
      "!doc": "<p>Implementers SHOULD provide this method definition.</p>\n\n<p>Implementation logic for detaching subscriptions done via\n<code>node.delegate(type, fn, filter)</code> or\n<code>Y.delegate(type, fn, container, filter)</code>.  This function\nshould clean up any subscriptions made in the\n<code>delegate()</code> phase.</p>",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "applyArgExtras": {
      "!type": "fn(extra: ?, sub: +Subscription)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#method_applyArgExtras",
      "!doc": "<p>Implementers MAY provide this method definition.</p>\n\n<p>Implement this function if you want extra data extracted during\nprocessArgs to be propagated to subscriptions on a per-node basis.\nThat is to say, if you call <code>Y.on(xyz, fn, xtra, div)</code>\nthe data returned from processArgs will be shared\nacross the subscription objects for all the divs.  If you want each\nsubscription to receive unique information, do that processing\nhere.</p>\n\n<p>The default implementation adds the data extracted by processArgs\nto the subscription object as <code>sub._extra</code>.</p>",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "getSubs": {
      "!type": "fn(node: +node.Node, args: [?], filter: fn(), first: bool) -> [+event_custom.EventHandle]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#method_getSubs",
      "!doc": "Returns the detach handles of subscriptions on a node that satisfy a\nsearch/filter function.  By default, the filter used is the\n<code>subMatch</code> method.",
      "!data": {
       "submodule": "event-synthetic"
      }
     },
     "subMatch": {
      "!type": "fn(sub: +Subscription, args: [?]) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SyntheticEvent.html#method_subMatch",
      "!doc": "<p>Implementers MAY override this to define what constitutes a\n&quot;same&quot; subscription.  Override implementations should\nconsider the lack of a comparator as a match, so calling\n<code>getSubs()</code> with no arguments will return all subs.</p>\n\n<p>Compares a set of subscription arguments against a Subscription\nobject to determine if they match.  The default implementation\ncompares the callback function against the second argument passed to\n<code>Y.on(...)</code> or <code>node.detach(...)</code> etc.</p>",
      "!data": {
       "submodule": "event-synthetic"
      }
     }
    }
   }
  },
  "file_flash": {
   "!data": {
    "module": "file-flash"
   },
   "FileFlash": {
    "!type": "fn(config: +config.FileFlashConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "startUpload": {
      "!type": "fn(url: string, parameters: +Object, fileFieldName: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#method_startUpload",
      "!doc": "Starts the upload of a specific file."
     },
     "cancelUpload": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/FileFlash.html#method_cancelUpload",
      "!doc": "Cancels the upload of a specific file, if currently in progress."
     }
    }
   }
  },
  "file_html5": {
   "!data": {
    "module": "file-html5"
   },
   "FileHTML5": {
    "!type": "fn(config: +config.FileHTML5Config)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "startUpload": {
      "!type": "fn(url: string, parameters: +Object, fileFieldName: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#method_startUpload",
      "!doc": "Starts the upload of a specific file."
     },
     "cancelUpload": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#method_cancelUpload",
      "!doc": "Cancels the upload of a specific file, if currently in progress."
     }
    },
    "isValidFile": {
     "!type": "fn(file: +file.File)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#method_isValidFile",
     "!doc": "Checks whether a specific native file instance is valid"
    },
    "canUpload": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/FileHTML5.html#method_canUpload",
     "!doc": "Checks whether the browser has a native upload capability\nvia XMLHttpRequest Level 2."
    }
   }
  },
  "gesture_simulate": {
   "!data": {
    "module": "gesture-simulate"
   },
   "Event": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html",
    "!data": {
     "for": "event.Event"
    },
    "prototype": {
     "rotate": {
      "!type": "fn(cb: fn(), center: [?], startRadius: number, endRadius: number, duration: number, start: number, rotation: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_rotate",
      "!doc": "The \"rotate\" and \"pinch\" methods are essencially same with the exact same\narguments. Only difference is the required parameters. The rotate method\nrequires \"rotation\" parameter while the pinch method requires \"startRadius\"\nand \"endRadius\" parameters."
     },
     "pinch": {
      "!type": "fn(cb: fn(), center: [?], startRadius: number, endRadius: number, duration: number, start: number, rotation: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_pinch",
      "!doc": "The \"rotate\" and \"pinch\" methods are essencially same with the exact same\narguments. Only difference is the required parameters. The rotate method\nrequires \"rotation\" parameter while the pinch method requires \"startRadius\"\nand \"endRadius\" parameters.\n\nThe \"pinch\" gesture can simulate various 2 finger gestures such as pinch,\nspread and/or rotation. The \"startRadius\" and \"endRadius\" are required.\nIf endRadius is larger than startRadius, it becomes a spread gesture\notherwise a pinch gesture."
     },
     "tap": {
      "!type": "fn(cb: fn(), point: [?], times: number, hold: number, delay: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_tap",
      "!doc": "The \"tap\" gesture can be used for various single touch point gestures\nsuch as single tap, N number of taps, long press. The default is a single\ntap."
     },
     "flick": {
      "!type": "fn(cb: fn(), point: [?], axis: string, distance: number, duration: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_flick",
      "!doc": "The \"flick\" gesture is a specialized \"move\" that has some velocity\nand the movement always runs either x or y axis. The velocity is calculated\nwith \"distance\" and \"duration\" arguments. If the calculated velocity is\nbelow than the minimum velocity, the given duration will be ignored and\nnew duration will be created to make a valid flick gesture."
     },
     "move": {
      "!type": "fn(cb: fn(), path: +Object, duration: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_move",
      "!doc": "The \"move\" gesture simulate the movement of any direction between\nthe straight line of start and end point for the given duration.\nThe path argument is an object with \"point\", \"xdist\" and \"ydist\" properties.\nThe \"point\" property is an array with x and y coordinations(relative to the\ntop left corner of the target node element) while \"xdist\" and \"ydist\"\nproperties are used for the distance along the x and y axis. A negative\ndistance number can be used to drag either left or up direction.\n\nIf no arguments are given, it will simulate the default move, which\nis moving 200 pixels from the center of the element to the positive X-axis\ndirection for 1 sec."
     }
    },
    "simulateGesture": {
     "!type": "fn(node: +HTMLElement|+node.Node, name: string, options?: +config.EventSimulateGestureConfig, cb?: fn(err: +Error))",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Event.html#method_simulateGesture",
     "!doc": "Simulates the higher user level gesture of the given name on a target.\nThis method generates a set of low level touch events(Apple specific gesture\nevents as well for the iOS platforms) asynchronously. Note that gesture\nsimulation is relying on `Y.Event.simulate()` method to generate\nthe touch events under the hood. The `Y.Event.simulate()` method\nitself is a synchronous method.\n\nUsers are suggested to use `Node.simulateGesture()` method which\nbasically calls this method internally. Supported gestures are `tap`,\n`doubletap`, `press`, `move`, `flick`, `pinch` and `rotate`.\n\nThe `pinch` gesture is used to simulate the pinching and spreading of two\nfingers. During a pinch simulation, rotation is also possible. Essentially\n`pinch` and `rotate` simulations share the same base implementation to allow\nboth pinching and rotation at the same time. The only difference is `pinch`\nrequires `start` and `end` option properties while `rotate` requires `rotation`\noption property.\n\nThe `pinch` and `rotate` gestures can be described as placing 2 fingers along a\ncircle. Pinching and spreading can be described by start and end circles while\nrotation occurs on a single circle. If the radius of the start circle is greater\nthan the end circle, the gesture becomes a pinch, otherwise it is a spread spread."
    }
   }
  },
  "get_nodejs": {
   "!data": {
    "module": "get-nodejs"
   },
   "GetNodeJS": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/GetNodeJS.html",
    "prototype": {
     "js": {
      "!type": "fn(s: [?]|string, options: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/GetNodeJS.html#method_js",
      "!doc": "Override for Get.script for loading local or remote YUI modules."
     },
     "script": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/GetNodeJS.html#method_script",
      "!doc": "Alias for `Y.Get.js`"
     }
    }
   }
  },
  "get": {
   "Get": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html",
    "cssOptions": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html#property_cssOptions",
     "!doc": "Default options for CSS requests. Options specified here will override\nglobal defaults for CSS requests.\n\nSee the `options` property for all available options."
    },
    "jsOptions": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html#property_jsOptions",
     "!doc": "Default options for JS requests. Options specified here will override global\ndefaults for JS requests.\n\nSee the `options` property for all available options."
    },
    "options": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html#property_options",
     "!doc": "Default options to use for all requests.\n\nNote that while all available options are documented here for ease of\ndiscovery, some options (like callback functions) only make sense at the\ntransaction level.\n\nCallback functions specified via the options object or the `options`\nparameter of the `css()`, `js()`, or `load()` methods will receive the\ntransaction object as a parameter. See `Y.Get.Transaction` for details on\nthe properties and methods available on transactions."
    },
    "abort": {
     "!type": "fn(transaction: +get.Get.Transaction)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html#method_abort",
     "!doc": "Aborts the specified transaction.\n\nThis will cause the transactions `onFailure` callback to be called and\nwill prevent any new script and link nodes from being added to the document,\nbut any resources that have already been requested will continue loading\n(theres no safe way to prevent this, unfortunately).\n\n*Note:* This method is deprecated as of 3.5.0, and will be removed in a\nfuture version of YUI. Use the transaction-level `abort()` method instead."
    },
    "css": {
     "!type": "fn(urls: string|+Object|[?], options?: +Object, callback?: fn(err: [?], transaction: +get.Get.Transaction)) -> +get.Get.Transaction",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html#method_css",
     "!doc": "Loads one or more CSS files.\n\nThe _urls_ parameter may be provided as a URL string, a request object,\nor an array of URL strings and/or request objects.\n\nA request object is just an object that contains a `url` property and zero\nor more options that should apply specifically to that request.\nRequest-specific options take priority over transaction-level options and\ndefault options.\n\nURLs may be relative or absolute, and do not have to have the same origin\nas the current page.\n\nThe `options` parameter may be omitted completely and a callback passed in\nits place, if desired."
    },
    "js": {
     "!type": "fn(urls: string|+Object|[?], options?: +Object, callback?: fn(err: [?], transaction: +get.Get.Transaction)) -> +get.Get.Transaction",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html#method_js",
     "!doc": "Loads one or more JavaScript resources.\n\nThe _urls_ parameter may be provided as a URL string, a request object,\nor an array of URL strings and/or request objects.\n\nA request object is just an object that contains a `url` property and zero\nor more options that should apply specifically to that request.\nRequest-specific options take priority over transaction-level options and\ndefault options.\n\nURLs may be relative or absolute, and do not have to have the same origin\nas the current page.\n\nThe `options` parameter may be omitted completely and a callback passed in\nits place, if desired.\n\nScripts will be executed in the order theyre specified unless the `async`\noption is `true`, in which case theyll be loaded in parallel and executed\nin whatever order they finish loading."
    },
    "load": {
     "!type": "fn(urls: string|+Object|[?], options?: +Object, callback?: fn(), err: [?], Transaction: +get.Get.Transaction) -> +get.Get.Transaction",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html#method_load",
     "!doc": "Loads one or more CSS and/or JavaScript resources in the same transaction.\n\nUse this method when you want to load both CSS and JavaScript in a single\ntransaction and be notified when all requested URLs have finished loading,\nregardless of type.\n\nBehavior and options are the same as for the `css()` and `js()` methods. If\na resource type isnt specified in per-request options or transaction-level\noptions, Get will guess the file type based on the URLs extension (`.css`\nor `.js`, with or without a following query string). If the file type cant\nbe guessed from the URL, a warning will be logged and Get will assume the\nURL is a JavaScript resource."
    },
    "script": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.html#method_script",
     "!doc": "Alias for `js()`."
    },
    "Transaction": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html",
     "prototype": {
      "data": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#property_data",
       "!doc": "Arbitrary data object associated with this transaction.\n\nThis object comes from the options passed to `Get.css()`, `Get.js()`, or\n`Get.load()`, and will be `undefined` if no data object was specified."
      },
      "errors": {
       "!type": "[+Object]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#property_errors",
       "!doc": "Array of errors that have occurred during this transaction, if any. Each error\nobject has the following properties:\n`errors.error`: Error message.\n`errors.request`: Request object related to the error."
      },
      "id": {
       "!type": "number",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#property_id",
       "!doc": "Numeric id for this transaction, unique among all transactions within the same\nYUI sandbox in the current pageview."
      },
      "nodes": {
       "!type": "[+HTMLElement]",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#property_nodes",
       "!doc": "HTMLElement nodes (native ones, not YUI Node instances) that have been inserted\nduring the current transaction."
      },
      "options": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#property_options",
       "!doc": "Options associated with this transaction.\n\nSee `Get.options` for the full list of available options."
      },
      "requests": {
       "!type": "+Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#property_requests",
       "!doc": "Request objects contained in this transaction. Each request object represents\none CSS or JS URL that will be (or has been) requested and loaded into the page."
      },
      "abort": {
       "!type": "fn(msg?: string)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#method_abort",
       "!doc": "Aborts this transaction.\n\nThis will cause the transactions `onFailure` callback to be called and\nwill prevent any new script and link nodes from being added to the document,\nbut any resources that have already been requested will continue loading\n(theres no safe way to prevent this, unfortunately)."
      },
      "execute": {
       "!type": "fn(callback: fn())",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#method_execute",
       "!doc": "Begins execting the transaction.\n\nTheres usually no reason to call this manually, since Get will call it\nautomatically when other pending transactions have finished. If you really\nwant to execute your transaction before Get does, you can, but be aware that\nthis transactions scripts may end up executing before the scripts in other\npending transactions.\n\nIf the transaction is already executing, the specified callback (if any)\nwill be queued and called after execution finishes. If the transaction has\nalready finished, the callback will be called immediately (the transaction\nwill not be executed again)."
      },
      "purge": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Get.Transaction.html#method_purge",
       "!doc": "Manually purges any `<script>` or `<link>` nodes this transaction has\ncreated.\n\nBe careful when purging a transaction that contains CSS requests, since\nremoving `<link>` nodes will also remove any styles they applied."
      }
     }
    }
   }
  },
  "graphics": {
   "!data": {
    "submodules": {
     "graphics-group": {}
    }
   },
   "CanvasCircle": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasCircle.html",
    "prototype": {
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasCircle.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasCircle.html#config_height",
      "!doc": "Indicates the height of the shape"
     },
     "radius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasCircle.html#config_radius",
      "!doc": "Radius of the circle"
     }
    }
   },
   "CanvasDrawing": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html",
    "prototype": {
     "lineTo": {
      "!type": "fn(point1: number, point2: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_lineTo",
      "!doc": "Draws a line segment from the current drawing position to the specified x and y coordinates."
     },
     "relativeLineTo": {
      "!type": "fn(point1: number, point2: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_relativeLineTo",
      "!doc": "Draws a line segment from the current drawing position to the relative x and y coordinates."
     },
     "moveTo": {
      "!type": "fn(x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_moveTo",
      "!doc": "Moves the current drawing position to specified x and y coordinates."
     },
     "relativeMoveTo": {
      "!type": "fn(x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_relativeMoveTo",
      "!doc": "Moves the current drawing position relative to specified x and y coordinates."
     },
     "curveTo": {
      "!type": "fn(cp1x: number, cp1y: number, cp2x: number, cp2y: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_curveTo",
      "!doc": "Draws a bezier curve."
     },
     "relativeCurveTo": {
      "!type": "fn(cp1x: number, cp1y: number, cp2x: number, cp2y: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_relativeCurveTo",
      "!doc": "Draws a bezier curve relative to the current coordinates."
     },
     "quadraticCurveTo": {
      "!type": "fn(cpx: number, cpy: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_quadraticCurveTo",
      "!doc": "Draws a quadratic bezier curve."
     },
     "relativeQuadraticCurveTo": {
      "!type": "fn(cpx: number, cpy: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_relativeQuadraticCurveTo",
      "!doc": "Draws a quadratic bezier curve relative to the current position."
     },
     "drawRect": {
      "!type": "fn(x: number, y: number, w: number, h: number, ew: number, eh: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_drawRect",
      "!doc": "Draws a rectangle with rounded corners."
     },
     "end": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_end",
      "!doc": "Completes a drawing operation."
     },
     "closePath": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_closePath",
      "!doc": "Ends a fill and stroke"
     },
     "clear": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasDrawing.html#method_clear",
      "!doc": "Clears the graphics object."
     }
    }
   },
   "CanvasEllipse": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasEllipse.html",
    "prototype": {
     "xRadius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasEllipse.html#config_xRadius",
      "!doc": "Horizontal radius for the ellipse."
     },
     "yRadius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasEllipse.html#config_yRadius",
      "!doc": "Vertical radius for the ellipse."
     }
    }
   },
   "CanvasGraphic": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html",
    "prototype": {
     "render": {
      "!type": "fn(parentNode: +HTMLElement)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_render",
      "!doc": "Adds the graphics node to the dom."
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_id",
      "!doc": "Unique id for class instance."
     },
     "shapes": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_shapes",
      "!doc": "Key value pairs in which a shape instance is associated with its id."
     },
     "contentBounds": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_contentBounds",
      "!doc": "Object containing size and coordinate data for the content of a Graphic in relation to the graphic instances position."
     },
     "node": {
      "!type": "+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_node",
      "!doc": "The outermost html element of the Graphic instance."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_width",
      "!doc": "Indicates the width of the `Graphic`."
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_height",
      "!doc": "Indicates the height of the `Graphic`."
     },
     "autoSize": {
      "!type": "bool|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_autoSize",
      "!doc": "Determines the sizing of the Graphic.\n\n<dl>\n    <dt>sizeContentToGraphic</dt><dd>The Graphics width and height attributes are, either explicitly set through the\n    <code>width</code> and <code>height</code> attributes or are determined by the dimensions of the parent element. The\n    content contained in the Graphic will be sized to fit with in the Graphic instances dimensions. When using this\n    setting, the <code>preserveAspectRatio</code> attribute will determine how the contents are sized.</dd>\n    <dt>sizeGraphicToContent</dt><dd>(Also accepts a value of true) The Graphics width and height are determined by the\n    size and positioning of the content.</dd>\n    <dt>false</dt><dd>The Graphics width and height attributes are, either explicitly set through the <code>width</code>\n    and <code>height</code> attributes or are determined by the dimensions of the parent element. The contents of the\n    Graphic instance are not affected by this setting.</dd>\n</dl>"
     },
     "preserveAspectRatio": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_preserveAspectRatio",
      "!doc": "Determines how content is sized when <code>autoSize</code> is set to <code>sizeContentToGraphic</code>.\n\n <dl>\n     <dt>none<dt><dd>Do not force uniform scaling. Scale the graphic content of the given element non-uniformly if necessary\n     such that the elements bounding box exactly matches the viewport rectangle.</dd>\n     <dt>xMinYMin</dt><dd>Force uniform scaling position along the top left of the Graphics node.</dd>\n     <dt>xMidYMin</dt><dd>Force uniform scaling horizontally centered and positioned at the top of the Graphics node.<dd>\n     <dt>xMaxYMin</dt><dd>Force uniform scaling positioned horizontally from the right and vertically from the top.</dd>\n     <dt>xMinYMid</dt>Force uniform scaling positioned horizontally from the left and vertically centered.</dd>\n     <dt>xMidYMid (the default)</dt><dd>Force uniform scaling with the content centered.</dd>\n     <dt>xMaxYMid</dt><dd>Force uniform scaling positioned horizontally from the right and vertically centered.</dd>\n     <dt>xMinYMax</dt><dd>Force uniform scaling positioned horizontally from the left and vertically from the bottom.</dd>\n     <dt>xMidYMax</dt><dd>Force uniform scaling horizontally centered and position vertically from the bottom.</dd>\n     <dt>xMaxYMax</dt><dd>Force uniform scaling positioned horizontally from the right and vertically from the bottom.</dd>\n </dl>"
     },
     "resizeDown": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_resizeDown",
      "!doc": "The contentBounds will resize to greater values but not smaller values. (for performance)\nWhen resizing the contentBounds down is desirable, set the resizeDown value to true."
     },
     "x": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_x",
      "!doc": "Indicates the x-coordinate for the instance."
     },
     "y": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_y",
      "!doc": "Indicates the y-coordinate for the instance."
     },
     "visible": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#config_visible",
      "!doc": "Indicates whether the `Graphic` and its children are visible."
     },
     "set": {
      "!type": "fn(name: string|+Object, value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_set",
      "!doc": "Sets the value of an attribute."
     },
     "getXY": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_getXY",
      "!doc": "Gets the current position of the graphic instance in page coordinates."
     },
     "destroy": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_destroy",
      "!doc": "Removes all nodes."
     },
     "addShape": {
      "!type": "fn(cfg: +Object) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_addShape",
      "!doc": "Generates a shape instance by type."
     },
     "removeShape": {
      "!type": "fn(shape: +graphics.Shape|string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_removeShape",
      "!doc": "Removes a shape instance from from the graphic instance."
     },
     "removeAllShapes": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_removeAllShapes",
      "!doc": "Removes all shape instances from the dom."
     },
     "clear": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_clear",
      "!doc": "Clears the graphics object."
     },
     "getShapeById": {
      "!type": "fn(id: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_getShapeById",
      "!doc": "Returns a shape based on the id of its dom node."
     },
     "batch": {
      "!type": "fn(method: fn())",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasGraphic.html#method_batch",
      "!doc": "Allows for creating multiple shapes in order to batch appending and redraw operations."
     }
    }
   },
   "CanvasPath": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPath.html",
    "prototype": {
     "end": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPath.html#method_end",
      "!doc": "Completes a drawing operation."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPath.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPath.html#config_height",
      "!doc": "Indicates the height of the shape"
     },
     "path": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPath.html#config_path",
      "!doc": "Indicates the path used for the node."
     }
    }
   },
   "CanvasPieSlice": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPieSlice.html",
    "prototype": {
     "startAngle": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPieSlice.html#config_startAngle",
      "!doc": "Starting angle in relation to a circle in which to begin the pie slice drawing."
     },
     "arc": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPieSlice.html#config_arc",
      "!doc": "Arc of the slice."
     },
     "radius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasPieSlice.html#config_radius",
      "!doc": "Radius of the circle in which the pie slice is drawn"
     }
    }
   },
   "CanvasShape": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html",
    "prototype": {
     "addClass": {
      "!type": "fn(className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_addClass",
      "!doc": "Add a class name to each node."
     },
     "removeClass": {
      "!type": "fn(className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_removeClass",
      "!doc": "Removes a class name from each node."
     },
     "getXY": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_getXY",
      "!doc": "Gets the current position of the node in page coordinates."
     },
     "setXY": {
      "!type": "fn(Contains: [?])",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_setXY",
      "!doc": "Set the position of the shape in page coordinates, regardless of how the node is positioned."
     },
     "contains": {
      "!type": "fn(needle: +graphics.CanvasShape|+HTMLElement) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_contains",
      "!doc": "Determines whether the node is an ancestor of another HTML element in the DOM hierarchy."
     },
     "test": {
      "!type": "fn(selector: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_test",
      "!doc": "Test if the supplied node matches the supplied selector."
     },
     "compareTo": {
      "!type": "fn(refNode: +HTMLElement|+node.Node) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_compareTo",
      "!doc": "Compares nodes to determine if they match.\nNode instances can be compared to each other and/or HTMLElements."
     },
     "set": {
      "!type": "fn(name: string|+Object, value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_set",
      "!doc": "Sets the value of an attribute."
     },
     "translate": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_translate",
      "!doc": "Specifies a 2d translation."
     },
     "translateX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_translateX",
      "!doc": "Translates the shape along the x-axis. When translating x and y coordinates,\nuse the `translate` method."
     },
     "translateY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_translateY",
      "!doc": "Performs a translate on the y-coordinate. When translating x and y coordinates,\nuse the `translate` method."
     },
     "skew": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_skew",
      "!doc": "Skews the shape around the x-axis and y-axis."
     },
     "skewX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_skewX",
      "!doc": "Skews the shape around the x-axis."
     },
     "skewY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_skewY",
      "!doc": "Skews the shape around the y-axis."
     },
     "rotate": {
      "!type": "fn(deg: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_rotate",
      "!doc": "Rotates the shape clockwise around it transformOrigin."
     },
     "scale": {
      "!type": "fn(val: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_scale",
      "!doc": "Specifies a 2d scaling operation."
     },
     "getBounds": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_getBounds",
      "!doc": "Returns the bounds for a shape.\n\nCalculates the a new bounding box from the original corner coordinates (base on size and position) and the transform matrix.\nThe calculated bounding box is used by the graphic instance to calculate its viewBox."
     },
     "toFront": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_toFront",
      "!doc": "Places the shape underneath all other shapes."
     },
     "destroy": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#method_destroy",
      "!doc": "Destroys the shape instance."
     },
     "transformOrigin": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_transformOrigin",
      "!doc": "An array of x, y values which indicates the transformOrigin in which to rotate the shape. Valid values range between 0 and 1 representing a\nfraction of the shapes corresponding bounding box dimension. The default value is [0.5, 0.5]."
     },
     "transform": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_transform",
      "!doc": "<p>A string containing, in order, transform operations applied to the shape instance. The `transform` string can contain the following values:\n\n   <dl>\n       <dt>rotate</dt><dd>Rotates the shape clockwise around it transformOrigin.</dd>\n       <dt>translate</dt><dd>Specifies a 2d translation.</dd>\n       <dt>skew</dt><dd>Skews the shape around the x-axis and y-axis.</dd>\n       <dt>scale</dt><dd>Specifies a 2d scaling operation.</dd>\n       <dt>translateX</dt><dd>Translates the shape along the x-axis.</dd>\n       <dt>translateY</dt><dd>Translates the shape along the y-axis.</dd>\n       <dt>skewX</dt><dd>Skews the shape around the x-axis.</dd>\n       <dt>skewY</dt><dd>Skews the shape around the y-axis.</dd>\n       <dt>matrix</dt><dd>Specifies a 2D transformation matrix comprised of the specified six values.</dd>\n   </dl>\n</p>\n<p>Applying transforms through the transform attribute will reset the transform matrix and apply a new transform. The shape class also contains\ncorresponding methods for each transform that will apply the transform to the current matrix. The below code illustrates how you might use the\n`transform` attribute to instantiate a recangle with a rotation of 45 degrees.</p>\n           var myRect = new Y.Rect({\n               type:\"rect\",\n               width: 50,\n               height: 40,\n               transform: \"rotate(45)\"\n           };\n<p>The code below would apply `translate` and `rotate` to an existing shape.</p>\n\n       myRect.set(\"transform\", \"translate(40, 50) rotate(45)\");"
     },
     "node": {
      "!type": "+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_node",
      "!doc": "Dom node for the shape"
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_id",
      "!doc": "Unique id for class instance."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_height",
      "!doc": "Indicates the height of the shape"
     },
     "x": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_x",
      "!doc": "Indicates the x position of shape."
     },
     "y": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_y",
      "!doc": "Indicates the y position of shape."
     },
     "visible": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_visible",
      "!doc": "Indicates whether the shape is visible."
     },
     "fill": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_fill",
      "!doc": "Contains information about the fill of the shape.\n <dl>\n     <dt>color</dt><dd>The color of the fill.</dd>\n     <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the fill. The default value is 1.</dd>\n     <dt>type</dt><dd>Type of fill.\n         <dl>\n             <dt>solid</dt><dd>Solid single color fill. (default)</dd>\n             <dt>linear</dt><dd>Linear gradient fill.</dd>\n             <dt>radial</dt><dd>Radial gradient fill.</dd>\n         </dl>\n     </dd>\n </dl>\n <p>If a `linear` or `radial` is specified as the fill type. The following additional property is used:\n <dl>\n     <dt>stops</dt><dd>An array of objects containing the following properties:\n         <dl>\n             <dt>color</dt><dd>The color of the stop.</dd>\n             <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stop. The default value is 1.\n             Note: No effect for IE 6 - 8</dd>\n             <dt>offset</dt><dd>Number between 0 and 1 indicating where the color stop is positioned.</dd>\n         </dl>\n     </dd>\n     <p>Linear gradients also have the following property:</p>\n     <dt>rotation</dt><dd>Linear gradients flow left to right by default. The rotation property allows you to change the\n     flow by rotation. (e.g. A rotation of 180 would make the gradient pain from right to left.)</dd>\n     <p>Radial gradients have the following additional properties:</p>\n     <dt>r</dt><dd>Radius of the gradient circle.</dd>\n     <dt>fx</dt><dd>Focal point x-coordinate of the gradient.</dd>\n     <dt>fy</dt><dd>Focal point y-coordinate of the gradient.</dd>\n </dl>\n <p>The corresponding `SVGShape` class implements the following additional properties.</p>\n <dl>\n     <dt>cx</dt><dd>\n         <p>The x-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>\n         <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and\n         `VMLShape` classes which are used on Android or IE 6 - 8.</p>\n     </dd>\n     <dt>cy</dt><dd>\n         <p>The y-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>\n         <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and `VMLShape`\n         classes which are used on Android or IE 6 - 8.</p>\n     </dd>\n </dl>\n <p>These properties are not currently implemented in `CanvasShape` or `VMLShape`.</p>"
     },
     "stroke": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_stroke",
      "!doc": "Contains information about the stroke of the shape.\n <dl>\n     <dt>color</dt><dd>The color of the stroke.</dd>\n     <dt>weight</dt><dd>Number that indicates the width of the stroke.</dd>\n     <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stroke. The default value is 1.</dd>\n     <dt>dashstyle</dt>Indicates whether to draw a dashed stroke. When set to \"none\", a solid stroke is drawn. When set\n     to an array, the first index indicates the length of the dash. The second index indicates the length of gap.\n     <dt>linecap</dt><dd>Specifies the linecap for the stroke. The following values can be specified:\n         <dl>\n             <dt>butt (default)</dt><dd>Specifies a butt linecap.</dd>\n             <dt>square</dt><dd>Specifies a sqare linecap.</dd>\n             <dt>round</dt><dd>Specifies a round linecap.</dd>\n         </dl>\n     </dd>\n     <dt>linejoin</dt><dd>Specifies a linejoin for the stroke. The following values can be specified:\n         <dl>\n             <dt>round (default)</dt><dd>Specifies that the linejoin will be round.</dd>\n             <dt>bevel</dt><dd>Specifies a bevel for the linejoin.</dd>\n             <dt>miter limit</dt><dd>An integer specifying the miter limit of a miter linejoin. If you want to specify a linejoin\n             of miter, you simply specify the limit as opposed to having separate miter and miter limit values.</dd>\n         </dl>\n     </dd>\n </dl>"
     },
     "data": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_data",
      "!doc": "Represents an SVG Path string. This will be parsed and added to shapes API to represent the SVG data across all\nimplementations. Note that when using VML or SVG implementations, part of this content will be added to the DOM using\nrespective VML/SVG attributes. If your content comes from an untrusted source, you will need to ensure that no\nmalicious code is included in that content."
     },
     "graphic": {
      "!type": "+graphics.Graphic",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CanvasShape.html#config_graphic",
      "!doc": "Reference to the container Graphic."
     }
    }
   },
   "CircleGroup": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/CircleGroup.html",
    "prototype": {
     "drawShape": {
      "!type": "fn(cfg: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/CircleGroup.html#method_drawShape",
      "!doc": "Algorithm for drawing shape.",
      "!data": {
       "submodule": "graphics-group"
      }
     }
    }
   },
   "AttributeLite": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeLite.html",
    "prototype": {
     "addAttrs": {
      "!type": "fn(cfg: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeLite.html#method_addAttrs",
      "!doc": "Initializes the attributes for a shape. If an attribute config is passed into the constructor of the host,\nthe initial values will be overwritten."
     },
     "get": {
      "!type": "fn(name: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeLite.html#method_get",
      "!doc": "For a given item, returns the value of the property requested, or undefined if not found."
     },
     "set": {
      "!type": "fn(name: string|+Object, value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/AttributeLite.html#method_set",
      "!doc": "Sets the value of an attribute."
     }
    }
   },
   "Drawing": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html",
    "prototype": {
     "curveTo": {
      "!type": "fn(cp1x: number, cp1y: number, cp2x: number, cp2y: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_curveTo",
      "!doc": "Draws a bezier curve."
     },
     "quadraticCurveTo": {
      "!type": "fn(cpx: number, cpy: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_quadraticCurveTo",
      "!doc": "Draws a quadratic bezier curve."
     },
     "drawRect": {
      "!type": "fn(x: number, y: number, w: number, h: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_drawRect",
      "!doc": "Draws a rectangle."
     },
     "drawRoundRect": {
      "!type": "fn(x: number, y: number, w: number, h: number, ew: number, eh: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_drawRoundRect",
      "!doc": "Draws a rectangle with rounded corners."
     },
     "lineTo": {
      "!type": "fn(point1: number, point2: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_lineTo",
      "!doc": "Draws a line segment using the current line style from the current drawing position to the specified x and y coordinates."
     },
     "relativeLineTo": {
      "!type": "fn(point1: number, point2: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_relativeLineTo",
      "!doc": "Draws a line segment using the current line style from the current drawing position to the relative x and y coordinates."
     },
     "moveTo": {
      "!type": "fn(x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_moveTo",
      "!doc": "Moves the current drawing position to specified x and y coordinates."
     },
     "relativeMoveTo": {
      "!type": "fn(x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_relativeMoveTo",
      "!doc": "Moves the current drawing position relative to specified x and y coordinates."
     },
     "end": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_end",
      "!doc": "Completes a drawing operation."
     },
     "clear": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_clear",
      "!doc": "Clears the path."
     },
     "closePath": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Drawing.html#method_closePath",
      "!doc": "Ends a fill and stroke"
     }
    }
   },
   "Shape": {
    "!type": "fn(cfg: +Object)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html",
    "prototype": {
     "addClass": {
      "!type": "fn(className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_addClass",
      "!doc": "Add a class name to each node."
     },
     "removeClass": {
      "!type": "fn(className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_removeClass",
      "!doc": "Removes a class name from each node."
     },
     "getXY": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_getXY",
      "!doc": "Gets the current position of the node in page coordinates."
     },
     "setXY": {
      "!type": "fn(Contains: [?])",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_setXY",
      "!doc": "Set the position of the shape in page coordinates, regardless of how the node is positioned."
     },
     "contains": {
      "!type": "fn(needle: +graphics.Shape|+HTMLElement) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_contains",
      "!doc": "Determines whether the node is an ancestor of another HTML element in the DOM hierarchy."
     },
     "compareTo": {
      "!type": "fn(refNode: +HTMLElement|+node.Node) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_compareTo",
      "!doc": "Compares nodes to determine if they match.\nNode instances can be compared to each other and/or HTMLElements."
     },
     "test": {
      "!type": "fn(selector: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_test",
      "!doc": "Test if the supplied node matches the supplied selector."
     },
     "set": {
      "!type": "fn(name: string|+Object, value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_set",
      "!doc": "Sets the value of an attribute."
     },
     "translate": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_translate",
      "!doc": "Specifies a 2d translation."
     },
     "translateX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_translateX",
      "!doc": "Translates the shape along the x-axis. When translating x and y coordinates,\nuse the `translate` method."
     },
     "translateY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_translateY",
      "!doc": "Translates the shape along the y-axis. When translating x and y coordinates,\nuse the `translate` method."
     },
     "skew": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_skew",
      "!doc": "Skews the shape around the x-axis and y-axis."
     },
     "skewX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_skewX",
      "!doc": "Skews the shape around the x-axis."
     },
     "skewY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_skewY",
      "!doc": "Skews the shape around the y-axis."
     },
     "rotate": {
      "!type": "fn(deg: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_rotate",
      "!doc": "Rotates the shape clockwise around it transformOrigin."
     },
     "scale": {
      "!type": "fn(val: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_scale",
      "!doc": "Specifies a 2d scaling operation."
     },
     "getBounds": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_getBounds",
      "!doc": "Returns the bounds for a shape.\n\nCalculates the a new bounding box from the original corner coordinates (base on size and position) and the transform matrix.\nThe calculated bounding box is used by the graphic instance to calculate its viewBox."
     },
     "destroy": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#method_destroy",
      "!doc": "Destroys the instance."
     },
     "transformOrigin": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_transformOrigin",
      "!doc": "An array of x, y values which indicates the transformOrigin in which to rotate the shape. Valid values range between 0 and 1 representing a\nfraction of the shapes corresponding bounding box dimension. The default value is [0.5, 0.5]."
     },
     "transform": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_transform",
      "!doc": "<p>A string containing, in order, transform operations applied to the shape instance. The `transform` string can contain the following values:\n\n   <dl>\n       <dt>rotate</dt><dd>Rotates the shape clockwise around it transformOrigin.</dd>\n       <dt>translate</dt><dd>Specifies a 2d translation.</dd>\n       <dt>skew</dt><dd>Skews the shape around the x-axis and y-axis.</dd>\n       <dt>scale</dt><dd>Specifies a 2d scaling operation.</dd>\n       <dt>translateX</dt><dd>Translates the shape along the x-axis.</dd>\n       <dt>translateY</dt><dd>Translates the shape along the y-axis.</dd>\n       <dt>skewX</dt><dd>Skews the shape around the x-axis.</dd>\n       <dt>skewY</dt><dd>Skews the shape around the y-axis.</dd>\n       <dt>matrix</dt><dd>Specifies a 2D transformation matrix comprised of the specified six values.</dd>\n   </dl>\n</p>\n<p>Applying transforms through the transform attribute will reset the transform matrix and apply a new transform. The shape class also contains\ncorresponding methods for each transform that will apply the transform to the current matrix. The below code illustrates how you might use the\n`transform` attribute to instantiate a recangle with a rotation of 45 degrees.</p>\n           var myRect = new Y.Rect({\n               type:\"rect\",\n               width: 50,\n               height: 40,\n               transform: \"rotate(45)\"\n           };\n<p>The code below would apply `translate` and `rotate` to an existing shape.</p>\n\n       myRect.set(\"transform\", \"translate(40, 50) rotate(45)\");"
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_id",
      "!doc": "Unique id for class instance."
     },
     "x": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_x",
      "!doc": "Indicates the x position of shape."
     },
     "y": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_y",
      "!doc": "Indicates the y position of shape."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_height",
      "!doc": "Indicates the height of the shape"
     },
     "visible": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_visible",
      "!doc": "Indicates whether the shape is visible."
     },
     "fill": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_fill",
      "!doc": "Contains information about the fill of the shape.\n <dl>\n     <dt>color</dt><dd>The color of the fill.</dd>\n     <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the fill. The default value is 1.</dd>\n     <dt>type</dt><dd>Type of fill.\n         <dl>\n             <dt>solid</dt><dd>Solid single color fill. (default)</dd>\n             <dt>linear</dt><dd>Linear gradient fill.</dd>\n             <dt>radial</dt><dd>Radial gradient fill.</dd>\n         </dl>\n     </dd>\n </dl>\n <p>If a `linear` or `radial` is specified as the fill type. The following additional property is used:\n <dl>\n     <dt>stops</dt><dd>An array of objects containing the following properties:\n         <dl>\n             <dt>color</dt><dd>The color of the stop.</dd>\n             <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stop. The default value is 1.\n             Note: No effect for IE 6 - 8</dd>\n             <dt>offset</dt><dd>Number between 0 and 1 indicating where the color stop is positioned.</dd>\n         </dl>\n     </dd>\n     <p>Linear gradients also have the following property:</p>\n     <dt>rotation</dt><dd>Linear gradients flow left to right by default. The rotation property allows you to change the\n     flow by rotation. (e.g. A rotation of 180 would make the gradient pain from right to left.)</dd>\n     <p>Radial gradients have the following additional properties:</p>\n     <dt>r</dt><dd>Radius of the gradient circle.</dd>\n     <dt>fx</dt><dd>Focal point x-coordinate of the gradient.</dd>\n     <dt>fy</dt><dd>Focal point y-coordinate of the gradient.</dd>\n     <dt>cx</dt><dd>\n         <p>The x-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>\n         <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and\n         `VMLShape` classes which are used on Android or IE 6 - 8.</p>\n     </dd>\n     <dt>cy</dt><dd>\n         <p>The y-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>\n         <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and `VMLShape`\n         classes which are used on Android or IE 6 - 8.</p>\n     </dd>\n </dl>"
     },
     "stroke": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_stroke",
      "!doc": "Contains information about the stroke of the shape.\n <dl>\n     <dt>color</dt><dd>The color of the stroke.</dd>\n     <dt>weight</dt><dd>Number that indicates the width of the stroke.</dd>\n     <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stroke. The default value is 1.</dd>\n     <dt>dashstyle</dt>Indicates whether to draw a dashed stroke. When set to \"none\", a solid stroke is drawn. When set\n     to an array, the first index indicates the length of the dash. The second index indicates the length of gap.\n     <dt>linecap</dt><dd>Specifies the linecap for the stroke. The following values can be specified:\n         <dl>\n             <dt>butt (default)</dt><dd>Specifies a butt linecap.</dd>\n             <dt>square</dt><dd>Specifies a sqare linecap.</dd>\n             <dt>round</dt><dd>Specifies a round linecap.</dd>\n         </dl>\n     </dd>\n     <dt>linejoin</dt><dd>Specifies a linejoin for the stroke. The following values can be specified:\n         <dl>\n             <dt>round (default)</dt><dd>Specifies that the linejoin will be round.</dd>\n             <dt>bevel</dt><dd>Specifies a bevel for the linejoin.</dd>\n             <dt>miter limit</dt><dd>An integer specifying the miter limit of a miter linejoin. If you want to specify a linejoin\n             of miter, you simply specify the limit as opposed to having separate miter and miter limit values.</dd>\n         </dl>\n     </dd>\n </dl>"
     },
     "node": {
      "!type": "+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_node",
      "!doc": "Dom node for the shape."
     },
     "data": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_data",
      "!doc": "Represents an SVG Path string. This will be parsed and added to shapes API to represent the SVG data across all\nimplementations. Note that when using VML or SVG implementations, part of this content will be added to the DOM using\nrespective VML/SVG attributes. If your content comes from an untrusted source, you will need to ensure that no\nmalicious code is included in that content."
     },
     "graphic": {
      "!type": "+graphics.Graphic",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Shape.html#config_graphic",
      "!doc": "Reference to the parent graphic instance"
     }
    }
   },
   "Circle": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Circle.html",
    "prototype": {
     "!proto": "graphics.Shape.prototype",
     "radius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Circle.html#config_radius",
      "!doc": "Radius of the circle"
     }
    }
   },
   "Path": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Path.html",
    "!data": {
     "extends": [
      "graphics.Drawing"
     ]
    },
    "prototype": {
     "!proto": "graphics.Shape.prototype",
     "path": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Path.html#config_path",
      "!doc": "Indicates the path used for the node."
     }
    }
   },
   "Graphic": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html",
    "prototype": {
     "render": {
      "!type": "fn(parentNode: +node.Node|string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#method_render",
      "!doc": "Adds the graphics node to the dom."
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_id",
      "!doc": "Unique id for class instance."
     },
     "shapes": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_shapes",
      "!doc": "Key value pairs in which a shape instance is associated with its id."
     },
     "contentBounds": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_contentBounds",
      "!doc": "Object containing size and coordinate data for the content of a Graphic in relation to the coordSpace node."
     },
     "node": {
      "!type": "+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_node",
      "!doc": "The html element that represents to coordinate system of the Graphic instance."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_width",
      "!doc": "Indicates the width of the `Graphic`."
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_height",
      "!doc": "Indicates the height of the `Graphic`."
     },
     "autoSize": {
      "!type": "bool|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_autoSize",
      "!doc": "Determines the sizing of the Graphic.\n\n<dl>\n    <dt>sizeContentToGraphic</dt><dd>The Graphics width and height attributes are, either explicitly set through the\n    <code>width</code> and <code>height</code> attributes or are determined by the dimensions of the parent element. The\n    content contained in the Graphic will be sized to fit with in the Graphic instances dimensions. When using this\n    setting, the <code>preserveAspectRatio</code> attribute will determine how the contents are sized.</dd>\n    <dt>sizeGraphicToContent</dt><dd>(Also accepts a value of true) The Graphics width and height are determined by the\n    size and positioning of the content.</dd>\n    <dt>false</dt><dd>The Graphics width and height attributes are, either explicitly set through the <code>width</code>\n    and <code>height</code> attributes or are determined by the dimensions of the parent element. The contents of the\n    Graphic instance are not affected by this setting.</dd>\n</dl>"
     },
     "preserveAspectRatio": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_preserveAspectRatio",
      "!doc": "Determines how content is sized when <code>autoSize</code> is set to <code>sizeContentToGraphic</code>.\n\n <dl>\n     <dt>none<dt><dd>Do not force uniform scaling. Scale the graphic content of the given element non-uniformly if necessary\n     such that the elements bounding box exactly matches the viewport rectangle.</dd>\n     <dt>xMinYMin</dt><dd>Force uniform scaling position along the top left of the Graphics node.</dd>\n     <dt>xMidYMin</dt><dd>Force uniform scaling horizontally centered and positioned at the top of the Graphics node.<dd>\n     <dt>xMaxYMin</dt><dd>Force uniform scaling positioned horizontally from the right and vertically from the top.</dd>\n     <dt>xMinYMid</dt>Force uniform scaling positioned horizontally from the left and vertically centered.</dd>\n     <dt>xMidYMid (the default)</dt><dd>Force uniform scaling with the content centered.</dd>\n     <dt>xMaxYMid</dt><dd>Force uniform scaling positioned horizontally from the right and vertically centered.</dd>\n     <dt>xMinYMax</dt><dd>Force uniform scaling positioned horizontally from the left and vertically from the bottom.</dd>\n     <dt>xMidYMax</dt><dd>Force uniform scaling horizontally centered and position vertically from the bottom.</dd>\n     <dt>xMaxYMax</dt><dd>Force uniform scaling positioned horizontally from the right and vertically from the bottom.</dd>\n </dl>"
     },
     "resizeDown": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_resizeDown",
      "!doc": "The contentBounds will resize to greater values but not to smaller values. (for performance)\nWhen resizing the contentBounds down is desirable, set the resizeDown value to true."
     },
     "x": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_x",
      "!doc": "Indicates the x-coordinate for the instance."
     },
     "y": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_y",
      "!doc": "Indicates the y-coordinate for the instance."
     },
     "visible": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#config_visible",
      "!doc": "Indicates whether the `Graphic` and its children are visible."
     },
     "getXY": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#method_getXY",
      "!doc": "Gets the current position of the graphic instance in page coordinates."
     },
     "destroy": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#method_destroy",
      "!doc": "Removes all nodes."
     },
     "addShape": {
      "!type": "fn(cfg: +Object) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#method_addShape",
      "!doc": "<p>Generates a shape instance by type. The method accepts an object that contains the shapes\ntype and attributes to be customized. For example, the code below would create a rectangle:</p>\n\n           var myRect = myGraphic.addShape({\n               type: \"rect\",\n               width: 40,\n               height: 30,\n               fill: {\n                   color: \"#9aa\"\n               },\n               stroke: {\n                   weight: 1,\n                   color: \"#000\"\n               }\n           });\n\n<p>The `Graphics` module includes a few basic shapes. More information on their creation\ncan be found in each shapes documentation:\n\n <ul>\n     <li><a href=\"Circle.html\">`Circle`</a></li>\n     <li><a href=\"Ellipse.html\">`Ellipse`</a></li>\n     <li><a href=\"Rect.html\">`Rect`</a></li>\n     <li><a href=\"Path.html\">`Path`</a></li>\n </ul>\n\n The `Graphics` module also allows for the creation of custom shapes. If a custom shape\n has been created, it can be instantiated with the `addShape` method as well. The attributes,\n required and optional, would need to be defined in the custom shape.\n\n           var myCustomShape = myGraphic.addShape({\n               type: Y.MyCustomShape,\n               width: 50,\n               height: 50,\n               fill: {\n                   color: \"#9aa\"\n               },\n               stroke: {\n                   weight: 1,\n                   color: \"#000\"\n               }\n           });"
     },
     "removeShape": {
      "!type": "fn(shape: +graphics.Shape|string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#method_removeShape",
      "!doc": "Removes a shape instance from from the graphic instance."
     },
     "removeAllShapes": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#method_removeAllShapes",
      "!doc": "Removes all shape instances from the dom."
     },
     "getShapeById": {
      "!type": "fn(id: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#method_getShapeById",
      "!doc": "Returns a shape based on the id of its dom node."
     },
     "batch": {
      "!type": "fn(method: fn())",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Graphic.html#method_batch",
      "!doc": "Allows for creating multiple shapes in order to batch appending and redraw operations."
     }
    }
   },
   "SVGCircle": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGCircle.html",
    "prototype": {
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGCircle.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGCircle.html#config_height",
      "!doc": "Indicates the height of the shape"
     },
     "radius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGCircle.html#config_radius",
      "!doc": "Radius of the circle"
     }
    }
   },
   "SVGDrawing": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html",
    "prototype": {
     "curveTo": {
      "!type": "fn(cp1x: number, cp1y: number, cp2x: number, cp2y: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_curveTo",
      "!doc": "Draws a bezier curve."
     },
     "relativeCurveTo": {
      "!type": "fn(cp1x: number, cp1y: number, cp2x: number, cp2y: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_relativeCurveTo",
      "!doc": "Draws a bezier curve relative to the current coordinates."
     },
     "quadraticCurveTo": {
      "!type": "fn(cpx: number, cpy: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_quadraticCurveTo",
      "!doc": "Draws a quadratic bezier curve relative to the current position."
     },
     "drawRect": {
      "!type": "fn(x: number, y: number, w: number, h: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_drawRect",
      "!doc": "Draws a rectangle."
     },
     "drawRoundRect": {
      "!type": "fn(x: number, y: number, w: number, h: number, ew: number, eh: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_drawRoundRect",
      "!doc": "Draws a rectangle with rounded corners."
     },
     "lineTo": {
      "!type": "fn(point1: number, point2: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_lineTo",
      "!doc": "Draws a line segment using the current line style from the current drawing position to the specified x and y coordinates."
     },
     "relativeLineTo": {
      "!type": "fn(point1: number, point2: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_relativeLineTo",
      "!doc": "Draws a line segment using the current line style from the current drawing position to the relative x and y coordinates."
     },
     "moveTo": {
      "!type": "fn(x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_moveTo",
      "!doc": "Moves the current drawing position to specified x and y coordinates."
     },
     "relativeMoveTo": {
      "!type": "fn(x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_relativeMoveTo",
      "!doc": "Moves the current drawing position relative to specified x and y coordinates."
     },
     "end": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_end",
      "!doc": "Completes a drawing operation."
     },
     "clear": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_clear",
      "!doc": "Clears the path."
     },
     "closePath": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGDrawing.html#method_closePath",
      "!doc": "Ends a fill and stroke"
     }
    }
   },
   "SVGEllipse": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGEllipse.html",
    "prototype": {
     "xRadius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGEllipse.html#config_xRadius",
      "!doc": "Horizontal radius for the ellipse."
     },
     "yRadius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGEllipse.html#config_yRadius",
      "!doc": "Vertical radius for the ellipse."
     }
    }
   },
   "SVGGraphic": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html",
    "prototype": {
     "render": {
      "!type": "fn(parentNode: +HTMLElement)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_render",
      "!doc": "Adds the graphics node to the dom."
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_id",
      "!doc": "Unique id for class instance."
     },
     "shapes": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_shapes",
      "!doc": "Key value pairs in which a shape instance is associated with its id."
     },
     "contentBounds": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_contentBounds",
      "!doc": "Object containing size and coordinate data for the content of a Graphic in relation to the coordSpace node."
     },
     "node": {
      "!type": "+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_node",
      "!doc": "The html element that represents to coordinate system of the Graphic instance."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_width",
      "!doc": "Indicates the width of the `Graphic`."
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_height",
      "!doc": "Indicates the height of the `Graphic`."
     },
     "autoSize": {
      "!type": "bool|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_autoSize",
      "!doc": "Determines the sizing of the Graphic.\n\n<dl>\n    <dt>sizeContentToGraphic</dt><dd>The Graphics width and height attributes are, either explicitly set through the\n    <code>width</code> and <code>height</code> attributes or are determined by the dimensions of the parent element. The\n    content contained in the Graphic will be sized to fit with in the Graphic instances dimensions. When using this\n    setting, the <code>preserveAspectRatio</code> attribute will determine how the contents are sized.</dd>\n    <dt>sizeGraphicToContent</dt><dd>(Also accepts a value of true) The Graphics width and height are determined by the\n    size and positioning of the content.</dd>\n    <dt>false</dt><dd>The Graphics width and height attributes are, either explicitly set through the <code>width</code>\n    and <code>height</code> attributes or are determined by the dimensions of the parent element. The contents of the\n    Graphic instance are not affected by this setting.</dd>\n</dl>"
     },
     "preserveAspectRatio": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_preserveAspectRatio",
      "!doc": "Determines how content is sized when <code>autoSize</code> is set to <code>sizeContentToGraphic</code>.\n\n <dl>\n     <dt>none<dt><dd>Do not force uniform scaling. Scale the graphic content of the given element non-uniformly if necessary\n     such that the elements bounding box exactly matches the viewport rectangle.</dd>\n     <dt>xMinYMin</dt><dd>Force uniform scaling position along the top left of the Graphics node.</dd>\n     <dt>xMidYMin</dt><dd>Force uniform scaling horizontally centered and positioned at the top of the Graphics node.<dd>\n     <dt>xMaxYMin</dt><dd>Force uniform scaling positioned horizontally from the right and vertically from the top.</dd>\n     <dt>xMinYMid</dt>Force uniform scaling positioned horizontally from the left and vertically centered.</dd>\n     <dt>xMidYMid (the default)</dt><dd>Force uniform scaling with the content centered.</dd>\n     <dt>xMaxYMid</dt><dd>Force uniform scaling positioned horizontally from the right and vertically centered.</dd>\n     <dt>xMinYMax</dt><dd>Force uniform scaling positioned horizontally from the left and vertically from the bottom.</dd>\n     <dt>xMidYMax</dt><dd>Force uniform scaling horizontally centered and position vertically from the bottom.</dd>\n     <dt>xMaxYMax</dt><dd>Force uniform scaling positioned horizontally from the right and vertically from the bottom.</dd>\n </dl>"
     },
     "resizeDown": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_resizeDown",
      "!doc": "The contentBounds will resize to greater values but not to smaller values. (for performance)\nWhen resizing the contentBounds down is desirable, set the resizeDown value to true."
     },
     "x": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_x",
      "!doc": "Indicates the x-coordinate for the instance."
     },
     "y": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#config_y",
      "!doc": "Indicates the y-coordinate for the instance."
     },
     "set": {
      "!type": "fn(name: string|+Object, value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_set",
      "!doc": "Sets the value of an attribute."
     },
     "getXY": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_getXY",
      "!doc": "Gets the current position of the graphic instance in page coordinates."
     },
     "destroy": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_destroy",
      "!doc": "Removes all nodes."
     },
     "addShape": {
      "!type": "fn(cfg: +Object) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_addShape",
      "!doc": "Generates a shape instance by type."
     },
     "removeShape": {
      "!type": "fn(shape: +graphics.Shape|string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_removeShape",
      "!doc": "Removes a shape instance from from the graphic instance."
     },
     "removeAllShapes": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_removeAllShapes",
      "!doc": "Removes all shape instances from the dom."
     },
     "clear": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_clear",
      "!doc": "Clears the graphics object."
     },
     "getShapeById": {
      "!type": "fn(id: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_getShapeById",
      "!doc": "Returns a shape based on the id of its dom node."
     },
     "batch": {
      "!type": "fn(method: fn())",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGGraphic.html#method_batch",
      "!doc": "Allows for creating multiple shapes in order to batch appending and redraw operations."
     }
    }
   },
   "SVGPath": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGPath.html",
    "prototype": {
     "path": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGPath.html#config_path",
      "!doc": "Indicates the path used for the node."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGPath.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGPath.html#config_height",
      "!doc": "Indicates the height of the shape"
     }
    }
   },
   "SVGPieSlice": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGPieSlice.html",
    "prototype": {
     "startAngle": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGPieSlice.html#config_startAngle",
      "!doc": "Starting angle in relation to a circle in which to begin the pie slice drawing."
     },
     "arc": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGPieSlice.html#config_arc",
      "!doc": "Arc of the slice."
     },
     "radius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGPieSlice.html#config_radius",
      "!doc": "Radius of the circle in which the pie slice is drawn"
     }
    }
   },
   "SVGShape": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html",
    "prototype": {
     "addClass": {
      "!type": "fn(className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_addClass",
      "!doc": "Add a class name to each node."
     },
     "removeClass": {
      "!type": "fn(className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_removeClass",
      "!doc": "Removes a class name from each node."
     },
     "getXY": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_getXY",
      "!doc": "Gets the current position of the node in page coordinates."
     },
     "setXY": {
      "!type": "fn(Contains: [?])",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_setXY",
      "!doc": "Set the position of the shape in page coordinates, regardless of how the node is positioned."
     },
     "contains": {
      "!type": "fn(needle: +graphics.SVGShape|+HTMLElement) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_contains",
      "!doc": "Determines whether the node is an ancestor of another HTML element in the DOM hierarchy."
     },
     "compareTo": {
      "!type": "fn(refNode: +HTMLElement|+node.Node) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_compareTo",
      "!doc": "Compares nodes to determine if they match.\nNode instances can be compared to each other and/or HTMLElements."
     },
     "test": {
      "!type": "fn(selector: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_test",
      "!doc": "Test if the supplied node matches the supplied selector."
     },
     "set": {
      "!type": "fn(name: string|+Object, value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_set",
      "!doc": "Sets the value of an attribute."
     },
     "translate": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_translate",
      "!doc": "Specifies a 2d translation."
     },
     "translateX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_translateX",
      "!doc": "Translates the shape along the x-axis. When translating x and y coordinates,\nuse the `translate` method."
     },
     "translateY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_translateY",
      "!doc": "Translates the shape along the y-axis. When translating x and y coordinates,\nuse the `translate` method."
     },
     "skew": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_skew",
      "!doc": "Skews the shape around the x-axis and y-axis."
     },
     "skewX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_skewX",
      "!doc": "Skews the shape around the x-axis."
     },
     "skewY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_skewY",
      "!doc": "Skews the shape around the y-axis."
     },
     "rotate": {
      "!type": "fn(deg: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_rotate",
      "!doc": "Rotates the shape clockwise around it transformOrigin."
     },
     "scale": {
      "!type": "fn(val: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_scale",
      "!doc": "Specifies a 2d scaling operation."
     },
     "getBounds": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_getBounds",
      "!doc": "Returns the bounds for a shape.\n\nCalculates the a new bounding box from the original corner coordinates (base on size and position) and the transform matrix.\nThe calculated bounding box is used by the graphic instance to calculate its viewBox."
     },
     "toFront": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_toFront",
      "!doc": "Places the shape underneath all other shapes."
     },
     "destroy": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#method_destroy",
      "!doc": "Destroys the shape instance."
     },
     "transformOrigin": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_transformOrigin",
      "!doc": "An array of x, y values which indicates the transformOrigin in which to rotate the shape. Valid values range between 0 and 1 representing a\nfraction of the shapes corresponding bounding box dimension. The default value is [0.5, 0.5]."
     },
     "transform": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_transform",
      "!doc": "<p>A string containing, in order, transform operations applied to the shape instance. The `transform` string can contain the following values:\n\n   <dl>\n       <dt>rotate</dt><dd>Rotates the shape clockwise around it transformOrigin.</dd>\n       <dt>translate</dt><dd>Specifies a 2d translation.</dd>\n       <dt>skew</dt><dd>Skews the shape around the x-axis and y-axis.</dd>\n       <dt>scale</dt><dd>Specifies a 2d scaling operation.</dd>\n       <dt>translateX</dt><dd>Translates the shape along the x-axis.</dd>\n       <dt>translateY</dt><dd>Translates the shape along the y-axis.</dd>\n       <dt>skewX</dt><dd>Skews the shape around the x-axis.</dd>\n       <dt>skewY</dt><dd>Skews the shape around the y-axis.</dd>\n       <dt>matrix</dt><dd>Specifies a 2D transformation matrix comprised of the specified six values.</dd>\n   </dl>\n</p>\n<p>Applying transforms through the transform attribute will reset the transform matrix and apply a new transform. The shape class also contains\ncorresponding methods for each transform that will apply the transform to the current matrix. The below code illustrates how you might use the\n`transform` attribute to instantiate a recangle with a rotation of 45 degrees.</p>\n           var myRect = new Y.Rect({\n               type:\"rect\",\n               width: 50,\n               height: 40,\n               transform: \"rotate(45)\"\n           };\n<p>The code below would apply `translate` and `rotate` to an existing shape.</p>\n\n       myRect.set(\"transform\", \"translate(40, 50) rotate(45)\");"
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_id",
      "!doc": "Unique id for class instance."
     },
     "x": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_x",
      "!doc": "Indicates the x position of shape."
     },
     "y": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_y",
      "!doc": "Indicates the y position of shape."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_height",
      "!doc": "Indicates the height of the shape"
     },
     "visible": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_visible",
      "!doc": "Indicates whether the shape is visible."
     },
     "shapeRendering": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_shapeRendering",
      "!doc": "Only implemented in SVG implementation.\nApplies the SVG shape-rendering attribute to the shape.\n <dl>\n     <dt>auto</dt>\n     <dd>Indicates that the user agent shall make appropriate tradeoffs to balance speed,\n     crisp edges and geometric precision, but with geometric precision given more importance than speed and crisp edges.</dd>\n     <dt>optimizeSpeed</dt>\n     <dd>Indicates that the user agent shall emphasize rendering speed over geometric precision and crisp edges.\n     This option will sometimes cause the user agent to turn off shape anti-aliasing.</dd>\n     <dt>crispEdges</dt>\n     <dd>Indicates that the user agent shall attempt to emphasize the contrast between clean edges of artwork over rendering\n     speed and geometric precision. To achieve crisp edges, the user agent might turn off anti-aliasing for all lines and curves\n     or possibly just for straight lines which are close to vertical or horizontal. Also, the user agent might adjust line\n     positions and line widths to align edges with device pixels.</dd>\n     <dt>geometricPrecision</dt>\n     <dd>Indicates that the user agent shall emphasize geometric precision over speed and crisp edges.</dd>\n </dl>"
     },
     "fill": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_fill",
      "!doc": "Contains information about the fill of the shape.\n <dl>\n     <dt>color</dt><dd>The color of the fill.</dd>\n     <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the fill. The default value is 1.</dd>\n     <dt>type</dt><dd>Type of fill.\n         <dl>\n             <dt>solid</dt><dd>Solid single color fill. (default)</dd>\n             <dt>linear</dt><dd>Linear gradient fill.</dd>\n             <dt>radial</dt><dd>Radial gradient fill.</dd>\n         </dl>\n     </dd>\n </dl>\n <p>If a `linear` or `radial` is specified as the fill type. The following additional property is used:\n <dl>\n     <dt>stops</dt><dd>An array of objects containing the following properties:\n         <dl>\n             <dt>color</dt><dd>The color of the stop.</dd>\n             <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stop. The default value is 1.\n             Note: No effect for IE 6 - 8</dd>\n             <dt>offset</dt><dd>Number between 0 and 1 indicating where the color stop is positioned.</dd>\n         </dl>\n     </dd>\n     <p>Linear gradients also have the following property:</p>\n     <dt>rotation</dt><dd>Linear gradients flow left to right by default. The rotation property allows you to change the\n     flow by rotation. (e.g. A rotation of 180 would make the gradient pain from right to left.)</dd>\n     <p>Radial gradients have the following additional properties:</p>\n     <dt>r</dt><dd>Radius of the gradient circle.</dd>\n     <dt>fx</dt><dd>Focal point x-coordinate of the gradient.</dd>\n     <dt>fy</dt><dd>Focal point y-coordinate of the gradient.</dd>\n     <dt>cx</dt><dd>\n         <p>The x-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>\n         <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and\n         `VMLShape` classes which are used on Android or IE 6 - 8.</p>\n     </dd>\n     <dt>cy</dt><dd>\n         <p>The y-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>\n         <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and `VMLShape`\n         classes which are used on Android or IE 6 - 8.</p>\n     </dd>\n </dl>"
     },
     "stroke": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_stroke",
      "!doc": "Contains information about the stroke of the shape.\n <dl>\n     <dt>color</dt><dd>The color of the stroke.</dd>\n     <dt>weight</dt><dd>Number that indicates the width of the stroke.</dd>\n     <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stroke. The default value is 1.</dd>\n     <dt>dashstyle</dt>Indicates whether to draw a dashed stroke. When set to \"none\", a solid stroke is drawn. When set\n     to an array, the first index indicates the length of the dash. The second index indicates the length of gap.\n     <dt>linecap</dt><dd>Specifies the linecap for the stroke. The following values can be specified:\n         <dl>\n             <dt>butt (default)</dt><dd>Specifies a butt linecap.</dd>\n             <dt>square</dt><dd>Specifies a sqare linecap.</dd>\n             <dt>round</dt><dd>Specifies a round linecap.</dd>\n         </dl>\n     </dd>\n     <dt>linejoin</dt><dd>Specifies a linejoin for the stroke. The following values can be specified:\n         <dl>\n             <dt>round (default)</dt><dd>Specifies that the linejoin will be round.</dd>\n             <dt>bevel</dt><dd>Specifies a bevel for the linejoin.</dd>\n             <dt>miter limit</dt><dd>An integer specifying the miter limit of a miter linejoin. If you want to specify a linejoin\n             of miter, you simply specify the limit as opposed to having separate miter and miter limit values.</dd>\n         </dl>\n     </dd>\n </dl>"
     },
     "node": {
      "!type": "+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_node",
      "!doc": "Dom node for the shape."
     },
     "data": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_data",
      "!doc": "Represents an SVG Path string. This will be parsed and added to shapes API to represent the SVG data across all\nimplementations. Note that when using VML or SVG implementations, part of this content will be added to the DOM using\nrespective VML/SVG attributes. If your content comes from an untrusted source, you will need to ensure that no\nmalicious code is included in that content."
     },
     "graphic": {
      "!type": "+graphics.SVGGraphic",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/SVGShape.html#config_graphic",
      "!doc": "Reference to the parent graphic instance"
     }
    }
   },
   "VMLCircle": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLCircle.html",
    "prototype": {
     "radius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLCircle.html#config_radius",
      "!doc": "Radius for the circle."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLCircle.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLCircle.html#config_height",
      "!doc": "Indicates the height of the shape"
     }
    }
   },
   "VMLDrawing": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html",
    "prototype": {
     "curveTo": {
      "!type": "fn(cp1x: number, cp1y: number, cp2x: number, cp2y: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_curveTo",
      "!doc": "Draws a bezier curve."
     },
     "relativeCurveTo": {
      "!type": "fn(cp1x: number, cp1y: number, cp2x: number, cp2y: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_relativeCurveTo",
      "!doc": "Draws a bezier curve."
     },
     "quadraticCurveTo": {
      "!type": "fn(cpx: number, cpy: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_quadraticCurveTo",
      "!doc": "Draws a quadratic bezier curve."
     },
     "relativeQuadraticCurveTo": {
      "!type": "fn(cpx: number, cpy: number, x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_relativeQuadraticCurveTo",
      "!doc": "Draws a quadratic bezier curve relative to the current position."
     },
     "drawRect": {
      "!type": "fn(x: number, y: number, w: number, h: number, ew: number, eh: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_drawRect",
      "!doc": "Draws a rectangle with rounded corners."
     },
     "lineTo": {
      "!type": "fn(point1: number, point2: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_lineTo",
      "!doc": "Draws a line segment from the current drawing position to the specified x and y coordinates."
     },
     "relativeLineTo": {
      "!type": "fn(point1: number, point2: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_relativeLineTo",
      "!doc": "Draws a line segment using the current line style from the current drawing position to the relative x and y coordinates."
     },
     "moveTo": {
      "!type": "fn(x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_moveTo",
      "!doc": "Moves the current drawing position to specified x and y coordinates."
     },
     "relativeMoveTo": {
      "!type": "fn(x: number, y: number) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_relativeMoveTo",
      "!doc": "Moves the current drawing position relative to specified x and y coordinates."
     },
     "end": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_end",
      "!doc": "Completes a drawing operation."
     },
     "closePath": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_closePath",
      "!doc": "Ends a fill and stroke"
     },
     "clear": {
      "!type": "fn() -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLDrawing.html#method_clear",
      "!doc": "Clears the path."
     }
    }
   },
   "VMLEllipse": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLEllipse.html",
    "prototype": {
     "xRadius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLEllipse.html#config_xRadius",
      "!doc": "Horizontal radius for the ellipse."
     },
     "yRadius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLEllipse.html#config_yRadius",
      "!doc": "Vertical radius for the ellipse."
     }
    }
   },
   "VMLGraphic": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html",
    "prototype": {
     "render": {
      "!type": "fn(parentNode: +HTMLElement)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_render",
      "!doc": "Adds the graphics node to the dom."
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_id",
      "!doc": "Unique id for class instance."
     },
     "shapes": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_shapes",
      "!doc": "Key value pairs in which a shape instance is associated with its id."
     },
     "contentBounds": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_contentBounds",
      "!doc": "Object containing size and coordinate data for the content of a Graphic in relation to the coordSpace node."
     },
     "node": {
      "!type": "+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_node",
      "!doc": "The html element that represents to coordinate system of the Graphic instance."
     },
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_width",
      "!doc": "Indicates the width of the `Graphic`."
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_height",
      "!doc": "Indicates the height of the `Graphic`."
     },
     "autoSize": {
      "!type": "bool|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_autoSize",
      "!doc": "Determines the sizing of the Graphic.\n\n<dl>\n    <dt>sizeContentToGraphic</dt><dd>The Graphics width and height attributes are, either explicitly set through the\n    <code>width</code> and <code>height</code> attributes or are determined by the dimensions of the parent element. The\n    content contained in the Graphic will be sized to fit with in the Graphic instances dimensions. When using this\n    setting, the <code>preserveAspectRatio</code> attribute will determine how the contents are sized.</dd>\n    <dt>sizeGraphicToContent</dt><dd>(Also accepts a value of true) The Graphics width and height are determined by the\n    size and positioning of the content.</dd>\n    <dt>false</dt><dd>The Graphics width and height attributes are, either explicitly set through the <code>width</code>\n    and <code>height</code> attributes or are determined by the dimensions of the parent element. The contents of the\n    Graphic instance are not affected by this setting.</dd>\n</dl>"
     },
     "preserveAspectRatio": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_preserveAspectRatio",
      "!doc": "Determines how content is sized when <code>autoSize</code> is set to <code>sizeContentToGraphic</code>.\n\n <dl>\n     <dt>none<dt><dd>Do not force uniform scaling. Scale the graphic content of the given element non-uniformly if necessary\n     such that the elements bounding box exactly matches the viewport rectangle.</dd>\n     <dt>xMinYMin</dt><dd>Force uniform scaling position along the top left of the Graphics node.</dd>\n     <dt>xMidYMin</dt><dd>Force uniform scaling horizontally centered and positioned at the top of the Graphics node.<dd>\n     <dt>xMaxYMin</dt><dd>Force uniform scaling positioned horizontally from the right and vertically from the top.</dd>\n     <dt>xMinYMid</dt>Force uniform scaling positioned horizontally from the left and vertically centered.</dd>\n     <dt>xMidYMid (the default)</dt><dd>Force uniform scaling with the content centered.</dd>\n     <dt>xMaxYMid</dt><dd>Force uniform scaling positioned horizontally from the right and vertically centered.</dd>\n     <dt>xMinYMax</dt><dd>Force uniform scaling positioned horizontally from the left and vertically from the bottom.</dd>\n     <dt>xMidYMax</dt><dd>Force uniform scaling horizontally centered and position vertically from the bottom.</dd>\n     <dt>xMaxYMax</dt><dd>Force uniform scaling positioned horizontally from the right and vertically from the bottom.</dd>\n </dl>"
     },
     "resizeDown": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_resizeDown",
      "!doc": "The contentBounds will resize to greater values but not values. (for performance)\nWhen resizing the contentBounds down is desirable, set the resizeDown value to true."
     },
     "x": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_x",
      "!doc": "Indicates the x-coordinate for the instance."
     },
     "y": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#config_y",
      "!doc": "Indicates the y-coordinate for the instance."
     },
     "set": {
      "!type": "fn(name: string|+Object, value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_set",
      "!doc": "Sets the value of an attribute."
     },
     "getXY": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_getXY",
      "!doc": "Gets the current position of the graphic instance in page coordinates."
     },
     "destroy": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_destroy",
      "!doc": "Removes all nodes."
     },
     "addShape": {
      "!type": "fn(cfg: +Object) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_addShape",
      "!doc": "Generates a shape instance by type."
     },
     "removeShape": {
      "!type": "fn(shape: +graphics.Shape|string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_removeShape",
      "!doc": "Removes a shape instance from from the graphic instance."
     },
     "removeAllShapes": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_removeAllShapes",
      "!doc": "Removes all shape instances from the dom."
     },
     "clear": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_clear",
      "!doc": "Clears the graphics object."
     },
     "setSize": {
      "!type": "fn(w: number, h: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_setSize",
      "!doc": "Sets the size of the graphics object."
     },
     "setPosition": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_setPosition",
      "!doc": "Sets the positon of the graphics object."
     },
     "getShapeById": {
      "!type": "fn(id: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_getShapeById",
      "!doc": "Returns a shape based on the id of its dom node."
     },
     "batch": {
      "!type": "fn(method: fn())",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLGraphic.html#method_batch",
      "!doc": "Allows for creating multiple shapes in order to batch appending and redraw operations."
     }
    }
   },
   "VMLPath": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLPath.html",
    "prototype": {
     "width": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLPath.html#config_width",
      "!doc": "Indicates the width of the shape"
     },
     "height": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLPath.html#config_height",
      "!doc": "Indicates the height of the shape"
     },
     "path": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLPath.html#config_path",
      "!doc": "Indicates the path used for the node."
     }
    }
   },
   "VMLPieSlice": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLPieSlice.html",
    "prototype": {
     "startAngle": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLPieSlice.html#config_startAngle",
      "!doc": "Starting angle in relation to a circle in which to begin the pie slice drawing."
     },
     "arc": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLPieSlice.html#config_arc",
      "!doc": "Arc of the slice."
     },
     "radius": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLPieSlice.html#config_radius",
      "!doc": "Radius of the circle in which the pie slice is drawn"
     }
    }
   },
   "VMLShape": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html",
    "prototype": {
     "addClass": {
      "!type": "fn(className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_addClass",
      "!doc": "Add a class name to each node."
     },
     "removeClass": {
      "!type": "fn(className: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_removeClass",
      "!doc": "Removes a class name from each node."
     },
     "getXY": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_getXY",
      "!doc": "Gets the current position of the node in page coordinates."
     },
     "setXY": {
      "!type": "fn(Contains: [?])",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_setXY",
      "!doc": "Set the position of the shape in page coordinates, regardless of how the node is positioned."
     },
     "contains": {
      "!type": "fn(needle: +graphics.VMLShape|+HTMLElement) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_contains",
      "!doc": "Determines whether the node is an ancestor of another HTML element in the DOM hierarchy."
     },
     "compareTo": {
      "!type": "fn(refNode: +HTMLElement|+node.Node) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_compareTo",
      "!doc": "Compares nodes to determine if they match.\nNode instances can be compared to each other and/or HTMLElements."
     },
     "test": {
      "!type": "fn(selector: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_test",
      "!doc": "Test if the supplied node matches the supplied selector."
     },
     "translate": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_translate",
      "!doc": "Specifies a 2d translation."
     },
     "translateX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_translateX",
      "!doc": "Translates the shape along the x-axis. When translating x and y coordinates,\nuse the `translate` method."
     },
     "translateY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_translateY",
      "!doc": "Performs a translate on the y-coordinate. When translating x and y coordinates,\nuse the `translate` method."
     },
     "skew": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_skew",
      "!doc": "Skews the shape around the x-axis and y-axis."
     },
     "skewX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_skewX",
      "!doc": "Skews the shape around the x-axis."
     },
     "skewY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_skewY",
      "!doc": "Skews the shape around the y-axis."
     },
     "rotate": {
      "!type": "fn(deg: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_rotate",
      "!doc": "Rotates the shape clockwise around it transformOrigin."
     },
     "scale": {
      "!type": "fn(val: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_scale",
      "!doc": "Specifies a 2d scaling operation."
     },
     "set": {
      "!type": "fn(name: string|+Object, value: ?)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_set",
      "!doc": "Sets the value of an attribute."
     },
     "getBounds": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_getBounds",
      "!doc": "Returns the bounds for a shape.\n\nCalculates the a new bounding box from the original corner coordinates (base on size and position) and the transform matrix.\nThe calculated bounding box is used by the graphic instance to calculate its viewBox."
     },
     "toFront": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_toFront",
      "!doc": "Places the shape underneath all other shapes."
     },
     "destroy": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#method_destroy",
      "!doc": "Destroys shape"
     },
     "transformOrigin": {
      "!type": "[?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_transformOrigin",
      "!doc": "An array of x, y values which indicates the transformOrigin in which to rotate the shape. Valid values range between 0 and 1 representing a\nfraction of the shapes corresponding bounding box dimension. The default value is [0.5, 0.5]."
     },
     "transform": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_transform",
      "!doc": "<p>A string containing, in order, transform operations applied to the shape instance. The `transform` string can contain the following values:\n\n   <dl>\n       <dt>rotate</dt><dd>Rotates the shape clockwise around it transformOrigin.</dd>\n       <dt>translate</dt><dd>Specifies a 2d translation.</dd>\n       <dt>skew</dt><dd>Skews the shape around the x-axis and y-axis.</dd>\n       <dt>scale</dt><dd>Specifies a 2d scaling operation.</dd>\n       <dt>translateX</dt><dd>Translates the shape along the x-axis.</dd>\n       <dt>translateY</dt><dd>Translates the shape along the y-axis.</dd>\n       <dt>skewX</dt><dd>Skews the shape around the x-axis.</dd>\n       <dt>skewY</dt><dd>Skews the shape around the y-axis.</dd>\n       <dt>matrix</dt><dd>Specifies a 2D transformation matrix comprised of the specified six values.</dd>\n   </dl>\n</p>\n<p>Applying transforms through the transform attribute will reset the transform matrix and apply a new transform. The shape class also contains\ncorresponding methods for each transform that will apply the transform to the current matrix. The below code illustrates how you might use the\n`transform` attribute to instantiate a recangle with a rotation of 45 degrees.</p>\n           var myRect = new Y.Rect({\n               type:\"rect\",\n               width: 50,\n               height: 40,\n               transform: \"rotate(45)\"\n           };\n<p>The code below would apply `translate` and `rotate` to an existing shape.</p>\n\n       myRect.set(\"transform\", \"translate(40, 50) rotate(45)\");"
     },
     "x": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_x",
      "!doc": "Indicates the x position of shape."
     },
     "y": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_y",
      "!doc": "Indicates the y position of shape."
     },
     "id": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_id",
      "!doc": "Unique id for class instance."
     },
     "width": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_width"
     },
     "height": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_height"
     },
     "visible": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_visible",
      "!doc": "Indicates whether the shape is visible."
     },
     "fill": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_fill",
      "!doc": "Contains information about the fill of the shape.\n <dl>\n     <dt>color</dt><dd>The color of the fill.</dd>\n     <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the fill. The default value is 1.</dd>\n     <dt>type</dt><dd>Type of fill.\n         <dl>\n             <dt>solid</dt><dd>Solid single color fill. (default)</dd>\n             <dt>linear</dt><dd>Linear gradient fill.</dd>\n             <dt>radial</dt><dd>Radial gradient fill.</dd>\n         </dl>\n     </dd>\n </dl>\n <p>If a `linear` or `radial` is specified as the fill type. The following additional property is used:\n <dl>\n     <dt>stops</dt><dd>An array of objects containing the following properties:\n         <dl>\n             <dt>color</dt><dd>The color of the stop.</dd>\n             <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stop. The default value is 1.\n             Note: No effect for IE 6 - 8</dd>\n             <dt>offset</dt><dd>Number between 0 and 1 indicating where the color stop is positioned.</dd>\n         </dl>\n     </dd>\n     <p>Linear gradients also have the following property:</p>\n     <dt>rotation</dt><dd>Linear gradients flow left to right by default. The rotation property allows you to change the\n     flow by rotation. (e.g. A rotation of 180 would make the gradient pain from right to left.)</dd>\n     <p>Radial gradients have the following additional properties:</p>\n     <dt>r</dt><dd>Radius of the gradient circle.</dd>\n     <dt>fx</dt><dd>Focal point x-coordinate of the gradient.</dd>\n     <dt>fy</dt><dd>Focal point y-coordinate of the gradient.</dd>\n </dl>\n <p>The corresponding `SVGShape` class implements the following additional properties.</p>\n <dl>\n     <dt>cx</dt><dd>\n         <p>The x-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>\n         <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and\n         `VMLShape` classes which are used on Android or IE 6 - 8.</p>\n     </dd>\n     <dt>cy</dt><dd>\n         <p>The y-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>\n         <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and `VMLShape`\n         classes which are used on Android or IE 6 - 8.</p>\n     </dd>\n </dl>\n <p>These properties are not currently implemented in `CanvasShape` or `VMLShape`.</p>"
     },
     "stroke": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_stroke",
      "!doc": "Contains information about the stroke of the shape.\n <dl>\n     <dt>color</dt><dd>The color of the stroke.</dd>\n     <dt>weight</dt><dd>Number that indicates the width of the stroke.</dd>\n     <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stroke. The default value is 1.</dd>\n     <dt>dashstyle</dt>Indicates whether to draw a dashed stroke. When set to \"none\", a solid stroke is drawn. When set\n     to an array, the first index indicates the length of the dash. The second index indicates the length of gap.\n     <dt>linecap</dt><dd>Specifies the linecap for the stroke. The following values can be specified:\n         <dl>\n             <dt>butt (default)</dt><dd>Specifies a butt linecap.</dd>\n             <dt>square</dt><dd>Specifies a sqare linecap.</dd>\n             <dt>round</dt><dd>Specifies a round linecap.</dd>\n         </dl>\n     </dd>\n     <dt>linejoin</dt><dd>Specifies a linejoin for the stroke. The following values can be specified:\n         <dl>\n             <dt>round (default)</dt><dd>Specifies that the linejoin will be round.</dd>\n             <dt>bevel</dt><dd>Specifies a bevel for the linejoin.</dd>\n             <dt>miter limit</dt><dd>An integer specifying the miter limit of a miter linejoin. If you want to specify a linejoin\n             of miter, you simply specify the limit as opposed to having separate miter and miter limit values.</dd>\n         </dl>\n     </dd>\n </dl>"
     },
     "node": {
      "!type": "+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_node",
      "!doc": "Dom node for the shape."
     },
     "data": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_data",
      "!doc": "Represents an SVG Path string. This will be parsed and added to shapes API to represent the SVG data across all\nimplementations. Note that when using VML or SVG implementations, part of this content will be added to the DOM using\nrespective VML/SVG attributes. If your content comes from an untrusted source, you will need to ensure that no\nmalicious code is included in that content."
     },
     "graphic": {
      "!type": "+graphics.Graphic",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/VMLShape.html#config_graphic",
      "!doc": "Reference to the container Graphic."
     }
    }
   },
   "Handlebars": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html",
    "!data": {
     "for": "handlebars.Handlebars"
    },
    "prototype": {
     "registerHelper": {
      "!type": "fn(name: string, fn: fn(), inverse?: bool)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html#method_registerHelper",
      "!doc": "Registers a helper function that will be made available to all templates.\n\nHelper functions receive the current template context as the `this` object, and\ncan also receive arguments passed by the template."
     },
     "registerPartial": {
      "!type": "fn(name: string, partial: fn())",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html#method_registerPartial",
      "!doc": "Registers a partial that will be made available to all templates.\n\nA partial is another template that can be used to render part of a larger\ntemplate. For example, a website with a common header and footer across all its\npages might use a template for each page, which would call shared partials to\nrender the headers and footers.\n\nPartials may be specified as uncompiled template strings or as compiled template\nfunctions."
     },
     "template": {
      "!type": "fn(template: fn()) -> fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html#method_template",
      "!doc": "Converts a precompiled template into a renderable template function."
     }
    }
   }
  },
  "handlebars": {
   "!data": {
    "submodules": {
     "handlebars-base": {},
     "handlebars-compiler": {}
    }
   },
   "Handlebars": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html",
    "prototype": {
     "log": {
      "!type": "fn(level: string, message: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html#method_log",
      "!doc": "Logs a debugging message. Note that messages will only be logged when the\nhandlebars module is loaded in \"debug\" mode.",
      "!data": {
       "submodule": "handlebars-compiler"
      }
     },
     "render": {
      "!type": "fn(string: string, context: +Object, options?: +config.HandlebarsRenderConfig) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html#method_render",
      "!doc": "Compiles and renders a Handlebars template string in a single step.\n\nIf youll be using a template more than once, its more efficient to compile it\ninto a function once using `compile()`, and then render it whenever you need to\nby simply executing the compiled function. However, if you only need to compile\nand render a template once, `render()` is a handy shortcut for doing both in a\nsingle step.",
      "!data": {
       "submodule": "handlebars-compiler"
      }
     },
     "compile": {
      "!type": "fn(string: string, options?: +Object) -> fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html#method_compile",
      "!doc": "Compiles a Handlebars template string into a function. To render the template,\ncall the function and pass in a context object.",
      "!data": {
       "submodule": "handlebars-compiler"
      }
     },
     "precompile": {
      "!type": "fn(string: string, options?: +Object) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Handlebars.html#method_precompile",
      "!doc": "Precompiles a Handlebars template string into a string of JavaScript code. This\ncan be used to precompile a template at build time or on the server, and the\nresulting template can then be rendered at runtime or on the client without\nneeding to go through a compile step.\n\nTo render a precompiled template, evaluate the code and then pass the resulting\nfunction to `Y.Handlebars.template()` to get back an executable template\nfunction.",
      "!data": {
       "submodule": "handlebars-compiler"
      }
     }
    }
   }
  },
  "highlight": {
   "!data": {
    "submodules": {
     "highlight-accentfold": {},
     "highlight-base": {}
    }
   },
   "Highlight": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html",
    "allFold": {
     "!type": "fn(haystack: string, needles: string|[string], options?: +config.HighlightAllFoldConfig) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_allFold",
     "!doc": "Accent-folding version of `all()`.",
     "!data": {
      "submodule": "highlight-accentfold"
     }
    },
    "startFold": {
     "!type": "fn(haystack: string, needles: string|[string]) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_startFold",
     "!doc": "Accent-folding version of `start()`.",
     "!data": {
      "submodule": "highlight-accentfold"
     }
    },
    "wordsFold": {
     "!type": "fn(haystack: string, needles: string|[string]) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_wordsFold",
     "!doc": "Accent-folding version of `words()`.",
     "!data": {
      "submodule": "highlight-accentfold"
     }
    },
    "all": {
     "!type": "fn(haystack: string, needles: string|[string], options?: +config.HighlightAllConfig) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_all",
     "!doc": "Highlights all occurrences in the _haystack_ string of the items in the\n_needles_ array, regardless of where they occur. The returned string will\nhave all HTML characters escaped except for the highlighting markup.",
     "!data": {
      "submodule": "highlight-base"
     }
    },
    "allCase": {
     "!type": "fn(haystack: string, needles: string|[string], options?: +Object) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_allCase",
     "!doc": "Same as `all()`, but case-sensitive by default.",
     "!data": {
      "submodule": "highlight-base"
     }
    },
    "start": {
     "!type": "fn(haystack: string, needles: string|[string], options?: +config.HighlightStartConfig) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_start",
     "!doc": "Highlights _needles_ that occur at the start of _haystack_. The returned\nstring will have all HTML characters escaped except for the highlighting\nmarkup.",
     "!data": {
      "submodule": "highlight-base"
     }
    },
    "startCase": {
     "!type": "fn(haystack: string, needles: string|[string]) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_startCase",
     "!doc": "Same as `start()`, but case-sensitive by default.",
     "!data": {
      "submodule": "highlight-base"
     }
    },
    "words": {
     "!type": "fn(haystack: string, needles: string|[string], options?: +config.HighlightWordsConfig) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_words",
     "!doc": "Highlights complete words in the _haystack_ string that are also in the\n_needles_ array. The returned string will have all HTML characters escaped\nexcept for the highlighting markup.",
     "!data": {
      "submodule": "highlight-base"
     }
    },
    "wordsCase": {
     "!type": "fn(haystack: string, needles: string|[string]) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Highlight.html#method_wordsCase",
     "!doc": "Same as `words()`, but case-sensitive by default.",
     "!data": {
      "submodule": "highlight-base"
     }
    }
   }
  },
  "history": {
   "!data": {
    "submodules": {
     "history-base": {},
     "history-hash-ie": {},
     "history-hash": {},
     "history-html5": {}
    }
   },
   "HistoryBase": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html",
    "NAME": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#property_NAME",
     "!doc": "Name of this component.",
     "!data": {
      "submodule": "history-base"
     }
    },
    "SRC_ADD": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#property_SRC_ADD",
     "!doc": "Constant used to identify state changes originating from the\n<code>add()</code> method.",
     "!data": {
      "submodule": "history-base"
     }
    },
    "SRC_REPLACE": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#property_SRC_REPLACE",
     "!doc": "Constant used to identify state changes originating from the\n<code>replace()</code> method.",
     "!data": {
      "submodule": "history-base"
     }
    },
    "html5": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#property_html5",
     "!doc": "Whether or not this browser supports the HTML5 History API.",
     "!data": {
      "submodule": "history-base"
     }
    },
    "nativeHashChange": {
     "!type": "bool",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#property_nativeHashChange",
     "!doc": "Whether or not this browser supports the <code>window.onhashchange</code>\nevent natively. Note that even if this is <code>true</code>, you may\nstill want to use HistoryHashs synthetic <code>hashchange</code> event\nsince it normalizes implementation differences and fixes spec violations\nacross various browsers.",
     "!data": {
      "submodule": "history-base"
     }
    },
    "prototype": {
     "force": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#property_force",
      "!doc": "If `true`, a `history:change` event will be fired whenever the URL\nchanges, even if there is no associated state change.",
      "!data": {
       "submodule": "history-base"
      }
     },
     "add": {
      "!type": "fn(state: +Object, options: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#method_add",
      "!doc": "Adds a state entry with new values for the specified keys. By default,\nthe new state will be merged into the existing state, and new values will\noverride existing values. Specifying a <code>null</code> or\n<code>undefined</code> value will cause that key to be removed from the\nnew state entry.",
      "!data": {
       "submodule": "history-base"
      }
     },
     "addValue": {
      "!type": "fn(key: string, value: string, options: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#method_addValue",
      "!doc": "Adds a state entry with a new value for a single key. By default, the new\nvalue will be merged into the existing state values, and will override an\nexisting value with the same key if there is one. Specifying a\n<code>null</code> or <code>undefined</code> value will cause the key to\nbe removed from the new state entry.",
      "!data": {
       "submodule": "history-base"
      }
     },
     "get": {
      "!type": "fn(key: string) -> +Object|string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#method_get",
      "!doc": "Returns the current value of the state parameter specified by <i>key</i>,\nor an object hash of key/value pairs for all current state parameters if\nno key is specified.",
      "!data": {
       "submodule": "history-base"
      }
     },
     "replace": {
      "!type": "fn(state: +Object, options: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#method_replace",
      "!doc": "Same as <code>add()</code> except that a new browser history entry will\nnot be created. Instead, the current history entry will be replaced with\nthe new state.",
      "!data": {
       "submodule": "history-base"
      }
     },
     "replaceValue": {
      "!type": "fn(key: string, value: string, options: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryBase.html#method_replaceValue",
      "!doc": "Same as <code>addValue()</code> except that a new browser history entry\nwill not be created. Instead, the current history entry will be replaced\nwith the new state.",
      "!data": {
       "submodule": "history-base"
      }
     }
    }
   },
   "HistoryHash": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html",
    "getIframeHash": {
     "!type": "fn() -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_getIframeHash",
     "!doc": "Gets the raw (not decoded) current location hash from the IE iframe,\nminus the preceding # character and the hashPrefix (if one is set).",
     "!data": {
      "submodule": "history-hash-ie"
     }
    },
    "SRC_HASH": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#property_SRC_HASH",
     "!doc": "Constant used to identify state changes originating from\n<code>hashchange</code> events.",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "hashPrefix": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#property_hashPrefix",
     "!doc": "<p>\nPrefix to prepend when setting the hash fragment. For example, if the\nprefix is <code>!</code> and the hash fragment is set to\n<code>#foo=bar&baz=quux</code>, the final hash fragment in the URL will\nbecome <code>#!foo=bar&baz=quux</code>. This can be used to help make an\nAjax application crawlable in accordance with Googles guidelines at\n<a href=\"http://code.google.com/web/ajaxcrawling/\">http://code.google.com/web/ajaxcrawling/</a>.\n</p>\n\n<p>\nNote that this prefix applies to all HistoryHash instances. Its not\npossible for individual instances to use their own prefixes since they\nall operate on the same URL.\n</p>",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "createHash": {
     "!type": "fn(params: +Object) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_createHash",
     "!doc": "Creates a location hash string from the specified object of key/value\npairs.",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "decode": {
     "!type": "fn(string: string) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_decode",
     "!doc": "Wrapper around <code>decodeURIComponent()</code> that also converts +\nchars into spaces.",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "encode": {
     "!type": "fn(string: string) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_encode",
     "!doc": "Wrapper around <code>encodeURIComponent()</code> that converts spaces to\n+ chars.",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "getHash": {
     "!type": "fn() -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_getHash",
     "!doc": "Gets the raw (not decoded) current location hash, minus the preceding #\ncharacter and the hashPrefix (if one is set).",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "getUrl": {
     "!type": "fn() -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_getUrl",
     "!doc": "Gets the current bookmarkable URL.",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "parseHash": {
     "!type": "fn(hash: string) -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_parseHash",
     "!doc": "Parses a location hash string into an object of key/value parameter\npairs. If <i>hash</i> is not specified, the current location hash will\nbe used.",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "replaceHash": {
     "!type": "fn(hash: string)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_replaceHash",
     "!doc": "Replaces the browsers current location hash with the specified hash\nand removes all forward navigation states, without creating a new browser\nhistory entry. Automatically prepends the <code>hashPrefix</code> if one\nis set.",
     "!data": {
      "submodule": "history-hash"
     }
    },
    "setHash": {
     "!type": "fn(hash: string)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHash.html#method_setHash",
     "!doc": "Sets the browsers location hash to the specified string. Automatically\nprepends the <code>hashPrefix</code> if one is set.",
     "!data": {
      "submodule": "history-hash"
     }
    }
   },
   "HistoryHTML5": {
    "!type": "fn(config: +config.HistoryHTML5Config)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHTML5.html",
    "prototype": {
     "!proto": "history.HistoryBase.prototype"
    },
    "SRC_POPSTATE": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/HistoryHTML5.html#property_SRC_POPSTATE",
     "!doc": "Constant used to identify state changes originating from\n<code>popstate</code> events.",
     "!data": {
      "submodule": "history-html5"
     }
    }
   },
   "config": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/config.html",
    "!data": {
     "for": "yui.config"
    },
    "prototype": {
     "useHistoryHTML5": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/config.html#property_useHistoryHTML5",
      "!doc": "<p>\nIf <code>true</code>, the <code>Y.History</code> alias will always point to\n<code>Y.HistoryHTML5</code> when the history-html5 module is loaded, even if\nthe current browser doesnt support HTML5 history.\n</p>\n\n<p>\nIf <code>false</code>, the <code>Y.History</code> alias will always point to\n<code>Y.HistoryHash</code> when the history-hash module is loaded, even if\nthe current browser supports HTML5 history.\n</p>\n\n<p>\nIf neither <code>true</code> nor <code>false</code>, the\n<code>Y.History</code> alias will point to the best available history adapter\nthat the browser supports. This is the default behavior.\n</p>",
      "!data": {
       "submodule": "history-html5"
      }
     }
    }
   }
  },
  "imageloader": {
   "ImgLoadGroup": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "addTrigger": {
      "!type": "fn(obj: +Object, type: string) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#method_addTrigger",
      "!doc": "Adds a trigger to the group. Arguments are passed to <code>Y.on</code>."
     },
     "addCustomTrigger": {
      "!type": "fn(name: string, obj: +Object) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#method_addCustomTrigger",
      "!doc": "Adds a custom event trigger to the group."
     },
     "registerImage": {
      "!type": "fn(config: +config.ImgLoadGroupRegisterImageConfig) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#method_registerImage",
      "!doc": "Registers an image with the group.\nArguments are passed through to a <code>Y.ImgLoadImgObj</code> constructor; see that class attribute documentation for detailed information. \"<code>domId</code>\" is a required attribute."
     },
     "fetch": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadGroup.html#method_fetch",
      "!doc": "Displays the images in the group.\nThis method is called when a trigger fires or the time limit expires; it shouldnt be called externally, but is not private in the rare event that it needs to be called immediately."
     }
    }
   },
   "ImgLoadImgObj": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html",
    "prototype": {
     "!proto": "base.Base.prototype",
     "fetch": {
      "!type": "fn(withinY: number) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/ImgLoadImgObj.html#method_fetch",
      "!doc": "Displays the image; puts the URL into the DOM.\nThis method shouldnt be called externally, but is not private in the rare event that it needs to be called immediately."
     }
    }
   }
  },
  "intl": {
   "Intl": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Intl.html",
    "!data": {
     "extends": [
      "event_custom.EventTarget"
     ],
     "for": "features.Intl"
    },
    "prototype": {
     "setLang": {
      "!type": "fn(module: string, lang: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Intl.html#method_setLang",
      "!doc": "Sets the active language for the given module.\n\nReturns false on failure, which would happen if the language had not been registered through the <a href=\"#method_add\">add()</a> method."
     },
     "getLang": {
      "!type": "fn(module: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Intl.html#method_getLang",
      "!doc": "Get the currently active language for the given module."
     },
     "add": {
      "!type": "fn(module: string, lang: string, strings: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Intl.html#method_add",
      "!doc": "Register a hash of localized resources for the given module and language"
     },
     "get": {
      "!type": "fn(module: string, key: string, lang: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Intl.html#method_get",
      "!doc": "Gets the modules localized resources for the currently active language (as provided by the <a href=\"#method_getLang\">getLang</a> method).\n<p>\nOptionally, the localized resources for alternate languages which have been added to Intl (see the <a href=\"#method_add\">add</a> method) can\nbe retrieved by providing the BCP 47 language tag as the lang parameter.\n</p>"
     },
     "getAvailableLangs": {
      "!type": "fn(module: string) -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Intl.html#method_getAvailableLangs",
      "!doc": "Gets the list of languages for which localized resources are available for a given module, based on the module\nmeta-data (part of loader). If loader is not on the page, returns an empty array."
     }
    }
   }
  },
  "io": {
   "!data": {
    "submodules": {
     "io-base": {},
     "io-form": {},
     "io-queue": {},
     "io-upload-iframe": {},
     "io-xdr": {}
    }
   },
   "IO": {
    "!type": "fn(config: +config.IOConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html",
    "prototype": {
     "start": {
      "!type": "fn(transaction: +Object, config: +config.IOStartConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_start",
      "!doc": "Fires event \"io:start\" and creates, fires a transaction-specific\nstart event, if `config.on.start` is defined.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "complete": {
      "!type": "fn(transaction: +Object, config: +config.IOCompleteConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_complete",
      "!doc": "Fires event \"io:complete\" and creates, fires a\ntransaction-specific \"complete\" event, if config.on.complete is\ndefined.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "end": {
      "!type": "fn(transaction: +Object, config: +config.IOEndConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_end",
      "!doc": "Fires event \"io:end\" and creates, fires a transaction-specific \"end\"\nevent, if config.on.end is defined.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "success": {
      "!type": "fn(transaction: +Object, config: +config.IOSuccessConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_success",
      "!doc": "Fires event \"io:success\" and creates, fires a transaction-specific\n\"success\" event, if config.on.success is defined.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "failure": {
      "!type": "fn(transaction: +Object, config: +config.IOFailureConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_failure",
      "!doc": "Fires event \"io:failure\" and creates, fires a transaction-specific\n\"failure\" event, if config.on.failure is defined.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "progress": {
      "!type": "fn(transaction: +Object, progress: +Object, config: +config.IOProgressConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_progress",
      "!doc": "Fires event \"io:progress\" and creates, fires a transaction-specific\n\"progress\" event -- for XMLHttpRequest file upload -- if\nconfig.on.progress is defined.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "load": {
      "!type": "fn(transaction: +Object, load: +Object, config: +config.IOLoadConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_load",
      "!doc": "Fires event \"io:complete\" and creates, fires a transaction-specific\n\"complete\" event -- for XMLHttpRequest file upload -- if\nconfig.on.complete is defined.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "error": {
      "!type": "fn(transaction: +Object, error: +Object, config: +config.IOErrorConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_error",
      "!doc": "Fires event \"io:failure\" and creates, fires a transaction-specific\n\"failure\" event -- for XMLHttpRequest file upload -- if\nconfig.on.failure is defined.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "setHeader": {
      "!type": "fn(name: string, value: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_setHeader",
      "!doc": "Stores default client headers for all transactions. If a label is\npassed with no value argument, the header will be deleted.",
      "!data": {
       "submodule": "io-base"
      }
     },
     "send": {
      "!type": "fn(uri: string, config: +config.IOSendConfig, id: number) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_send",
      "!doc": "Requests a transaction. `send()` is implemented as `Y.io()`.  Each\ntransaction may include a configuration object.  Its properties are:\n\n<dl>\n  <dt>method</dt>\n    <dd>HTTP method verb (e.g., GET or POST). If this property is not\n        not defined, the default value will be GET.</dd>\n\n  <dt>data</dt>\n    <dd>This is the name-value string that will be sent as the\n    transaction data. If the request is HTTP GET, the data become\n    part of querystring. If HTTP POST, the data are sent in the\n    message body.</dd>\n\n  <dt>xdr</dt>\n    <dd>Defines the transport to be used for cross-domain requests.\n    By setting this property, the transaction will use the specified\n    transport instead of XMLHttpRequest. The properties of the\n    transport object are:\n    <dl>\n      <dt>use</dt>\n        <dd>The transport to be used: flash or native</dd>\n      <dt>dataType</dt>\n        <dd>Set the value to XML if that is the expected response\n        content type.</dd>\n      <dt>credentials</dt>\n        <dd>Set the value to true to set XHR.withCredentials property to true.</dd>\n    </dl></dd>\n\n  <dt>form</dt>\n    <dd>Form serialization configuration object.  Its properties are:\n    <dl>\n      <dt>id</dt>\n        <dd>Node object or id of HTML form</dd>\n      <dt>useDisabled</dt>\n        <dd>`true` to also serialize disabled form field values\n        (defaults to `false`)</dd>\n    </dl></dd>\n\n  <dt>on</dt>\n    <dd>Assigns transaction event subscriptions. Available events are:\n    <dl>\n      <dt>start</dt>\n        <dd>Fires when a request is sent to a resource.</dd>\n      <dt>complete</dt>\n        <dd>Fires when the transaction is complete.</dd>\n      <dt>success</dt>\n        <dd>Fires when the HTTP response status is within the 2xx\n        range.</dd>\n      <dt>failure</dt>\n        <dd>Fires when the HTTP response status is outside the 2xx\n        range, if an exception occurs, if the transation is aborted,\n        or if the transaction exceeds a configured `timeout`.</dd>\n      <dt>end</dt>\n        <dd>Fires at the conclusion of the transaction\n           lifecycle, after `success` or `failure`.</dd>\n    </dl>\n\n    <p>Callback functions for `start` and `end` receive the id of the\n    transaction as a first argument. For `complete`, `success`, and\n    `failure`, callbacks receive the id and the response object\n    (usually the XMLHttpRequest instance).  If the `arguments`\n    property was included in the configuration object passed to\n    `Y.io()`, the configured data will be passed to all callbacks as\n    the last argument.</p>\n    </dd>\n\n  <dt>sync</dt>\n    <dd>Pass `true` to make a same-domain transaction synchronous.\n    <strong>CAVEAT</strong>: This will negatively impact the user\n    experience. Have a <em>very</em> good reason if you intend to use\n    this.</dd>\n\n  <dt>context</dt>\n    <dd>The \"`this\" object for all configured event handlers. If a\n    specific context is needed for individual callbacks, bind the\n    callback to a context using `Y.bind()`.</dd>\n\n  <dt>headers</dt>\n    <dd>Object map of transaction headers to send to the server. The\n    object keys are the header names and the values are the header\n    values.</dd>\n\n  <dt>username</dt>\n    <dd>Username to use in a HTTP authentication.</dd>\n\n  <dt>password</dt>\n    <dd>Password to use in a HTTP authentication.</dd>\n\n  <dt>timeout</dt>\n    <dd>Millisecond threshold for the transaction before being\n    automatically aborted.</dd>\n\n  <dt>arguments</dt>\n    <dd>User-defined data passed to all registered event handlers.\n    This value is available as the second argument in the \"start\" and\n    \"end\" event handlers. It is the third argument in the \"complete\",\n    \"success\", and \"failure\" event handlers. <strong>Be sure to quote\n    this property name in the transaction configuration as\n    \"arguments\" is a reserved word in JavaScript</strong> (e.g.\n    `Y.io({ ..., \"arguments\": stuff })`).</dd>\n</dl>",
      "!data": {
       "submodule": "io-base"
      }
     },
     "xdr": {
      "!type": "fn(uri: string, o: +Object, c: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_xdr",
      "!doc": "Method for accessing the transports interface for making a\ncross-domain transaction.",
      "!data": {
       "submodule": "io-xdr"
      }
     },
     "xdrResponse": {
      "!type": "fn(e: string, o: +Object, c: +Object) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_xdrResponse",
      "!doc": "Response controller for cross-domain requests when using the\nFlash transport or IE8s XDomainRequest object.",
      "!data": {
       "submodule": "io-xdr"
      }
     },
     "transport": {
      "!type": "fn(o: +Object)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_transport",
      "!doc": "Initializes the desired transport.",
      "!data": {
       "submodule": "io-xdr"
      }
     }
    },
    "stringify": {
     "!type": "fn(form: +node.Node|string, options?: +config.IOStringifyConfig) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_stringify",
     "!doc": "Enumerate through an HTML forms elements collection\nand return a string comprised of key-value pairs.",
     "!data": {
      "submodule": "io-form"
     }
    },
    "queue": {
     "!type": "fn() -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_queue",
     "!doc": "Method for queueing a transaction before the request is sent to the\nresource, to ensure sequential processing.",
     "!data": {
      "submodule": "io-queue"
     }
    },
    "promote": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_promote",
     "!doc": "Method for promoting a transaction to the top of the queue.",
     "!data": {
      "submodule": "io-queue"
     }
    },
    "empty": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_empty",
     "!doc": "Method for cancel all pending transaction from\nthe queue.",
     "!data": {
      "submodule": "io-queue"
     }
    },
    "delay": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#property_delay",
     "!doc": "Delay value to calling the Flash transport, in the\nevent io.swf has not finished loading.  Once the E_XDR_READY\nevent is fired, this value will be set to 0.",
     "!data": {
      "submodule": "io-xdr"
     }
    }
   },
   "YUI": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html",
    "!data": {
     "extends": [
      "event_custom.EventTarget"
     ],
     "for": "yui.YUI"
    },
    "io": {
     "!type": "fn(url: string, config: +config.YUIIoConfig) -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_io",
     "!doc": "Method for initiating an ajax call.  The first argument is the url end\npoint for the call.  The second argument is an object to configure the\ntransaction and attach event subscriptions.  The configuration object\nsupports the following properties:\n\n<dl>\n  <dt>method</dt>\n    <dd>HTTP method verb (e.g., GET or POST). If this property is not\n        not defined, the default value will be GET.</dd>\n\n  <dt>data</dt>\n    <dd>This is the name-value string that will be sent as the\n    transaction data. If the request is HTTP GET, the data become\n    part of querystring. If HTTP POST, the data are sent in the\n    message body.</dd>\n\n  <dt>xdr</dt>\n    <dd>Defines the transport to be used for cross-domain requests.\n    By setting this property, the transaction will use the specified\n    transport instead of XMLHttpRequest. The properties of the\n    transport object are:\n    <dl>\n      <dt>use</dt>\n        <dd>The transport to be used: flash or native</dd>\n      <dt>dataType</dt>\n        <dd>Set the value to XML if that is the expected response\n        content type.</dd>\n    </dl></dd>\n\n  <dt>form</dt>\n    <dd>Form serialization configuration object.  Its properties are:\n    <dl>\n      <dt>id</dt>\n        <dd>Node object or id of HTML form</dd>\n      <dt>useDisabled</dt>\n        <dd>`true` to also serialize disabled form field values\n        (defaults to `false`)</dd>\n    </dl></dd>\n\n  <dt>on</dt>\n    <dd>Assigns transaction event subscriptions. Available events are:\n    <dl>\n      <dt>start</dt>\n        <dd>Fires when a request is sent to a resource.</dd>\n      <dt>complete</dt>\n        <dd>Fires when the transaction is complete.</dd>\n      <dt>success</dt>\n        <dd>Fires when the HTTP response status is within the 2xx\n        range.</dd>\n      <dt>failure</dt>\n        <dd>Fires when the HTTP response status is outside the 2xx\n        range, if an exception occurs, if the transation is aborted,\n        or if the transaction exceeds a configured `timeout`.</dd>\n      <dt>end</dt>\n        <dd>Fires at the conclusion of the transaction\n           lifecycle, after `success` or `failure`.</dd>\n    </dl>\n\n    <p>Callback functions for `start` and `end` receive the id of the\n    transaction as a first argument. For `complete`, `success`, and\n    `failure`, callbacks receive the id and the response object\n    (usually the XMLHttpRequest instance).  If the `arguments`\n    property was included in the configuration object passed to\n    `Y.io()`, the configured data will be passed to all callbacks as\n    the last argument.</p>\n    </dd>\n\n  <dt>sync</dt>\n    <dd>Pass `true` to make a same-domain transaction synchronous.\n    <strong>CAVEAT</strong>: This will negatively impact the user\n    experience. Have a <em>very</em> good reason if you intend to use\n    this.</dd>\n\n  <dt>context</dt>\n    <dd>The \"`this\" object for all configured event handlers. If a\n    specific context is needed for individual callbacks, bind the\n    callback to a context using `Y.bind()`.</dd>\n\n  <dt>headers</dt>\n    <dd>Object map of transaction headers to send to the server. The\n    object keys are the header names and the values are the header\n    values.</dd>\n\n  <dt>timeout</dt>\n    <dd>Millisecond threshold for the transaction before being\n    automatically aborted.</dd>\n\n  <dt>arguments</dt>\n    <dd>User-defined data passed to all registered event handlers.\n    This value is available as the second argument in the \"start\" and\n    \"end\" event handlers. It is the third argument in the \"complete\",\n    \"success\", and \"failure\" event handlers. <strong>Be sure to quote\n    this property name in the transaction configuration as\n    \"arguments\" is a reserved word in JavaScript</strong> (e.g.\n    `Y.io({ ..., \"arguments\": stuff })`).</dd>\n</dl>",
     "!data": {
      "submodule": "io-base"
     }
    },
    "header": {
     "!type": "fn(name: string, value: string)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_header",
     "!doc": "Method for setting and deleting IO HTTP headers to be sent with every\nrequest.\n\nHosted as a property on the `io` function (e.g. `Y.io.header`).",
     "!data": {
      "submodule": "io-base"
     }
    }
   },
   "JSON": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html",
    "!data": {
     "for": "json.JSON"
    },
    "_default": {
     "!type": "string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#property__default",
     "!doc": "The ID of the default IO transport, defaults to `xhr`"
    },
    "defaultTransport": {
     "!type": "fn(id?: string) -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#method_defaultTransport"
    },
    "transports": {
     "!type": "+Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#property_transports",
     "!doc": "An object hash of custom transports available to IO"
    },
    "customTransport": {
     "!type": "fn(id: string)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#method_customTransport",
     "!doc": "Create a custom transport of type and return its object"
    },
    "prototype": {
     "notify": {
      "!type": "fn(event: string, transaction: +Object, config: +config.JSONNotifyConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#method_notify",
      "!doc": "Fired from the notify method of the transport which in turn fires\nthe event on the IO object."
     }
    }
   }
  },
  "io_nodejs": {
   "!data": {
    "module": "io-nodejs"
   },
   "IO": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html",
    "!data": {
     "for": "io.IO"
    },
    "request": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_request",
     "!doc": "Passthru to the NodeJS <a href=\"https://github.com/mikeal/request\">request</a> module.\nThis method is return of `require(request)` so you can use it inside NodeJS without\nthe IO abstraction."
    },
    "transports.nodejs": {
     "!type": "fn() -> +Object",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/IO.html#method_transports.nodejs",
     "!doc": "NodeJS IO transport, uses the NodeJS <a href=\"https://github.com/mikeal/request\">request</a>\nmodule under the hood to perform all network IO."
    }
   }
  },
  "json": {
   "!data": {
    "submodules": {
     "json-parse": {},
     "json-stringify": {}
    }
   },
   "JSON": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html",
    "parse": {
     "!type": "fn(s: string, reviver: fn()) -> +MIXED",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#method_parse",
     "!doc": "Parse a JSON string, returning the native JavaScript representation.",
     "!data": {
      "submodule": "json-parse"
     }
    },
    "dateToString": {
     "!type": "fn(d: +datatype_date.Date) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#method_dateToString",
     "!doc": "Serializes a Date instance as a UTC date string.  Used internally by\nstringify.  Override this method if you need Dates serialized in a\ndifferent format.",
     "!data": {
      "submodule": "json-stringify"
     }
    },
    "stringify": {
     "!type": "fn(o: +MIXED, w: [?], ind: number|string) -> string",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#method_stringify",
     "!doc": "<p>Converts an arbitrary value to a JSON string representation.</p>\n\n<p>Objects with cyclical references will trigger an exception.</p>\n\n<p>If a whitelist is provided, only matching object keys will be\nincluded.  Alternately, a replacer function may be passed as the\nsecond parameter.  This function is executed on every value in the\ninput, and its return value will be used in place of the original value.\nThis is useful to serialize specialized objects or class instances.</p>\n\n<p>If a positive integer or non-empty string is passed as the third\nparameter, the output will be formatted with carriage returns and\nindentation for readability.  If a String is passed (such as \"\\t\") it\nwill be used once for each indentation level.  If a number is passed,\nthat number of spaces will be used.</p>",
     "!data": {
      "submodule": "json-stringify"
     }
    },
    "charCacheThreshold": {
     "!type": "number",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/JSON.html#property_charCacheThreshold",
     "!doc": "<p>Number of occurrences of a special character within a single call to\nstringify that should trigger promotion of that character to a dedicated\npreprocess step for future calls.  This is only used in environments\nthat dont support native JSON, or when useNativeJSONStringify is set to\nfalse.</p>\n\n<p>So, if set to 50 and an object is passed to stringify that includes\nstrings containing the special character \\x07 more than 50 times,\nsubsequent calls to stringify will process object strings through a\nfaster serialization path for \\x07 before using the generic, slower,\nreplacement process for all special characters.</p>\n\n<p>To prime the preprocessor cache, set this value to 1, then call\n<code>Y.JSON.stringify(\"<em>(all special characters to\ncache)</em>\");</code>, then return this setting to a more conservative\nvalue.</p>\n\n<p>Special characters \\ \" \\b \\t \\n \\f \\r are already cached.</p>",
     "!data": {
      "submodule": "json-stringify"
     }
    }
   }
  },
  "jsonp": {
   "!data": {
    "submodules": {
     "jsonp-url": {}
    }
   },
   "JSONPRequest": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/JSONPRequest.html",
    "prototype": {
     "_defaultCallback": {
      "!type": "fn(url: string, config: +config.JSONPRequest_defaultCallbackConfig) -> fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/JSONPRequest.html#method__defaultCallback",
      "!doc": "Override this method to provide logic to default the success callback if\nit is not provided at construction.  This is overridden by jsonp-url to\nparse the callback from the url string."
     },
     "send": {
      "!type": "fn(args: ?) -> !this",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/JSONPRequest.html#method_send",
      "!doc": "Issues the JSONP request."
     }
    }
   },
   "YUI": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html",
    "!data": {
     "extends": [
      "event_custom.EventTarget"
     ],
     "for": "yui.YUI"
    },
    "jsonp": {
     "!type": "fn(url: string, c: fn(), args: ?) -> +jsonp.JSONPRequest",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#method_jsonp"
    }
   }
  },
  "loader": {
   "!data": {
    "submodules": {
     "loader-base": {},
     "rollup": {},
     "loader-yui3": {}
    }
   },
   "YUI": {
    "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html",
    "!data": {
     "extends": [
      "event_custom.EventTarget"
     ],
     "for": "yui.YUI"
    },
    "prototype": {
     "meta": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/YUI.html#property_meta",
      "!doc": "The component metadata is stored in Y.Env.meta.\nPart of the loader module.",
      "!data": {
       "submodule": "loader-base"
      }
     }
    }
   },
   "Loader": {
    "!type": "fn(config: +config.LoaderConfig)",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html",
    "prototype": {
     "onSuccess": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_onSuccess",
      "!doc": "Callback that will be executed when the loader is finished\nwith an insert",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "onFailure": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_onFailure",
      "!doc": "Callback that will be executed if there is a failure",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "onProgress": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_onProgress",
      "!doc": "Callback executed each time a script or css file is loaded",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "onTimeout": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_onTimeout",
      "!doc": "Callback that will be executed if a timeout occurs",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "context": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_context",
      "!doc": "The execution context for all callbacks",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "data": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_data",
      "!doc": "Data that is passed to all callbacks",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "insertBefore": {
      "!type": "string|+HTMLElement",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_insertBefore",
      "!doc": "Node reference or id where new nodes should be inserted before",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "charset": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_charset",
      "!doc": "The charset attribute for inserted nodes",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "cssAttributes": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_cssAttributes",
      "!doc": "An object literal containing attributes to add to link nodes",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "jsAttributes": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_jsAttributes",
      "!doc": "An object literal containing attributes to add to script nodes",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "base": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_base",
      "!doc": "The base directory.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "comboBase": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_comboBase",
      "!doc": "Base path for the combo service",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "combine": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_combine",
      "!doc": "If configured, the loader will attempt to use the combo\nservice for YUI resources and configured external resources.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "comboSep": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_comboSep",
      "!doc": "The default seperator to use between files in a combo URL",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "maxURLLength": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_maxURLLength",
      "!doc": "Max url length for combo urls.  The default is 1024. This is the URL\nlimit for the Yahoo! hosted combo servers.  If consuming\na different combo service that has a different URL limit\nit is possible to override this default by supplying\nthe maxURLLength config option.  The config option will\nonly take effect if lower than the default.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "ignoreRegistered": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_ignoreRegistered",
      "!doc": "Ignore modules registered on the YUI global",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "root": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_root",
      "!doc": "Root path to prepend to module path for the combo\nservice",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "timeout": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_timeout",
      "!doc": "Timeout value in milliseconds.  If set, self value will be used by\nthe get utility.  the timeout event will fire if\na timeout occurs.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "ignore": {
      "!type": "[string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_ignore",
      "!doc": "A list of modules that should not be loaded, even if\nthey turn up in the dependency tree",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "force": {
      "!type": "[string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_force",
      "!doc": "A list of modules that should always be loaded, even\nif they have already been inserted into the page.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "allowRollup": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_allowRollup",
      "!doc": "Should we allow rollups",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "filter": {
      "!type": "string|+searchExp",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_filter",
      "!doc": "A filter to apply to result urls.  This filter will modify the default\npath for all modules.  The default path for the YUI library is the\nminified version of the files (e.g., event-min.js).  The filter property\ncan be a predefined filter or a custom filter.  The valid predefined\nfilters are:\n<dl>\n <dt>DEBUG</dt>\n <dd>Selects the debug versions of the library (e.g., event-debug.js).\n     This option will automatically include the Logger widget</dd>\n <dt>RAW</dt>\n <dd>Selects the non-minified version of the library (e.g., event.js).\n </dd>\n</dl>\nYou can also define a custom filter, which must be an object literal\ncontaining a search expression and a replace string:\n\n     myFilter: {\n         searchExp: \"-min\\\\.js\",\n         replaceStr: \"-debug.js\"\n     }",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "filters": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_filters",
      "!doc": "per-component filter specification.  If specified for a given\ncomponent, this overrides the filter config.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "required": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_required",
      "!doc": "The list of requested modules",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "patterns": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_patterns",
      "!doc": "If a module name is predefined when requested, it is checked againsts\nthe patterns provided in this property.  If there is a match, the\nmodule is added with the default configuration.\n\nAt the moment only supporting module prefixes, but anticipate\nsupporting at least regular expressions.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "undefined": {
      "!type": "fn()",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html",
      "!doc": "Internal loader instance metadata. Use accessor `getModuleInfo()` instead.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "skin": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_skin",
      "!doc": "Provides the information used to skin the skinnable components.\nThe following skin definition would result in skin1 and skin2\nbeing loaded for calendar (if calendar was requested), and\nsam for all other skinnable components:\n\n     skin: {\n         // The default skin, which is automatically applied if not\n         // overriden by a component-specific skin definition.\n         // Change this in to apply a different skin globally\n         defaultSkin: sam,\n\n         // This is combined with the loader base property to get\n         // the default root directory for a skin. ex:\n         // http://yui.yahooapis.com/2.3.0/build/assets/skins/sam/\n         base: assets/skins/,\n\n         // Any component-specific overrides can be specified here,\n         // making it possible to load different skins for different\n         // components.  It is possible to load more than one skin\n         // for a given component as well.\n         overrides: {\n             calendar: [skin1, skin2]\n         }\n     }",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "loaded": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_loaded",
      "!doc": "Set when beginning to compute the dependency tree.\nComposed of what YUI reports to be loaded combined\nwith what has been loaded by any instance on the page\nwith the version number specified in the metadata.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "async": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_async",
      "!doc": "Should Loader fetch scripts in `async`, defaults to `true`",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "rollups": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_rollups",
      "!doc": "List of rollup files found in the library metadata",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "loadOptional": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_loadOptional",
      "!doc": "Whether or not to load optional dependencies for\nthe requested modules",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "sorted": {
      "!type": "[string]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_sorted",
      "!doc": "All of the derived dependencies in sorted order, which\nwill be populated when either calculate() or insert()\nis called",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "dirty": {
      "!type": "bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_dirty",
      "!doc": "Flag to indicate the dependency tree needs to be recomputed\nif insert is called again.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "inserted": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_inserted",
      "!doc": "List of modules inserted by the utility",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "skipped": {
      "!type": "?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#property_skipped",
      "!doc": "List of skipped modules during insert() because the module\nwas not defined",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "getModuleInfo": {
      "!type": "fn(name: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_getModuleInfo",
      "!doc": "Gets the module info from the local moduleInfo hash, or from the\ndefault metadata and populate the local moduleInfo hash.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "formatSkin": {
      "!type": "fn(skin: string, mod: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_formatSkin",
      "!doc": "Returns the skin module name for the specified skin name.  If a\nmodule name is supplied, the returned skin module name is\nspecific to the module passed in.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "addAlias": {
      "!type": "fn(use: [?], name: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_addAlias",
      "!doc": "Adds an alias module to the system",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "addGroup": {
      "!type": "fn(config: +config.LoaderAddGroupConfig, name: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_addGroup",
      "!doc": "Add a new module group",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "addModule": {
      "!type": "fn(config: +config.LoaderAddModuleConfig, name?: string) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_addModule",
      "!doc": "Add a new module to the component metadata.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "require": {
      "!type": "fn(what: [string]|string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_require",
      "!doc": "Add a requirement for one or more module",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "filterRequires": {
      "!type": "fn(r: [?]) -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_filterRequires",
      "!doc": "Explodes the required array to remove aliases and replace them with real modules",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "_canBeAttached": {
      "!type": "fn(module: string) -> bool",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method__canBeAttached",
      "!doc": "Returns `true` if the module can be attached to the YUI instance. Runs\nthe modules test if there is one and caches its result.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "getRequires": {
      "!type": "fn(mod: +Object) -> [?]",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_getRequires",
      "!doc": "Returns an object containing properties for all modules required\nin order to load the requested module",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "isCSSLoaded": {
      "!type": "fn(name: string, skip: bool) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_isCSSLoaded",
      "!doc": "Check to see if named css module is already loaded on the page",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "getProvides": {
      "!type": "fn(name: string) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_getProvides",
      "!doc": "Returns a hash of module names the supplied module satisfies.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "calculate": {
      "!type": "fn(o: +Object, type: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_calculate",
      "!doc": "Calculates the dependency tree, the result is stored in the sorted\nproperty.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "getLangPackName": {
      "!type": "fn(lang: string, mname: string) -> string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_getLangPackName",
      "!doc": "Builds a module name for a language pack",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "getModule": {
      "!type": "fn(mname: string) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_getModule",
      "!doc": "Gets the loader meta data for the requested module",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "insert": {
      "!type": "fn(o: +Object, type: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_insert",
      "!doc": "inserts the requested modules and their dependencies.\n<code>type</code> can be \"js\" or \"css\".  Both script and\ncss are inserted if type is not provided.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "loadNext": {
      "!type": "fn(mname: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_loadNext",
      "!doc": "Executed every time a module is loaded, and if we are in a load\ncycle, we attempt to load the next script.  Public so that it\nis possible to call this if using a method other than\nY.register to determine when scripts are fully loaded",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "resolve": {
      "!type": "fn(calc?: bool, sorted?: [?]) -> +Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_resolve",
      "!doc": "Returns an Object hash of file arrays built from `loader.sorted` or from an arbitrary list of sorted modules.",
      "!data": {
       "submodule": "loader-base"
      }
     },
     "load": {
      "!type": "fn(cb: fn())",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Loader.html#method_load",
      "!doc": "Shortcut to calculate, resolve and load all modules.\n\n    var loader = new Y.Loader({\n        ignoreRegistered: true,\n        modules: {\n            mod: {\n                path: mod.js\n            }\n        },\n        requires: [ mod ]\n    });\n    loader.load(function() {\n        console.log(All modules have loaded..);\n    });",
      "!data": {
       "submodule": "loader-base"
      }
     }
    }
   }
  },
  "matrix": {
   "Matrix": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html",
    "prototype": {
     "multiple": {
      "!type": "fn(a: number, b: number, c: number, d: number, dx: number, dy: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_multiple",
      "!doc": "Updates the matrix."
     },
     "applyCSSText": {
      "!type": "fn(val: string)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_applyCSSText",
      "!doc": "Parses a string and updates the matrix."
     },
     "getTransformArray": {
      "!type": "fn(val: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_getTransformArray",
      "!doc": "Parses a string and returns an array of transform arrays."
     },
     "init": {
      "!type": "fn(config: +config.MatrixInitConfig)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_init",
      "!doc": "Initializes a matrix."
     },
     "scale": {
      "!type": "fn(val: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_scale",
      "!doc": "Applies a scale transform"
     },
     "skew": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_skew",
      "!doc": "Applies a skew transformation."
     },
     "skewX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_skewX",
      "!doc": "Applies a skew to the x-coordinate"
     },
     "skewY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_skewY",
      "!doc": "Applies a skew to the y-coordinate"
     },
     "toCSSText": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_toCSSText",
      "!doc": "Returns a string of text that can be used to populate a the css transform property of an element."
     },
     "toFilterText": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_toFilterText",
      "!doc": "Returns a string that can be used to populate the css filter property of an element."
     },
     "rad2deg": {
      "!type": "fn(rad: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_rad2deg",
      "!doc": "Converts a radian value to a degree."
     },
     "deg2rad": {
      "!type": "fn(deg: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_deg2rad",
      "!doc": "Converts a degree value to a radian."
     },
     "rotate": {
      "!type": "fn(deg: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_rotate",
      "!doc": "Applies a rotate transform."
     },
     "translate": {
      "!type": "fn(x: number, y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_translate",
      "!doc": "Applies translate transformation."
     },
     "translateX": {
      "!type": "fn(x: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_translateX",
      "!doc": "Applies a translate to the x-coordinate"
     },
     "translateY": {
      "!type": "fn(y: number)",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_translateY",
      "!doc": "Applies a translate to the y-coordinate"
     },
     "identity": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_identity",
      "!doc": "Returns an identity matrix."
     },
     "getMatrixArray": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_getMatrixArray",
      "!doc": "Returns a 3x3 Matrix array\n\n/                                             \\\n| matrix[0][0]   matrix[1][0]    matrix[2][0] |\n| matrix[0][1]   matrix[1][1]    matrix[2][1] |\n| matrix[0][2]   matrix[1][2]    matrix[2][2] |\n\\                                             /"
     },
     "getContentRect": {
      "!type": "fn(width: number, height: number, x: number, y: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_getContentRect",
      "!doc": "Returns the left, top, right and bottom coordinates for a transformed\nitem."
     },
     "getDeterminant": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_getDeterminant",
      "!doc": "Returns the determinant of the matrix."
     },
     "inverse": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_inverse",
      "!doc": "Returns the inverse (in array form) of the matrix."
     },
     "transpose": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_transpose",
      "!doc": "Returns the transpose of the matrix"
     },
     "decompose": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Matrix.html#method_decompose",
      "!doc": "Returns an array of transform commands that represent the matrix."
     }
    }
   },
   "MatrixUtil": {
    "!type": "fn()",
    "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html",
    "prototype": {
     "rad2deg": {
      "!type": "fn(rad: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_rad2deg",
      "!doc": "Converts a radian value to a degree."
     },
     "deg2rad": {
      "!type": "fn(deg: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_deg2rad",
      "!doc": "Converts a degree value to a radian."
     },
     "angle2rad": {
      "!type": "fn(val: +Objecxt) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_angle2rad",
      "!doc": "Converts an angle to a radian"
     },
     "getnxn": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_getnxn",
      "!doc": "Converts a transform object to an array of column vectors.\n\n/                                             \\\n| matrix[0][0]   matrix[1][0]    matrix[2][0] |\n| matrix[0][1]   matrix[1][1]    matrix[2][1] |\n| matrix[0][2]   matrix[1][2]    matrix[2][2] |\n\\                                             /"
     },
     "getDeterminant": {
      "!type": "fn(matrix: [?]) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_getDeterminant",
      "!doc": "Returns the determinant of a given matrix.\n\n/                                             \\\n| matrix[0][0]   matrix[1][0]    matrix[2][0] |\n| matrix[0][1]   matrix[1][1]    matrix[2][1] |\n| matrix[0][2]   matrix[1][2]    matrix[2][2] |\n| matrix[0][3]   matrix[1][3]    matrix[2][3] |\n\\                                             /"
     },
     "inverse": {
      "!type": "fn(Array: ?) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_inverse",
      "!doc": "Returns the inverse of a matrix"
     },
     "scalarMultiply": {
      "!type": "fn(matrix: [?], multiplier: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_scalarMultiply",
      "!doc": "Multiplies a matrix by a numeric value."
     },
     "transpose": {
      "!type": "fn(matrix: ?) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_transpose",
      "!doc": "Returns the transpose for an nxn matrix."
     },
     "getMinors": {
      "!type": "fn(matrix: [?], columnIndex: number, rowIndex: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_getMinors",
      "!doc": "Returns a matrix of minors based on a matrix, column index and row index."
     },
     "sign": {
      "!type": "fn(val: number) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_sign",
      "!doc": "Returns the sign of value"
     },
     "vectorMatrixProduct": {
      "!type": "fn(vector: [?], matrix: [?]) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_vectorMatrixProduct",
      "!doc": "Multiplies a vector and a matrix"
     },
     "decompose": {
      "!type": "fn(matrix: [?]) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_decompose",
      "!doc": "Breaks up a 2d transform matrix into a series of transform operations."
     },
     "getTransformArray": {
      "!type": "fn(val: string) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_getTransformArray",
      "!doc": "Parses a transform string and returns an array of transform arrays."
     },
     "getTransformFunctionArray": {
      "!type": "fn() -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_getTransformFunctionArray",
      "!doc": "Returns an array of transform arrays representing transform functions and arguments."
     },
     "compareTransformSequence": {
      "!type": "fn(list1: [?], list2: [?]) -> ?",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#method_compareTransformSequence",
      "!doc": "Compares to arrays or transform functions to ensure both contain the same functions in the same\norder."
     },
     "transformMethods": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/MatrixUtil.html#property_transformMethods",
      "!doc": "Mapping of possible transform method names."
     }
    }
   }
  },
  "node_flick": {
   "!data": {
    "module": "node-flick"
   },
   "Plugin": {
    "Flick": {
     "!type": "fn(config: +config.Plugin.FlickConfig)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html",
     "NAME": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#property_NAME",
      "!doc": "The NAME of the Flick class. Used to prefix events generated\nby the plugin."
     },
     "NS": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#property_NS",
      "!doc": "The namespace for the plugin. This will be the property on the node, which will\nreference the plugin instance, when its plugged in."
     },
     "prototype": {
      "initializer": {
       "!type": "fn(config: +config.Plugin.FlickInitializerConfig)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#method_initializer",
       "!doc": "The initializer lifecycle implementation."
      },
      "setBounds": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#method_setBounds",
       "!doc": "Sets the min/maxÂ boundaries for the flick animation,\nbased on the boundingBox dimensions."
      }
     },
     "VELOCITY_THRESHOLD": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#property_VELOCITY_THRESHOLD",
      "!doc": "The threshold used to determine when the decelerated velocity of the node\nis practically 0."
     },
     "SNAP_DURATION": {
      "!type": "number",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#property_SNAP_DURATION",
      "!doc": "The duration to use for the bounce snap-back transition"
     },
     "EASING": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#property_EASING",
      "!doc": "The default easing to use for the main flick movement transition"
     },
     "SNAP_EASING": {
      "!type": "string",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#property_SNAP_EASING",
      "!doc": "The default easing to use for the bounce snap-back transition"
     },
     "CLASS_NAMES": {
      "!type": "+Object",
      "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Flick.html#property_CLASS_NAMES",
      "!doc": "The default CSS class names used by the plugin"
     }
    }
   }
  },
  "node_focusmanager": {
   "!data": {
    "module": "node-focusmanager"
   },
   "plugin": {
    "NodeFocusManager": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html",
     "prototype": {
      "focus": {
       "!type": "fn(index: number|+node.Node)",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#method_focus",
       "!doc": "Focuses the active descendant and sets the\n<code>focused</code> attribute to true."
      },
      "blur": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#method_blur",
       "!doc": "Blurs the current active descendant and sets the\n<code>focused</code> attribute to false."
      },
      "start": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#method_start",
       "!doc": "Enables the Focus Manager."
      },
      "stop": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#method_stop",
       "!doc": "Disables the Focus Manager by detaching all event handlers."
      },
      "refresh": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeFocusManager.html#method_refresh",
       "!doc": "Refreshes the Focus Managers descendants by re-executing the\nCSS selector query specified by the <code>descendants</code> attribute."
      }
     }
    }
   }
  },
  "node_menunav": {
   "!data": {
    "module": "node-menunav"
   },
   "plugin": {
    "NodeMenuNav": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeMenuNav.html",
     "prototype": {
      "SHIM_TEMPLATE_TITLE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeMenuNav.html#property_SHIM_TEMPLATE_TITLE",
       "!doc": "String representing the value for the <code>title</code>\nattribute for the shim used to prevent <code>&#60;select&#62;</code> elements\nfrom poking through menus in IE 6."
      },
      "SHIM_TEMPLATE": {
       "!type": "string",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/plugin.NodeMenuNav.html#property_SHIM_TEMPLATE",
       "!doc": "String representing the HTML used to create the\n<code>&#60;iframe&#62;</code> shim used to prevent\n<code>&#60;select&#62;</code> elements from poking through menus in IE 6."
      }
     }
    }
   }
  },
  "node_scroll_info": {
   "!data": {
    "module": "node-scroll-info"
   },
   "Plugin": {
    "ScrollInfo": {
     "!type": "fn()",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollInfo.html",
     "prototype": {
      "!proto": "plugin.Plugin.Base.prototype",
      "getOffscreenNodes": {
       "!type": "fn(selector?: string, margin?: number) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollInfo.html#method_getOffscreenNodes",
       "!doc": "Returns a NodeList containing all offscreen nodes inside the host node that\nmatch the given CSS selector. An offscreen node is any node that is entirely\noutside the visible (onscreen) region of the host node based on the current\nscroll location."
      },
      "getOnscreenNodes": {
       "!type": "fn(selector?: string, margin?: number) -> +node.NodeList",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollInfo.html#method_getOnscreenNodes",
       "!doc": "Returns a NodeList containing all onscreen nodes inside the host node that\nmatch the given CSS selector. An onscreen node is any node that is fully or\npartially within the visible (onscreen) region of the host node based on the\ncurrent scroll location."
      },
      "getScrollInfo": {
       "!type": "fn() -> +Object",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollInfo.html#method_getScrollInfo",
       "!doc": "Returns an object hash containing information about the current scroll\nposition of the host node. This is the same information thats mixed into\nthe event facade of the `scroll` event and other scroll-related events."
      },
      "isNodeOnscreen": {
       "!type": "fn(node: +HTMLElement|+node.Node|string, margin?: number) -> bool",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollInfo.html#method_isNodeOnscreen",
       "!doc": "Returns `true` if _node_ is at least partially onscreen within the host\nnode, `false` otherwise."
      },
      "refreshDimensions": {
       "!type": "fn()",
       "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.ScrollInfo.html#method_refreshDimensions",
       "!doc": "Refreshes cached position, height, and width dimensions for the host node.\nIf the host node is the body, then the viewport height and width will be\nused.\n\nThis info is cached to improve performance during scroll events, since its\nexpensive to touch the DOM for these values. Dimensions are automatically\nrefreshed whenever the browser is resized, but if you change the dimensions\nor position of the host node in JS, you may need to call\n`refreshDimensions()` manually to cache the new dimensions."
      }
     }
    }
   }
  },
  "align_plugin": {
   "!data": {
    "module": "align-plugin"
   },
   "Plugin": {
    "Align": {
     "!type": "fn(User: +Object)",
     "!url": "http://yuilibrary.com/yui/docs/api/classes/Plugin.Align.html",
    