/**
 * Copyright (c) Red Hat, Inc., contributors and others 2013 - 2014. All rights reserved
 *
 * Licensed under the Eclipse Public License version 1.0, available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.jboss.tools.forge.ui.internal.ext.wizards;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.wizard.IWizardContainer;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.swt.widgets.Shell;
import org.jboss.forge.addon.ui.controller.CommandController;
import org.jboss.forge.addon.ui.controller.WizardCommandController;
import org.jboss.forge.addon.ui.result.CompositeResult;
import org.jboss.forge.addon.ui.result.Failed;
import org.jboss.forge.addon.ui.result.Result;
import org.jboss.tools.forge.ui.internal.ForgeUIPlugin;
import org.jboss.tools.forge.ui.internal.ext.context.UIContextImpl;
import org.jboss.tools.forge.ui.internal.ext.context.UISelectionImpl;
import org.jboss.tools.forge.ui.internal.ext.dialog.InterruptableProgressMonitor;
import org.jboss.tools.forge.ui.notifications.NotificationType;

/**
 * 
 * @author <a href="ggastald@redhat.com">George Gastaldi</a>
 */
public class ForgeWizard extends MutableWizard {

	private final CommandController controller;
	private final UIContextImpl uiContext;
	private ForgeWizardHelper helper = new ForgeWizardHelper();
	private InterruptableProgressMonitor progressMonitor;

	public ForgeWizard(String command, CommandController controller,
			UIContextImpl contextImpl) {
		this.controller = controller;
		this.uiContext = contextImpl;
		setWindowTitle(constructTitle(command));
		setNeedsProgressMonitor(true);
		setForcePreviousAndNextButtons(isWizard());
		helper.onCreate(contextImpl);
	}

	public UIContextImpl getUIContext() {
		return uiContext;
	}

	private String constructTitle(String command) {
		UISelectionImpl<?> currentSelection = uiContext.getInitialSelection();
		StringBuilder title = new StringBuilder(command);
		if (!currentSelection.isEmpty()) {
			String currentSelectionLabel;
			IResource resource = currentSelection.getResource();
			if (resource != null) {
				currentSelectionLabel = resource.getFullPath().toOSString();
			} else {
				currentSelectionLabel = currentSelection.get().toString();
			}
			title.append(" [Current Selection: ").append(currentSelectionLabel)
					.append("]");
		}
		return title.toString();
	}

	@Override
	public void addPages() {
		addPage(createPage());
	}

	@Override
	public boolean canFinish() {
		return controller.canExecute();
	}

	@Override
	public boolean performFinish() {
		try {
			IWizardContainer container = getContainer();
			performFinish(container, container.getShell());
		} catch (Exception e) {
			ForgeUIPlugin.log(e);
		}
		return true;
	}

	public void performFinish(final IRunnableContext runnableContext,
			final Shell shell) throws InvocationTargetException,
			InterruptedException {
		runnableContext.run(true, true, new IRunnableWithProgress() {
			@Override
			public void run(IProgressMonitor monitor)
					throws InvocationTargetException, InterruptedException {
				try {
					if (progressMonitor != null)
						progressMonitor.setRunnableThread(Thread
								.currentThread());

					monitor.beginTask("Executing Forge Wizard",
							IProgressMonitor.UNKNOWN);
					Map<Object, Object> attributeMap = uiContext
							.getAttributeMap();
					attributeMap.put(IProgressMonitor.class, monitor);
					attributeMap.put(Shell.class, shell);

					Result commandResult = controller.execute();
					displayResult(commandResult);
					helper.onFinish(getUIContext());
					monitor.done();
				} catch (Exception e) {
					ForgeUIPlugin.displayMessage(getWindowTitle(),
							"Error while executing task, check Error log view",
							NotificationType.ERROR);
					ForgeUIPlugin.log(e);
				} finally {
					try {
						controller.close();
					} catch (Exception e) {
						ForgeUIPlugin.log(e);
					}
				}
			}
		});
	}

	public void setProgressMonitor(InterruptableProgressMonitor progressMonitor) {
		this.progressMonitor = progressMonitor;
	}

	private void displayResult(Result result) {
		if (result instanceof CompositeResult) {
			for (Result thisResult : ((CompositeResult) result).getResults()) {
				displayResult(thisResult);
			}
		} else if (result != null) {
			String message = result.getMessage();
			if (message != null) {
				NotificationType notificationType = result instanceof Failed ? NotificationType.ERROR
						: NotificationType.INFO;
				ForgeUIPlugin.displayMessage(getWindowTitle(), message,
						notificationType);
			}
			if (result instanceof Failed) {
				Throwable exception = ((Failed) result).getException();
				if (exception != null) {
					ForgeUIPlugin.log(exception);
					ForgeUIPlugin.displayMessage(getWindowTitle(),
							String.valueOf(exception.getMessage()),
							NotificationType.ERROR);
				}
			}
		}
	}

	@Override
	public boolean performCancel() {
		helper.onCancel(getUIContext());
		return true;
	}

	protected ForgeWizardPage createPage() {
		return new ForgeWizardPage(this, controller);
	}

	private boolean isWizard() {
		return controller instanceof WizardCommandController;
	}

	@Override
	public IWizardPage getNextPage(IWizardPage page) {
		IWizardPage nextPage = super.getNextPage(page);
		if (nextPage != null) {
			// Subsequent pages are stale. Remove all subsequent pages
			removeSubsequentPages(nextPage);
			nextPage = null;
		}
		if (nextPage == null) {
			try {
				addPage(createPage());
				nextPage = super.getNextPage(page);
			} catch (Exception e) {
				ForgeUIPlugin.log(e);
			}
		}
		return nextPage;
	}

	/**
	 * @param page
	 */
	private void removeSubsequentPages(IWizardPage page) {
		List<ForgeWizardPage> pageList = getPageList();
		int idx = pageList.indexOf(page);
		List<ForgeWizardPage> subList = pageList.subList(idx, pageList.size());
		for (ForgeWizardPage forgeWizardPage : subList) {
			forgeWizardPage.dispose();
		}
		subList.clear();
	}
}
